
//-----------------------------------------------------------------------------
#include "PdfExportDialog.h"

//-----------------------------------------------------------------------------
#include <QButtonGroup>
#include <QFileDialog>
#include <QLabel>
#include <QLineEdit>
#include <QMessageBox>
#include <QTextEdit>
#include <QCheckBox>
#include <QHBoxLayout>

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/Qt/SoQt.h>
#include <Inventor/Qt/viewers/SoQtExaminerViewer.h>
#include <VolumeViz/nodes/SoVolumeData.h>

#include <HardCopy/SoToU3DAction.h>
#include <HardCopy/SoToPDFAction.h>

//-----------------------------------------------------------------------------
PdfExportDialog::PdfExportDialog()
{
  setupUi( this );

  QObject::connect( m_closeButton, SIGNAL(clicked()), this, SLOT(close()) );
  QObject::connect( m_inputButton, SIGNAL(clicked()), this, SLOT(inputButtonClicked()) );
  QObject::connect( m_pdfOutputButton, SIGNAL(clicked()), this, SLOT(pdfOutputButtonClicked()) );
  QObject::connect( m_u3dOutputButton, SIGNAL(clicked()), this, SLOT(u3dOutputButtonClicked()) );
  QObject::connect( m_exportButton, SIGNAL(clicked()), this, SLOT(exportButtonClicked()) );
  QObject::connect( m_pdfGroup, SIGNAL(toggled(bool)), this, SLOT(pdfGroupToggled()) );
  QObject::connect( m_u3dGroup, SIGNAL(toggled(bool)), this, SLOT(u3dGroupToggled()) );
  QObject::connect( m_writePassEdit, SIGNAL(textChanged(const QString&)), this, SLOT(writePassChanged()) );

  m_statusLabel->setText( QString() );
  pdfGroupToggled();
  writePassChanged();
}

//-----------------------------------------------------------------------------
void PdfExportDialog::inputButtonClicked()
{
  m_inputEdit->setText( QFileDialog::getOpenFileName(this, QString(), QString(), "Inventor (*.iv)") );
}

//-----------------------------------------------------------------------------
void PdfExportDialog::pdfOutputButtonClicked()
{
  m_pdfOutputEdit->setText( QFileDialog::getSaveFileName( this, QString(), QString(), "PDF (*.pdf)")  );
}

//-----------------------------------------------------------------------------
void PdfExportDialog::u3dOutputButtonClicked()
{
  m_u3dOutputEdit->setText( QFileDialog::getSaveFileName(this, QString(), QString(), "U3D (*.u3d)") );
}

//-----------------------------------------------------------------------------
void PdfExportDialog::exportButtonClicked()
{
  m_statusLabel->setText("Please wait ...");
  qApp->processEvents();

  SoSeparator* root = NULL;
  try
  {
    QString ivFile = m_inputEdit->text();

    SoInput input;
    if ( !input.openFile(ivFile.toLocal8Bit().data()) ) 
      throw QString("Can not open file " + ivFile);
    root = SoDB::readAll( &input );
    input.closeFile();
    if ( !root ) 
      throw QString("Can not read file " + ivFile);
    root->ref();

    SoSearchAction search;
    search.setType(SoVolumeData::getClassTypeId());
    search.apply(root);

    if (search.getPath()) 
    {
      SoQt::init(this);

      QDialog viewerDialog(this);
      viewerDialog.setModal( true );
      viewerDialog.setWindowTitle("Please, close this window when you are satisfied by VolumeViz refinement.");

      SoQtExaminerViewer* viewer = new SoQtExaminerViewer(&viewerDialog);
      viewer->setSceneGraph(root);
      viewer->viewAll();

      QHBoxLayout * layout = new QHBoxLayout(&viewerDialog);
      layout->addWidget(viewer->getWidget());

      viewerDialog.exec();
    }

    bool result;

    // export to U3D
    if ( m_u3dGroup->isChecked() )
    {
      SoToU3DAction u3dAction;
      QString u3dFile = m_u3dOutputEdit->text();
      result = u3dAction.openFile(u3dFile.toLocal8Bit().data());
      if ( !result ) 
        throw QString("Can not open file " + u3dFile);
      u3dAction.apply( root );
      result = u3dAction.closeFile();
      if ( !result ) 
        throw QString("U3D export failed.");
    }

    if ( m_pdfGroup->isChecked() )
    {
      // export to PDF
      SoToPDFAction pdfAction;
      pdfAction.setPassword(m_writePassEdit->text().toLocal8Bit().data(), m_readPassEdit->text().toLocal8Bit().data());
      pdfAction.setTitle(m_titleEdit->text().toLocal8Bit().data());
      pdfAction.setText(m_textEdit->toPlainText().toLocal8Bit().data());
      pdfAction.setLandscape( m_landscapeCheckBox->isChecked() );

      QString pdfFile = m_pdfOutputEdit->text();
      result = pdfAction.openFile(pdfFile.toLocal8Bit().data());
      if ( !result ) 
        throw QString("Can not open file " + pdfFile);
      pdfAction.apply( root );
      result = pdfAction.closeFile();
      if ( !result ) 
        throw QString("PDF export failed.");
    }

    QMessageBox(QMessageBox::Information, "Success", "Export done.", QMessageBox::Ok, this);
  }
  catch ( QString err )
  {
    QMessageBox(QMessageBox::Critical, "Error", err, QMessageBox::Abort, this);
  }

  if ( root )
    root->unref();

  m_statusLabel->setText( QString() );
}

//-----------------------------------------------------------------------------
void PdfExportDialog::pdfGroupToggled()
{
  m_u3dGroup->setChecked( ! m_pdfGroup->isChecked() );
}

//-----------------------------------------------------------------------------
void PdfExportDialog::u3dGroupToggled()
{
  m_pdfGroup->setChecked( ! m_u3dGroup->isChecked() );
}

//-----------------------------------------------------------------------------
void PdfExportDialog::writePassChanged()
{
  m_readPassEdit->setEnabled( ! m_writePassEdit->text().isEmpty() );
  if ( m_writePassEdit->text().isEmpty() )
    m_readPassEdit->setText( QString() );
}


