// Hack incompatilities between NutCracker and Windows header files
#if defined( _WIN32 )
// EnumPrinters (winspool.h) is not available on NutCracker :
// there are conflicts between VC++ 6SP2 and MKS Toolkit 7.5 NutCracker include files.
#define ENUMPRINTERS_IS_AVAILABLE
#include <wtypes.h>
#include <winspool.h>
#endif

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoDirectionalLight.h>

#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/actions/SoSearchAction.h>

#include <Inventor/nodekits/SoBaseKit.h>

#include <DialogViz/SoDialogVizAll.h>

#include <HardCopy/SoVectorizePSAction.h>
#include <HardCopy/SoVectorizeHPGLAction.h>
#include <HardCopy/SoVectorizeCGMAction.h>
#if defined( _WIN32 )
#include <HardCopy/SoVectorizeGDIAction.h>
#endif

enum PrintFormats
{
  A4_FORMAT = 0,
  A3_FORMAT,
  A2_FORMAT,
  A1_FORMAT,
  A0_FORMAT
};

// Dimensions of A0 to A4 with 20 mm reserved for margins
#define A4_WIDTH 190
#define A4_HEIGHT 277
#define A3_WIDTH A4_HEIGHT
#define A3_HEIGHT 400
#define A2_WIDTH A3_HEIGHT
#define A2_HEIGHT 574
#define A1_WIDTH A2_HEIGHT
#define A1_HEIGHT 820
#define A0_WIDTH A1_HEIGHT
#define A0_HEIGHT 1168

SbVec2f DrawingFormatDims[5] = { SbVec2f( A4_WIDTH, A4_HEIGHT ),
                                 SbVec2f( A3_WIDTH, A3_HEIGHT ),
                                 SbVec2f( A2_WIDTH, A2_HEIGHT ),
                                 SbVec2f( A1_WIDTH, A1_HEIGHT ),
                                 SbVec2f( A0_WIDTH, A0_HEIGHT ) };

int CurrentFormat = A4_FORMAT;

enum controlID
{
  FORMAT_ID = 0,
  HLHSR_MODE_ID,
  LIGHTING_ID,
  SHADING_MODEL_ID,
  PRINT_PS_ID,
  PRINT_HPGL_ID,
  PRINT_CGM_ID,
#if defined( _WIN32 )
  PRINT_GDI_ID,
  PRINT_GDI_PRINTER_ID,
  PRINT_GDI_PRINTER_NAME,
#ifdef ENUMPRINTERS_IS_AVAILABLE
  PRINT_GDI_PRINTER_LIST,
#else /* ENUMPRINTERS_IS_AVAILABLE */
  PRINT_GDI_PRINTER_NAME_AREA,
#endif /* ENUMPRINTERS_IS_AVAILABLE */
#endif
  QUIT_ID,
  NB_ELEMENTS_ID
};

SoRef<SoSeparator> scene = NULL;
SoVectorizeAction::HLHSRMode HLHSRMode = SoVectorizeAction::NO_HLHSR;
SbBool isLightingEnabled = FALSE;
SoVectorizeAction::ShadeModel ShadingModel = SoVectorizeAction::FLAT;

#ifdef ENUMPRINTERS_IS_AVAILABLE
SbString* printerList = NULL;
unsigned char* printerInfoList = NULL;
unsigned long numPrinters = 0;
char* printerName = NULL;
#else
char* printerName = (char*) "\\\\PRINTER SERVER NAME\\PRINTER NAME";
#endif /* ENUMPRINTERS_IS_AVAILABLE */

enum FileFormat
{
  PS = 0,
  HGL,
  CGM,
#if defined( _WIN32 )
  EMF,
#endif
  UNKNOWN
};

const char* fileFormatExtensions[] = { ".ps",
                                       ".hgl",
                                       ".cgm",
#if defined( _WIN32 )
                                       ".emf"
#endif
};

/*---------------------------------------------------------------------------*/

SoSeparator*
readFile( const char* filename )
{
  char message[256];

  // Open the input file
  SoInput mySceneInput;
  if ( !mySceneInput.openFile( filename ) )
  {
    sprintf( message, "Cannot open file %s !\n", filename );
#ifdef _WIN32
    MessageBox( NULL, message, "ivPlot", MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
    fprintf( stderr, message );
#endif
    exit( -1 );
  }

  // Read the whole file into the database
  SoSeparator* myGraph = SoDB::readAll( &mySceneInput );
  if ( myGraph == NULL )
  {
    sprintf( message, "Problem reading file %s\n", filename );
#ifdef _WIN32
    MessageBox( NULL, message, "ivPlot", MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
    fprintf( stderr, message );
#endif
    exit( -1 );
  }
  mySceneInput.closeFile();

  return myGraph;
} /*---------------------------------------------------------------------------*/

void
printPS( SoSeparator* scene, const char* outFile )
{
  SoVectorizeAction* vectAction = new SoVectorizePSAction;
  vectAction->setLineEndStyle( SoVectorizeAction::SQUARE_END );
  vectAction->setDrawingDimensions( DrawingFormatDims[CurrentFormat] );
  vectAction->getVectorOutput()->openFile( outFile );
  vectAction->setHLHSRMode( HLHSRMode );
  vectAction->setShadeModel( ShadingModel );
  vectAction->enableLighting( isLightingEnabled );

  if ( ShadingModel == SoVectorizeAction::SMOOTH )
    ( (SoPSVectorOutput*) vectAction->getVectorOutput() )->setLevel( 3 );
  else
    ( (SoPSVectorOutput*) vectAction->getVectorOutput() )->setLevel( 2 );

  SoDebugError::postInfo( "printPS", "Begin Printing..." );
  vectAction->apply( scene );
  SoDebugError::postInfo( "printPS", "PostScript output generated." );
  vectAction->getVectorOutput()->closeFile();
  delete vectAction;
}

void
printHGL( SoSeparator* scene, const char* outFile )
{
  SoVectorizeAction* vectAction = new SoVectorizeHPGLAction;
  vectAction->setDrawingDimensions( DrawingFormatDims[CurrentFormat] );
  vectAction->getVectorOutput()->openFile( outFile );
  vectAction->setHLHSRMode( HLHSRMode );
  vectAction->enableLighting( isLightingEnabled );
  SoDebugError::postInfo( "printHGL", "Begin Printing..." );
  vectAction->apply( scene );
  SoDebugError::postInfo( "printHGL", "HPGL output generated." );
  vectAction->getVectorOutput()->closeFile();
  delete vectAction;
}

void
printCGM( SoSeparator* scene, const char* outFile )
{
  SoVectorizeAction* vectAction = new SoVectorizeCGMAction;
  vectAction->setDrawingDimensions( DrawingFormatDims[CurrentFormat] );
  ( (SoCGMVectorOutput*) vectAction->getVectorOutput() )->setBinary( TRUE );
  ( (SoCGMVectorOutput*) vectAction->getVectorOutput() )->setIndexed( FALSE );
  vectAction->setHLHSRMode( HLHSRMode );
  vectAction->enableLighting( isLightingEnabled );
  vectAction->getVectorOutput()->openFile( outFile );
  SoDebugError::postInfo( "printCGM", "Begin Printing..." );
  vectAction->apply( scene );
  SoDebugError::postInfo( "printCGM", "CGM output generated." );
  vectAction->getVectorOutput()->closeFile();
  delete vectAction;
}

#if defined( _WIN32 )
void
printGDI( SoSeparator* scene, char* outFile )
{
  SoVectorizeAction* vectAction = new SoVectorizeGDIAction;
  vectAction->getVectorOutput()->openFile( outFile );
  vectAction->setHLHSRMode( HLHSRMode );
  vectAction->enableLighting( isLightingEnabled );
  SoDebugError::postInfo( "printGDI", "Begin Printing..." );
  vectAction->apply( scene );
  SoDebugError::postInfo( "printGDI", "Windows Metafile output generated." );
  vectAction->getVectorOutput()->closeFile();
}
#endif

class FormatAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    CurrentFormat = cpt->selectedItem.getValue();
  }
};

class HLHSRAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    HLHSRMode = (SoVectorizeAction::HLHSRMode) cpt->selectedItem.getValue();
  }
};

class LightingAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* cpt )
  {
    isLightingEnabled = ( cpt->state.getValue() == TRUE );
  }
};

class ShadingModelAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* cpt )
  {
    ShadingModel = ( SoVectorizeAction::ShadeModel )( cpt->state.getValue() == TRUE );
  }
};

class PrintPSAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    printPS( scene.ptr(), "HardCopyOutput.ps" );
  }
};

class PrintHPGLAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    printHGL( scene.ptr(), "HardCopyOutput.hgl" );
  }
};

class PrintCGMAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    printCGM( scene.ptr(), "HardCopyOutput.cgm" );
  }
};

#if defined( _WIN32 )

class PrintGDIAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    printGDI( scene.ptr(), "HardCopyOutput.emf" );
  }
};

class GDIPrinterAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    SoVectorizeAction* vectAction = new SoVectorizeGDIAction;
    SoGDIVectorOutput* output = (SoGDIVectorOutput*) ( vectAction->getVectorOutput() );
    output->openFile();
    vectAction->setHLHSRMode( HLHSRMode );
    vectAction->enableLighting( isLightingEnabled );
    SoDebugError::postInfo( "printGDI", "Begin Printing..." );
    vectAction->apply( scene.ptr() );
    if ( ((SoVectorizeGDIAction*) vectAction)->hasPrinter() )
      SoDebugError::postInfo( "printGDI", "Printing done!" );
    else
      SoDebugError::postInfo( "printGDI", "Printing aborted!" );
    vectAction->getVectorOutput()->closeFile();
  }
};

class GDIPrinterNameAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    // Prints in GDI from the printer (selected) name
    SoVectorizeAction* vectAction = new SoVectorizeGDIAction;
    SoGDIVectorOutput* output = (SoGDIVectorOutput*) ( vectAction->getVectorOutput() );
    HDC printerDC = CreateDC( NULL, printerName, NULL, NULL );
    output->openFile( printerDC );
    vectAction->setHLHSRMode( HLHSRMode );
    vectAction->enableLighting( isLightingEnabled );
    SoDebugError::postInfo( "printGDI", "Begin Printing..." );
    vectAction->apply( scene.ptr() );
    if ( ((SoVectorizeGDIAction*) vectAction)->hasPrinter() )
    {
      SoDebugError::postInfo( "printGDI", "Printing done!" );
    }
    else
    {
      SoDebugError::postInfo( "printGDI", "Printing aborted!" );
    }
    vectAction->getVectorOutput()->closeFile();
  }
};

#ifdef ENUMPRINTERS_IS_AVAILABLE

class PrinterListAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    printerName = (char*) printerList[cpt->selectedItem.getValue()].toLatin1();
  }
};

#else /* ENUMPRINTERS_IS_AVAILABLE */

class PrinterNameAuditor : public SoDialogEditTextAuditor
{
public:
  void dialogEditText( SoDialogEditText* cpt )
  {
    if ( printerName != NULL )
    {
      free( printerName );
    }
    printerName = strdup( cpt->editText.getValue().getString() );
  }
};

#endif /* ENUMPRINTERS_IS_AVAILABLE */

#endif

class QuitAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    exit( 0 );
  }
};

static void
buildDialog( SoTopLevelDialog* dialog, Widget rootWidget )
{
  static SbString printFormats[5] = { "A4 - 210 x 297  mm",
                                      "A3 - 297 x 420  mm",
                                      "A2 - 420 x 594  mm",
                                      "A1 - 594 x 840  mm",
                                      "A0 - 840 x 1188 mm" };

  static SbString HLHSRMode[6] = { "NO_HLHSR",
                                   "HLHSR_SIMPLE_PAINTER",
                                   "HLHSR_PAINTER",
                                   "HLHSR_PAINTER_SURFACE_REMOVAL",
                                   "HLHSR_RASTER",
                                   "HIDDEN_LINES" };

  SoDialogComboBox* format = new SoDialogComboBox();
  format->label.setValue( "Format :" );
  format->items.setValuesPointer( 5, printFormats );
  format->addAuditor( new FormatAuditor() );
  dialog->addChild( format );

  SoDialogComboBox* hlhsrMode = new SoDialogComboBox();
  hlhsrMode->label.setValue( "HLHSR Mode :" );
  hlhsrMode->items.setValuesPointer( 6, HLHSRMode );
  hlhsrMode->addAuditor( new HLHSRAuditor() );
  dialog->addChild( hlhsrMode );

  SoDialogCheckBox* lighting = new SoDialogCheckBox();
  lighting->label.setValue( "Lighting :" );
  lighting->state.setValue( FALSE );
  lighting->onString.setValue( "On" );
  lighting->offString.setValue( "Off" );
  lighting->addAuditor( new LightingAuditor() );
  dialog->addChild( lighting );

  SoDialogCheckBox* shadingModel = new SoDialogCheckBox();
  shadingModel->label.setValue( "Shading model :" );
  shadingModel->state.setValue( FALSE );
  shadingModel->onString.setValue( "Smooth (PS 3 only)" );
  shadingModel->offString.setValue( "Flat" );
  shadingModel->addAuditor( new ShadingModelAuditor() );
  dialog->addChild( shadingModel );

  SoDialogPushButton* printPSButton = new SoDialogPushButton();
  printPSButton->label.setValue( "Print :" );
  printPSButton->buttonLabel.setValue( "HardCopyOutput.ps" );
  printPSButton->addAuditor( new PrintPSAuditor() );
  dialog->addChild( printPSButton );

  SoDialogPushButton* printHPGLButton = new SoDialogPushButton();
  printHPGLButton->label.setValue( "Print :" );
  printHPGLButton->buttonLabel.setValue( "HardCopyOutput.hgl" );
  printHPGLButton->addAuditor( new PrintHPGLAuditor() );
  dialog->addChild( printHPGLButton );

  SoDialogPushButton* printCGMButton = new SoDialogPushButton();
  printCGMButton->label.setValue( "Print :" );
  printCGMButton->buttonLabel.setValue( "HardCopyOutput.cgm" );
  printCGMButton->addAuditor( new PrintCGMAuditor() );
  dialog->addChild( printCGMButton );

#if defined( _WIN32 )
  SoDialogPushButton* printEMFButton = new SoDialogPushButton();
  printEMFButton->label.setValue( "Print :" );
  printEMFButton->buttonLabel.setValue( "HardCopyOutput.emf" );
  printEMFButton->addAuditor( new PrintGDIAuditor() );
  dialog->addChild( printEMFButton );

  SoDialogPushButton* GDIPrinter = new SoDialogPushButton();
  GDIPrinter->label.setValue( "Print :" );
  GDIPrinter->buttonLabel.setValue( "Printer dialog..." );
  GDIPrinter->addAuditor( new GDIPrinterAuditor() );
  dialog->addChild( GDIPrinter );

  SoDialogPushButton* GDIPrinterName = new SoDialogPushButton();
  GDIPrinterName->label.setValue( "Print :" );
  GDIPrinterName->buttonLabel.setValue( "On following printer..." );
  GDIPrinterName->addAuditor( new GDIPrinterNameAuditor() );
  dialog->addChild( GDIPrinterName );

#ifdef ENUMPRINTERS_IS_AVAILABLE
  unsigned long numCopied = 0;
  SbBool anyPrinter = EnumPrinters( PRINTER_ENUM_LOCAL, NULL, 5, printerInfoList, 255, &numCopied, &numPrinters );

  if ( anyPrinter == 0 )
    numPrinters = 0;

  printerList = new SbString[numPrinters];

  for ( unsigned long i = 0; i < numPrinters; i++ )
    printerList[i] = ( ((PRINTER_INFO_5*) printerInfoList)[i] ).pPrinterName;

  SoDialogComboBox* printersChoiceList = new SoDialogComboBox();
  printersChoiceList->label.setValue( "Choose Printer :" );
  if ( numPrinters > 0 )
    printersChoiceList->items.setValuesPointer( numPrinters, printerList );
  printersChoiceList->addAuditor( new PrinterListAuditor() );
  dialog->addChild( printersChoiceList );

#else /* ENUMPRINTERS_IS_AVAILABLE */
  SoDialogEditText* textEdit = new SoDialogEditText();
  textEdit->label.setValue( "Printer name :" );
  textEdit->addAuditor( new PrinterNameAuditor() );
  dialog->addChild( textEdit );
#endif /* ENUMPRINTERS_IS_AVAILABLE */
#endif

  SoDialogPushButton* quitButton = new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialog->addChild( quitButton );

  dialog->buildDialog( rootWidget, FALSE );
} /*---------------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int
main( int argc, char** argv )
{
  bool isCli = ( argc > 2 );

  if ( argc < 2 || argc > 4 )
  {
    printf( "Usage: %s <inputFile.iv> [ <format> [<outputFile>] ]\n"
            "where format is one of : ps, hgl, cgm, emf (Windows only)\n"
            "If only the input file is given, the gui is launched.\n",
            argv[0] );
#if defined( _WIN32 )
    system( "PAUSE" );
#endif
    return 1;
  }

  SoDB::init();
  SoHardCopy::init();

  SoBaseKit::setSearchingChildren( TRUE );

  // Read the file
  scene = readFile( argv[1] );

  if ( isCli )
  {
    isLightingEnabled = TRUE;
    ShadingModel = SoVectorizeAction::SMOOTH;
    HLHSRMode = SoVectorizeAction::HLHSR_RASTER;

    // Search a camera
    SoSearchAction searchAction;
    searchAction.setType( SoCamera::getClassTypeId() );
    searchAction.apply( scene.ptr() );
    if ( !searchAction.getPath() )
    {
      SoPerspectiveCamera* camera = new SoPerspectiveCamera;
      camera->viewAll( scene.ptr(), SbViewportRegion( 1, 1 ) );
      scene->insertChild( camera, 0 );
    }

    // Search a light
    searchAction.setType( SoLight::getClassTypeId() );
    searchAction.apply( scene.ptr() );
    if ( !searchAction.getPath() )
    {
      SoDirectionalLight* light = new SoDirectionalLight;
      scene->insertChild( light, 0 );
    }

    FileFormat fileFormat = UNKNOWN;
    if ( !strcmp( argv[2], "ps" ) )
      fileFormat = PS;
    else if ( !strcmp( argv[2], "hgl" ) )
      fileFormat = HGL;
    else if ( !strcmp( argv[2], "cgm" ) )
      fileFormat = CGM;
#if defined( _WIN32 )
    else if ( !strcmp( argv[2], "emf" ) )
      fileFormat = EMF;
#endif

    if ( fileFormat == UNKNOWN )
    {
      printf( "Unknown file format : %s\n"
              "Please use one of these : ps, hgl, cgm, emf (Windows only)\n",
              argv[2] );
      return 1;
    }

    std::string outFile;
    if ( argc > 3 )
      outFile = argv[3];
    else
      outFile = std::string( argv[1] ) + std::string( fileFormatExtensions[fileFormat] );

    switch ( fileFormat )
    {
    case PS:
      printPS( scene.ptr(), &outFile[0] );
      break;
    case HGL:
      printHGL( scene.ptr(), &outFile[0] );
      break;
    case CGM:
      printCGM( scene.ptr(), &outFile[0] );
      break;
#if defined( _WIN32 )
    case EMF:
      printGDI( scene.ptr(), &outFile[0] );
      break;
#endif
    default:
      break;
    }
  }
  else
  {
    Widget myWindow = SoXt::init( argv[0] );
#if defined( _WIN32 )
#ifdef ENUMPRINTERS_IS_AVAILABLE
  printerInfoList = new unsigned char[255];
#endif /* ENUMPRINTERS_IS_AVAILABLE */
#endif

    SoDialogViz::init();

    SoTopLevelDialog* dialog = new SoTopLevelDialog();
    buildDialog( dialog, myWindow );
    dialog->show();

    // Create a viewer
    SoXtExaminerViewer* myViewer = new SoXtExaminerViewer( myWindow );
    myViewer->setSceneGraph( scene.ptr() );
    myViewer->setTitle( "File Reader" );
    myViewer->viewAll();

    // show viewer
    myViewer->show();

    // Loop forever
    SoXt::show( myWindow );
    SoXt::mainLoop();

    delete myViewer;

    SoDialogViz::finish();
  }

  scene = NULL;

  SoHardCopy::finish();
  if ( !isCli )
    SoXt::finish();
  SoDB::finish();

  return 0;
} /*---------------------------------------------------------------------------*/
