/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
 **=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

/*-----------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 9, example 1.
 *
 *  Printing example.
 *  Read in an Inventor file and display it in ExaminerViewer.  Press
 *  the "p" key and the scene renders into a PostScript
 *  file for printing.
 *----------------------------------------------------------*/

#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoOffscreenRenderArea.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoWWWInline.h>

struct CallbackData
{
  SoXtExaminerViewer* vwr;
  SbString filename;
  SoNode* scene;
};

#ifdef _WIN32
static const SbString defaultExportFilename( "offscreen.bmp" );
static const SbString exportLabel( "Bitmap" );
static const SbColorRGBA exportBackgroundColor( 1., 1., 1., 1. );
#else
static const SbString defaultExportFilename( "offscreen.ps" );
static const SbString exportLabel( "Postscript" );
static const SbColorRGBA exportBackgroundColor( 0., 0., 0., 1. );
#endif

//////////////////////////////////////////////////////////////
// CODE FOR The Inventor Mentor STARTS HERE

SbBool
printTo( SoNode* root, SbString file, SoXtExaminerViewer* viewer, int printerDPI )
{
  // Calculate size of the images in inches which is equal to
  // the size of the viewport in pixels divided by the number
  // of pixels per inch of the screen device.  This size in
  // inches will be the size of the image that will be generated.
  const SbViewportRegion& viewport = viewer->getViewportRegion();
  const SbVec2s& imagePixSize = viewport.getViewportSizePixels();
  const float pixPerInch = viewport.getPixelsPerInch();
  const SbVec2f imageInches( (float) imagePixSize[0] / pixPerInch, (float) imagePixSize[1] / pixPerInch );

  // The resolution to render the scene for the printer
  // is equal to the size of the image in inches times
  // the printer DPI;
  const SbVec2s postScriptRes( (short) ( imageInches[0] * printerDPI ), (short) ( imageInches[1] * printerDPI ) );

  // Create a viewport to render the scene into.
  SbViewportRegion myViewport;
  myViewport.setWindowSize( postScriptRes );
  myViewport.setPixelsPerInch( (float) printerDPI );

  // Create the SoOffscreenRenderArea
  SoRef<SoOffscreenRenderArea> myOffscreen = new SoOffscreenRenderArea( viewer->getNormalSoContext() );
  myOffscreen->setViewportRegion( myViewport );
  myOffscreen->setSceneGraph( root );

  // White/black background uses up a lot less ink...
  myOffscreen->setClearColor( exportBackgroundColor );

  // Render
  return myOffscreen->renderToFile( file );
}

// CODE FOR The Inventor Mentor ENDS HERE
//////////////////////////////////////////////////////////////

static void
processKeyEvents( void* data, SoEventCallback* cb )
{
  if ( SO_KEY_PRESS_EVENT( cb->getEvent(), P ) )
  {
    CallbackData* cbData = static_cast<CallbackData*>(data);

    std::cout << "Printing scene..." << std::endl << std::flush;
    if ( !printTo( cbData->scene, cbData->filename, cbData->vwr, 72 ) )
    {
      std::cerr << "Cannot print image" << std::endl;
      exit( 1 );
    }

    std::cout << "  ...done printing." << std::endl << std::flush;
    cb->setHandled();
  }
}

int
main( int argc, char** argv )
{
  // Initialize Inventor and Xt
  Widget appWindow = SoXt::init( argv[0] );
  if ( appWindow == NULL )
  {
    exit( 1 );
  }
  std::cout << "To print the scene: press the 'p' key while in picking mode" << std::endl;

  // Make a scene containing an event callback node
  SoRef<SoSeparator> root = new SoSeparator;
  SoEventCallback* eventCB = new SoEventCallback;
  root->addChild( eventCB );

  // Read the geometry from a file and add to the scene

  const char *defaultFileName = "$OIVHOME/data/models/toys/jackInTheBox.iv";
  const char* fileName = ( argc >= 2 ? argv[1] : defaultFileName );

  SoInput myInput;
  if ( myInput.openFile( fileName ) == FALSE )
  {
    std::cerr << "Usage: " << argv[0] << "filename.iv" << std::endl;
    std::cerr << "Could not open file " << argv[1] << std::endl;
    exit( 1 );
  }
  SoWWWInline::setReadAsSoFile( TRUE );

  SoSeparator* geomObject = SoDB::readAll( &myInput );
  if ( geomObject == NULL )
  {
    exit( 1 );
  }
  root->addChild( geomObject );

  SoXtExaminerViewer* viewer = new SoXtExaminerViewer( appWindow, NULL, TRUE, SoXtExaminerViewer::BUILD_ALL, SoXtExaminerViewer::EDITOR );
  viewer->setSceneGraph( root.ptr() );
  viewer->setTitle( "Print to " + exportLabel );

  // Setup the event callback data and routine for performing the print
  CallbackData* data = new CallbackData;
  data->vwr = viewer;
  data->filename = defaultExportFilename;
  if ( argc >= 3 )
    data->filename = argv[2];
  data->scene = viewer->getSceneGraph();
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), processKeyEvents, data );
  viewer->show();

  SoXt::show( appWindow );
  SoXt::mainLoop();

  root = NULL;
  delete data;
  delete viewer;
  SoXt::finish();

  return 0;
}
