/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
/*-----------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 9, example 2.
 *
 *  Using the offscreen renderarea to generate a texture map.
 *  Generate simple scene and grab the image to use as
 *  a texture map.
 *----------------------------------------------------------*/



#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoOffscreenRenderArea.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>


SbBool 
generateTextureMap(SoNode *root, SoTexture2 *texture, 
                   short textureWidth, short textureHeight)
{
  SoRef<SoOffscreenRenderArea> myRenderer = new SoOffscreenRenderArea;
  myRenderer->setSceneGraph(root);
  myRenderer->setSize( SbVec2i32( textureWidth, textureHeight ) );
  myRenderer->setClearColor(SbColorRGBA(0.3f, 0.3f, 0.3f, 1.f));

  SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject;
  myRenderer->renderToBuffer(buffer.ptr());

  // Generate the texture
  texture->image.setValue(SbVec2s(textureWidth, textureHeight), 3,
    SoSFImage::UNSIGNED_BYTE, buffer.ptr(), SoSFImage::COPY);

  return TRUE;
}

int
main(int, char **argv)
{
  // Initialize Inventor and Xt
  Widget appWindow = SoXt::init(argv[0]);
  if (appWindow == NULL)
    exit(1);
  
  // Make a scene from reading in a file
  SoSeparator *texRoot = new SoSeparator;
  
  SoInput in;
  SoNode *result;
  
  texRoot->ref();
  in.openFile("$OIVHOME/examples/data/Inventor/jumpyMan.iv");
  SoDB::read(&in, result);
  
  SoPerspectiveCamera *myCamera = new SoPerspectiveCamera;
  SoRotationXYZ *rot = new SoRotationXYZ;
  rot->axis  = SoRotationXYZ::X;
  rot->angle = (float)M_PI_2;
  myCamera->position.setValue(SbVec3f(-0.2f, -0.2f, 2.0f));
  myCamera->scaleHeight(0.4f); 
  texRoot->addChild(myCamera);
  texRoot->addChild(new SoDirectionalLight);
  texRoot->addChild(rot);
  texRoot->addChild(result);
  
  // Generate the texture map
  SoTexture2 *texture = new SoTexture2;
  if (generateTextureMap(texRoot, texture, 64, 64))
    printf ("Successfully generated texture map\n");
  else
    printf ("Could not generate texture map\n");
  texRoot->unref();
  
  // Make a scene with a cube and apply the texture to it
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild(texture);
  root->addChild(new SoCube);
  
  // Initialize an Examiner Viewer
  SoXtExaminerViewer *viewer =
    new SoXtExaminerViewer(appWindow);
  viewer->setSceneGraph(root);
  viewer->setTitle("Offscreen Rendered Texture");
  
  // In Inventor 2.1, if the machine does not have hardware texture
  // mapping, we must override the default drawStyle to display textures.
  viewer->setDrawStyle(SoXtViewer::STILL, SoXtViewer::VIEW_AS_IS);
  
  viewer->show();
  
  SoXt::show(appWindow);
  SoXt::mainLoop();

  root->unref();
  delete viewer;
  SoXt::finish();
  
  return 0;
}


