/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

/*-------------------------------------------------------------
 *  This is an example from The Inventor Mentor
 *  chapter 10, example 2.
 *
 *  This demonstrates using SoXtRenderArea::setEventCallback().
 *  which causes events to be sent directly to the application
 *  without being sent into the scene graph.
 *  
 * Clicking the left mouse button and dragging will draw 
 *       points in the xy plane beneath the mouse cursor.
 * Clicking middle mouse and holding causes the point set 
 *       to rotate about the Y axis. 
 * Clicking right mouse clears all points drawn so far out 
 *       of the point set.
 *-----------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPointSet.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/sensors/SoTimerSensor.h>

// Timer sensor : rotate 90 degrees every second, update 30 times a second
SoTimerSensor *myTicker;
#define UPDATE_RATE 1.0f/30.0f
#define ROTATION_ANGLE M_PI/60.0f

void
myProjectPoint(SoXtExaminerViewer *myViewer, 
               int mousex, int mousey, SbVec3f &intersection)
{
  // Take the x,y position of mouse, and normalize to [0,1].
  // X windows have 0,0 at the upper left,
  // Inventor expects 0,0 to be the lower left.
  SbVec2s size = myViewer->getSceneManager()->getSize();
  float x = float(mousex) / size[0];
  float y = float(size[1] - mousey) / size[1];
  
  // Get the camera and view volume
  SoGroup *root = (SoGroup *) myViewer->getSceneGraph();
  SoCamera *myCamera = (SoCamera *) root->getChild(0);

  SbViewVolume myViewVolume;
  myViewVolume = myCamera->getViewVolume();
  
  // Project the mouse point to a line
  SbVec3f p0, p1;
  myViewVolume.projectPointToLine(SbVec2f(x,y), p0, p1);
  
  // Midpoint of the line intersects a plane thru the origin
  intersection = (p0 + p1) / 2.0f;
}

void
myAddPoint(SoXtExaminerViewer *, const SbVec3f point)
{
  SoCoordinate3 *coord = (SoCoordinate3 *)SoNode::getByName("MyCoords");
  SoPointSet *pointSet = (SoPointSet *)SoNode::getByName("MyPointSet");
  
  coord->point.set1Value(coord->point.getNum(), point);
  pointSet->numPoints.setValue(coord->point.getNum());
}

void
myClearPoints(SoXtExaminerViewer *)
{
  SoCoordinate3 *coord = (SoCoordinate3 *)SoNode::getByName("MyCoords");
  SoPointSet *pointSet = (SoPointSet *)SoNode::getByName("MyPointSet");
  
  // Delete all values starting from 0
  coord->point.deleteValues(0); 
  pointSet->numPoints.setValue(0);
}

void
tickerCallback(void *userData, SoSensor *)
{
  SoCamera *myCamera = (SoCamera *) userData;
  SbRotation rot;
  SbMatrix mtx;
  SbVec3f pos;
  
  // Adjust the position
  pos = myCamera->position.getValue();
  rot = SbRotation(SbVec3f(0.0f, 1.0f, 0.0f), (float)(ROTATION_ANGLE));
  mtx.setRotate(rot);
  mtx.multVecMatrix(pos, pos);
  myCamera->position.setValue(pos);
  
  // Adjust the orientation
  myCamera->orientation.setValue(myCamera->orientation.getValue() * rot);
}

///////////////////////////////////////////////////////////////
// CODE FOR The Inventor Mentor STARTS HERE  (part 1)

SbBool
myAppEventHandler(void *userData, XAnyEvent *anyevent)
{
  SoXtExaminerViewer *myViewer = (SoXtExaminerViewer *)userData;
  SbVec3f vec;
  SbBool handled = TRUE;

#ifdef _WIN32
  // Hack to remember if left-mouse or shift-left-mouse (which is
  // logically middle-mouse) was pressed
  static SbBool leftIsMiddle = FALSE;
  
  switch (anyevent->message) {
    
  case WM_LBUTTONDOWN:
    {
      // let shift-left-mouse be a synonym for middle-mouse on Win32
      if (anyevent->wParam & MK_SHIFT) {
        leftIsMiddle = TRUE;
        myTicker->schedule();
      }
      else {
        leftIsMiddle = FALSE;
        POINT *myButtonEvent = &anyevent->pt;
        myProjectPoint(myViewer, myButtonEvent->x, myButtonEvent->y, vec);
        myAddPoint(myViewer, vec);
      }
    }
    break;
    
  case WM_MBUTTONDOWN:
    {
      myTicker->schedule();  // start spinning the camera
    }
    break;
    
  case WM_RBUTTONDOWN:
    {
      myClearPoints(myViewer);  // clear the point set
    }
    break;
    
  case WM_LBUTTONUP:
    {
      // let shift-left-mouse be a synonym for middle-mouse on Win32
      if (leftIsMiddle) {
        myTicker->unschedule();
      }
    }
    break;
    
  case WM_MBUTTONUP:
    {
      myTicker->unschedule();  // stop spinning the camera
    }
    break;
    
  case WM_MOUSEMOVE:
    {
      POINT *myMotionEvent = &anyevent->pt;
      // let shift-left-mouse be a synonym for middle-mouse on Win32
      if ((anyevent->wParam & MK_LBUTTON) && !leftIsMiddle) {
        myProjectPoint(myViewer, myMotionEvent->x, myMotionEvent->y, vec);
        myAddPoint(myViewer, vec);
      }
    }
    break;
    
  default:
    {
      handled = FALSE;
    }
    break;
  }

#else

  switch (anyevent->type) {
    
  case ButtonPress:
    {
      XButtonEvent *myButtonEvent = (XButtonEvent *) anyevent;
      if (myButtonEvent->button == Button1) {
        myProjectPoint(myViewer, 
          myButtonEvent->x, myButtonEvent->y, vec);
        myAddPoint(myViewer, vec);
      }
      else if (myButtonEvent->button == Button2) {
        myTicker->schedule();  // start spinning the camera
      }
      else if (myButtonEvent->button == Button3) {
        myClearPoints(myViewer);  // clear the point set
      }
    }
    break;
    
  case ButtonRelease:
    {
      XButtonEvent *myButtonEvent = (XButtonEvent *) anyevent;
      if (myButtonEvent->button == Button2) {
        myTicker->unschedule();  // stop spinning the camera
      }
    }
    break;
    
  case MotionNotify:
    {
      XMotionEvent *myMotionEvent = (XMotionEvent *) anyevent;
      if (myMotionEvent->state & Button1Mask) {  
        myProjectPoint(myViewer, myMotionEvent->x, myMotionEvent->y, vec);
        myAddPoint(myViewer, vec);
      }
    }
    break;
    
  default:
    {
      handled = FALSE;
    }
    break;
  }
#endif
  
  return handled;
}

// CODE FOR The Inventor Mentor ENDS HERE
////////////////////////////////////////////////////////////

int
main(int, char **argv)
{
  // Print out usage instructions
  printf("Mouse buttons:\n");
  printf("\tLeft (with mouse motion): adds points\n");
  printf("\tMiddle: rotates points about the Y axis\n");
  printf("\tRight: deletes all the points\n");
  
  // Initialize Inventor and Xt
  Widget appWindow = SoXt::init(argv[0]);
  if (appWindow == NULL) exit(1);
  
  // Create and set up the root node
  SoSeparator *root = new SoSeparator;
  root->ref();
  
  // Add a camera
  SoPerspectiveCamera *myCamera = new SoPerspectiveCamera;
  myCamera->viewportMapping = SoCamera::LEAVE_ALONE;
  root->addChild(myCamera);
  
  // Use the base color light model so we don't need to 
  // specify normals
  SoLightModel *myLightModel = new SoLightModel;
  myLightModel->model = SoLightModel::BASE_COLOR;
  root->addChild(myLightModel);
  
  // Set up the camera view volume
  myCamera->position.setValue(0.0f, 0.0f, 4.0f);
  myCamera->nearDistance.setValue(1.0f);
  myCamera->farDistance.setValue(7.0f);
  myCamera->heightAngle.setValue((float)(M_PI / 3.0f));
  
  // Add a coordinate and point set
  SoCoordinate3 *myCoord = new SoCoordinate3;
  SoPointSet *myPointSet = new SoPointSet;
  myCoord->setName("MyCoords");
  myPointSet->setName("MyPointSet");
  root->addChild(myCoord);
  root->addChild(myPointSet);
  
  // Timer sensor to tick off time while middle mouse is down
  myTicker = new SoTimerSensor(tickerCallback, myCamera);
  myTicker->setInterval(UPDATE_RATE);
  
  // Create a render area for viewing the scene
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(appWindow);
  myViewer->setPopupMenuEnabled(FALSE);
  myViewer->setSceneGraph(root);
  myViewer->setDecoration(FALSE);
  myViewer->setViewing(FALSE);
  myViewer->setTitle("My Event Handler");
  
  //////////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE  (part 2)
  
  // Have render area send events to us instead of the scene 
  // graph.  We pass the render area as user data.
  myViewer->setEventCallback(myAppEventHandler, myViewer);
  
  // CODE FOR The Inventor Mentor ENDS HERE
  //////////////////////////////////////////////////////////////
  
  // Show our application window, and loop forever...
  myViewer->show();
  SoXt::show(appWindow);
  SoXt::mainLoop();

  delete myViewer;
  SoXt::finish();

  return 0;
}

