/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

/*-------------------------------------------------------------
 *  This is an example from The Inventor Mentor,
 *  chapter 10, example 8.
 *
 *  This example demonstrates the use of the pick filter
 *  callback to always select nodekits. This makes it especially
 *  easy to edit attributes of objects because the nodekit takes
 *  care of the part creation details.
 *------------------------------------------------------------*/

#if !defined(_WIN32) && !defined(__APPLE__)
#  include <X11/StringDefs.h>
#  include <X11/Intrinsic.h>
#endif

#include <Inventor/SoPath.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtMaterialEditor.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/actions/SoBoxHighlightRenderAction.h>
#include <Inventor/nodekits/SoShapeKit.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoTransform.h>

typedef struct {
  SoSelection *sel;
  SoXtMaterialEditor *editor;
  SbBool ignore;
} UserData;

//////////////////////////////////////////////////////////////
// CODE FOR The Inventor Mentor STARTS HERE  (part 1)

// Truncate the pick path so a nodekit is selected
SoPath *
pickFilterCB(void *, const SoPickedPoint *pick)
{    
  // See which child of selection got picked
  SoPath *p = pick->getPath();
  int i;
  for (i = p->getLength() - 1; i >= 0; i--) {
    SoNode *n = p->getNode(i);
    if (n->isOfType(SoShapeKit::getClassTypeId()))
      break;
  }
  
  // Copy the path down to the nodekit
  return p->copy(0, i+1);
}

// CODE FOR The Inventor Mentor ENDS HERE  
///////////////////////////////////////////////////////////////

// Create a sample scene graph
SoNode *
buildScene()
{
  SoGroup *g = new SoGroup;
  SoShapeKit *k;
  SoTransform *xf;
  
  // Place a dozen shapes in circular formation
  for (int i = 0; i < 12; i++) {
    k = new SoShapeKit;
    k->setPart("shape", new SoCube);
    xf = k->getPart<SoTransform>("transform");
    xf->translation.setValue((float)(8 * sin(i * M_PI / 6)),
                             (float)(8 * cos(i * M_PI / 6)), 
                             0.0F);
    g->addChild(k);
  }
  
  return g;
}

// Update the material editor to reflect the selected object
void
selectCB(void *userData, SoPath *path)
{
  SoShapeKit *kit = (SoShapeKit *)path->getTail();
  SoMaterial* kitMtl = kit->getPart<SoMaterial>( "material");
  
  UserData *ud = (UserData *) userData;
  ud->ignore = TRUE;
  ud->editor->setMaterial(*kitMtl);
  ud->ignore = FALSE;
}

// This is called when the user chooses a new material
// in the material editor. This updates the material
// part of each selected node kit.
void
mtlChangeCB(void *userData, const SoMaterial *mtl)
{
  // Our material change callback is invoked when the
  // user changes the material, and when we change it
  // through a call to SoXtMaterialEditor::setMaterial.
  // In this latter case, we ignore the callback invocation.
  UserData *ud = (UserData *) userData;
  if (ud->ignore) {
    return;
  }
  SoSelection *sel = ud->sel;
  
  // Our pick filter guarantees the path tail will
  // be a shape kit.
  for (int i = 0; i < sel->getNumSelected(); i++) {
    SoPath *p = sel->getPath(i);
    SoShapeKit *kit = (SoShapeKit *) p->getTail();
    SoMaterial *kitMtl = kit->getPart<SoMaterial>("material");
    kitMtl->copyFieldValues(mtl);
  }
}

int
main(int, char **argv)
{
  // Initialization
  Widget mainWindow = SoXt::init(argv[0]);
  
  // Create our scene graph.
  SoSelection *sel = new SoSelection;
  sel->ref();
  sel->addChild(buildScene());
  
  // Create a viewer with a render action that displays highlights
  SoXtExaminerViewer *viewer = new SoXtExaminerViewer(mainWindow);
  viewer->setSceneGraph(sel);
  viewer->setGLRenderAction(new SoBoxHighlightRenderAction());
  viewer->redrawOnSelectionChange(sel);
  viewer->setTitle("Select Node Kits");
  viewer->show();
  
  // Create a material editor
  SoXtMaterialEditor *ed = new SoXtMaterialEditor();
  ed->show();
  
  // User data for our callbacks
  UserData userData;
  userData.sel = sel;
  userData.editor = ed;
  userData.ignore = FALSE;
  
  // Selection and material change callbacks
  ed->addMaterialChangedCallback(mtlChangeCB, &userData);
  sel->setPickFilterCallback(pickFilterCB);
  sel->addSelectionCallback(selectCB, &userData);
  
  SoXt::show(mainWindow);
  SoXt::mainLoop();

  delete viewer;
  SoXt::finish();

  return 0;
}



