/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

/*--------------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 12, example 4.
 *
 *  Timer sensors.  An object is rotated by a timer sensor.
 *  (called "rotatingSensor").  The interval between calls 
 *  controls how often it rotates.
 *  A second timer (called "schedulingSensor") goes off
 *  every 5 seconds and changes the interval of 
 *  "rotatingSensor".  The interval alternates between
 *  once per second and 10 times per second.
 *  This example could also be done using engines.
 *------------------------------------------------------------*/

#include <Inventor/SoDB.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/nodes/SoWWWInline.h>


///////////////////////////////////////////////////////////
// CODE FOR The Inventor Mentor STARTS HERE

// This function is called either 10 times/second or once every
// second; the scheduling changes every 5 seconds (see below):
static void
rotatingSensorCallback(void *data, SoSensor *)
{
  // Rotate an object...
  SoRotation *myRotation = (SoRotation *)data;
  SbRotation currentRotation = myRotation->rotation.getValue();
  currentRotation = 
    SbRotation(SbVec3f(0.0f, 0.0f, 1.0f), 
    (float)(M_PI/90.0f)) * currentRotation;
  myRotation->rotation.setValue(currentRotation);
}

// This function is called once every 5 seconds, and
// reschedules the other sensor.
static void
schedulingSensorCallback(void *data, SoSensor *)
{
  SoTimerSensor *rotatingSensor = (SoTimerSensor *)data;
  rotatingSensor->unschedule();
  if (rotatingSensor->getInterval() == 1.0f)
    rotatingSensor->setInterval(1.0f/10.0f);
  else rotatingSensor->setInterval(1.0f);
  rotatingSensor->schedule();
}

// CODE FOR The Inventor Mentor ENDS HERE
///////////////////////////////////////////////////////////

int
main(int argc, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) exit(1);
  
  SoSeparator *root = new SoSeparator;
  root->ref();
  
  ///////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE
  
  SoRotation *myRotation = new SoRotation;
  root->addChild(myRotation);
  
  SoTimerSensor *rotatingSensor =
    new SoTimerSensor(rotatingSensorCallback, myRotation);
  rotatingSensor->setInterval(1.0f); // scheduled once per second
  rotatingSensor->schedule();
  
  SoTimerSensor *schedulingSensor =
    new SoTimerSensor(schedulingSensorCallback, rotatingSensor);
  schedulingSensor->setInterval(5.0f); // once per 5 seconds
  schedulingSensor->schedule();
  
  // CODE FOR The Inventor Mentor ENDS HERE
  ///////////////////////////////////////////////////////////
  const char *defaultFileName = "$OIVHOME/data/models/toys/jackInTheBox.iv";

  const char *fileName = 
    (argc >= 2 ? argv[1] : defaultFileName);
  
  SoInput inputFile;
  if (inputFile.openFile(fileName) == FALSE) {
    fprintf(stderr, "Usage: %s filename.iv\n", argv[0]);
    fprintf(stderr, "Could not open file %s\n", argv[1]);
    exit(1);
  }
  SoWWWInline::setReadAsSoFile(TRUE);

  root->addChild(SoDB::readAll(&inputFile));
  
  SoXtExaminerViewer *myViewer =
    new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Two Timers");
  myViewer->show();
  
  SoXt::show(myWindow);  // Display main window
  SoXt::mainLoop();        // Main Inventor event loop

  root->unref();

  delete myViewer;
  SoXt::finish();

  return 0;
}


