/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

/*----------------------------------------------------------------
 *  This is an example from the Inventor Mentor
 *  chapter 14, example 2.
 *
 *  Use nodekits to create a scene with a desk into an 
 *  SoWrapperKit.  Then, add a material editor for the desk and 
 *  a light editor on the light.
 *  
 *  The scene is organized using an SoSceneKit, which contains
 *  lists for grouping lights (lightList), cameras (cameraList), 
 *  and objects (childList) in a scene.
 *  
 *  Once the scene is created, a material editor is attached to 
 *  the wrapperKit's 'material' part, and a directional light editor
 *  is attached to the light's 'directionalLight' part.
 *----------------------------------------------------------------*/



#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtDirectionalLightEditor.h>
#include <Inventor/Xt/SoXtMaterialEditor.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/nodekits/SoCameraKit.h>
#include <Inventor/nodekits/SoLightKit.h>
#include <Inventor/nodekits/SoSceneKit.h>
#include <Inventor/nodekits/SoWrapperKit.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>


int
main(int, char **argv)
{
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL)
    exit(1);
  // SCENE!
  SoSceneKit *myScene = new SoSceneKit;
  myScene->ref();
  
  // LIGHTS! Add an SoLightKit to the "lightList." The 
  // SoLightKit creates an SoDirectionalLight by default.
  myScene->setPart("lightList[0]", new SoLightKit);
  
  // CAMERA!! Add an SoCameraKit to the "cameraList." The 
  // SoCameraKit creates an SoPerspectiveCamera by default.
  myScene->setPart("cameraList[0]", new SoCameraKit);
  myScene->setCameraNumber(0);
  
  // Read an object from file. 
  SoInput myInput;
  if (!myInput.openFile("$OIVHOME/examples/data/Inventor/desk.iv")) 
    exit(1);
  SoSeparator *fileContents = SoDB::readAll(&myInput);
  if (fileContents == NULL)
    exit(1);
  
  // OBJECT!! Create an SoWrapperKit and set its contents to
  // be what you read from file.
  SoWrapperKit *myDesk = new SoWrapperKit();
  myDesk->setPart("contents", fileContents);
  myScene->setPart("childList[0]", myDesk);
  
  // Give the desk a good starting color
  myDesk->set((char*)"appearance.material { diffuseColor .8 .3 .1 }");
  
  // MATERIAL EDITOR!!  Attach it to myDesk's material node.
  // Use the SO_GET_PART macro to get this part from myDesk.
  SoXtMaterialEditor *mtlEditor = new SoXtMaterialEditor();
  SoMaterial *mtl = SO_GET_PART(myDesk,"appearance.material",SoMaterial);
  mtlEditor->setTitle("Material of Desk");
  mtlEditor->show();
  mtlEditor->attach(mtl);
  
  // DIRECTIONAL LIGHT EDITOR!! Attach it to the 
  // SoDirectionalLight node within the SoLightKit we made.
  SoXtDirectionalLightEditor *ltEditor = 
    new SoXtDirectionalLightEditor();
  SoPath *ltPath = 
    myScene->createPathToPart("lightList[0].light", TRUE);
  ltEditor->setTitle("Lighting of Desk");
  ltEditor->show();
  ltEditor->attach(ltPath);
  
  SoXtRenderArea *myRenderArea = new SoXtRenderArea(myWindow);
  // Set up Camera with ViewAll...
  // -- use the SO_GET_PART macro to get the camera node.
  // -- viewall is a method on the 'camera' part of 
  //    the cameraKit, not on the cameraKit itself.  So the part
  //    we ask for is not 'cameraList[0]' (which is of type 
  //    SoPerspectiveCameraKit), but 
  //    'cameraList[0].camera' (which is of type 
  //    SoPerspectiveCamera).
  SoPerspectiveCamera *myCamera = 
    SO_GET_PART(myScene, "cameraList[0].camera", SoPerspectiveCamera);
  SbViewportRegion myRegion(myRenderArea->getSize());
  myCamera->viewAll(myScene, myRegion);

  myRenderArea->setSceneGraph(myScene);
  myRenderArea->setTitle("Main Window: Desk In A Scene Kit");
  myRenderArea->show();
  
  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete myRenderArea;
  SoXt::finish();

  return 0;
}



