/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

/*----------------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 14, example 3.
 *
 *  This example illustrates the creation of motion hierarchies
 *  using nodekits by creating a model of a balance-style scale.
 *
 *  It adds an SoEventCallback to the "callback" list in the 
 *      nodekit called 'support.'
 *  The callback will have the following response to events:
 *  Pressing right arrow key == lower the right pan
 *  Pressing left arrow key  == lower the left pan
 *  The pans are lowered by animating three rotations in the 
 *      motion hierarchy.
 *  Use an SoText2Kit to print instructions to the user as part
 *      of the scene.
 *----------------------------------------------------------------*/


#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodekits/SoCameraKit.h>
#include <Inventor/nodekits/SoLightKit.h>
#include <Inventor/nodekits/SoSceneKit.h>
#include <Inventor/nodekits/SoShapeKit.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtRenderArea.h>

#include <Inventor/actions/SoWriteAction.h>


// Callback Function for Animating the Balance Scale.
// -- used to make the balance tip back and forth
// -- Note: this routine is only called in response to KeyPress
//    events since the call 'setEventInterest(KeyPressMask)' is
//    made on the SoEventCallback node that uses it.
// -- The routine checks if the key pressed was left arrow (which
//    is XK_Left in X-windows talk), or right arrow (which is
//    XK_Right)
// -- The balance is made to tip by rotating the beam part of the
//    scale (to tip it) and then compensating (making the strings
//    vertical again) by rotating the string parts in the opposite
//    direction.
void
tipTheBalance(void *userData, // The nodekit representing 'support', the
              // fulcrum of the balance. Passed in during
              // main routine, below. 
              SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();
  
  // Which Key was pressed?
  // If Right or Left Arrow key, then continue...
  if (SO_KEY_PRESS_EVENT(ev, RIGHT_ARROW) || 
    SO_KEY_PRESS_EVENT(ev, LEFT_ARROW)) {
    SoShapeKit  *support, *beam1, *string1, *string2;
    SbRotation  startRot, beamIncrement, stringIncrement;
    
    // Get the different nodekits from the userData.
    support = (SoShapeKit *) userData;
    
    // These three parts are extracted based on knowledge of the
    // motion hierarchy (see the diagram in the main routine.
    beam1 = support->getPart<SoShapeKit>( "childList[0]" );
    string1 = beam1->getPart<SoShapeKit>( "childList[0]" );
    string2 = beam1->getPart<SoShapeKit>( "childList[1]" );
    
    // Set angular increments to be .1 Radians about the Z-Axis
    // The strings rotate opposite the beam, and the two types
    // of key press produce opposite effects.
    if (SO_KEY_PRESS_EVENT(ev, RIGHT_ARROW)) {
      beamIncrement.setValue(SbVec3f(0.0f, 0.0f, 1.0f), -0.1f);
      stringIncrement.setValue(SbVec3f(0.0f, 0.0f, 1.0f), 0.1f);
    } 
    else {
      beamIncrement.setValue(SbVec3f(0.0f, 0.0f, 1.0f), 0.1f);
      stringIncrement.setValue(SbVec3f(0.0f, 0.0f, 1.0f), -0.1f);
    }
    
    // Use SO_GET_PART to find the transform for each of the 
    // rotating parts and modify their rotations.
    
    SoTransform *xf;
    xf = SO_GET_PART(beam1, "transform", SoTransform);
    startRot = xf->rotation.getValue();
    xf->rotation.setValue(startRot *  beamIncrement);
    
    xf = SO_GET_PART(string1, "transform", SoTransform);
    startRot = xf->rotation.getValue();
    xf->rotation.setValue(startRot *  stringIncrement);
    
    xf = SO_GET_PART(string2, "transform", SoTransform);
    startRot = xf->rotation.getValue();
    xf->rotation.setValue(startRot *  stringIncrement);
    
    eventCB->setHandled();
  }
}

int
main(int, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) exit(1);
  
  SoSceneKit *myScene = new SoSceneKit;
  myScene->ref();
  
  myScene->setPart("lightList[0]", new SoLightKit);
  myScene->setPart("cameraList[0]", new SoCameraKit);
  myScene->setCameraNumber(0);
  
  // Create the Balance Scale -- put each part in the 
  // childList of its parent, to build up this hierarchy:
  //
  //                    myScene
  //                       |
  //                     support
  //                       |
  //                     beam
  //                       |
  //                   --------
  //                   |       |
  //                string1  string2
  //                   |       |
  //                tray1     tray2
  
  SoShapeKit *support = new SoShapeKit();
  support->setPart("shape", new SoCone);
  support->set((char*)"shape { height 3 bottomRadius .3 }");
  myScene->setPart("childList[0]", support);
  
  SoShapeKit *beam = new SoShapeKit();
  beam->setPart("shape", new SoCube);
  beam->set((char*)"shape { width 3 height .2 depth .2 }");
  beam->set((char*)"transform { translation 0 1.5 0 } ");
  support->setPart("childList[0]", beam);
  
  SoShapeKit *string1 = new SoShapeKit;
  string1->setPart("shape", new SoCylinder);
  string1->set((char*)"shape { radius .05 height 2}");
  string1->set((char*)"transform { translation -1.5 -1 0 }");
  string1->set((char*)"transform { center 0 1 0 }");
  beam->setPart("childList[0]", string1);
  
  SoShapeKit *string2 = new SoShapeKit;
  string2->setPart("shape", new SoCylinder);
  string2->set((char*)"shape { radius .05 height 2}");
  string2->set((char*)"transform { translation 1.5 -1 0 } ");
  string2->set((char*)"transform { center 0 1 0 } ");
  beam->setPart("childList[1]", string2);
  
  SoShapeKit *tray1 = new SoShapeKit;
  tray1->setPart("shape", new SoCylinder);
  tray1->set((char*)"shape { radius .75 height .1 }");
  tray1->set((char*)"transform { translation 0 -1 0 } ");
  string1->setPart("childList[0]", tray1);
  
  SoShapeKit *tray2 = new SoShapeKit;
  tray2->setPart("shape", new SoCylinder);
  tray2->set((char*)"shape { radius .75 height .1 }");
  tray2->set((char*)"transform { translation 0 -1 0 } ");
  string2->setPart("childList[0]", tray2);
  
  // Add EventCallback so Balance Responds to Events
  SoEventCallback *myCallbackNode = new SoEventCallback;
  myCallbackNode->addEventCallback(
    SoKeyboardEvent::getClassTypeId(), 
    tipTheBalance, support); 
  support->setPart("callbackList[0]", myCallbackNode);
  
  // Add Instructions as Text in the Scene...
  SoShapeKit *myText = new SoShapeKit;
  myText->setPart("shape", new SoText2);
  myText->set((char*)"shape { string \"Press Left or Right Arrow Key\" }");
  myText->set((char*)"shape { justification CENTER }");
  myText->set((char*)"font { name \"Helvetica-Bold\" }");
  myText->set((char*)"font { size 16.0 }");
  myText->set((char*)"transform { translation 0 -2 0 }");
  myScene->setPart("childList[1]", myText);
  
  SoXtRenderArea *myRenderArea = new SoXtRenderArea(myWindow);
  
  // Get camera from scene and tell it to viewAll...
  SoPerspectiveCamera *myCamera = SO_GET_PART(myScene,
    "cameraList[0].camera", SoPerspectiveCamera);
  myCamera->viewAll(myScene, myRenderArea->getSize());
  
  myRenderArea->setSceneGraph(myScene);
  myRenderArea->setTitle("Balance Scale Made of Nodekits");
  myRenderArea->show();
  
  SoXt::show(myWindow);
  SoXt::mainLoop();

  myScene->unref();

  delete myRenderArea;
  SoXt::finish();

  return 0;
}


