/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
/*-----------------------------------------------------------
 *  This is an example from The Inventor Mentor,
 *  chapter 17, example 3.
 * 
 * This example draws the same scene as Example 17.2, 
 *  but using a GLX window.
 *---------------------------------------------------------*/

#ifdef _WIN32
#  include <windows.h>
#  include <Inventor/Win/SoWin.h>
#else
#  include <Inventor/sys/SoGLX.h>
#  include <unistd.h>
#endif

#include <Inventor/sys/SoGL.h>
#include <Inventor/sys/SoGLU.h>
#include <stdio.h>

#include <Inventor/SoDB.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/SoSceneManager.h>

#define WINWIDTH 400
#define WINHEIGHT 400

static const float g_cameraFovY = 90.0f;
static const float g_cameraAspect = float(WINWIDTH) / float(WINHEIGHT);
static const float g_cameraNear = 2.0f;
static const float g_cameraFar = 12.0f;
static const float g_cameraPosZ = 5.0f;
static const SbColor g_backgroundColor(0.8f, 0.8f, 0.8f);

float floorObj[81][3];

// Build an Inventor scene with two objects and some light
void
buildScene(SoGroup *root)
{
  SoPerspectiveCamera* camera = new SoPerspectiveCamera;
  camera->position.setValue( 0.0f, 0.0f, g_cameraPosZ );
  camera->orientation.setValue( SbVec3f(0.0f, 0.0f, 1.0f), 0.0f );
  camera->nearDistance.setValue( g_cameraNear );
  camera->farDistance.setValue( g_cameraFar );
  camera->aspectRatio.setValue( g_cameraAspect );
  camera->heightAngle.setValue( g_cameraFovY * float(M_PI) / 180.0f );
  root->addChild(camera);

  // Some light
  root->addChild(new SoLightModel);
  root->addChild(new SoDirectionalLight);

  // A red cube translated to the left and down
  SoTransform *myTrans = new SoTransform;
  myTrans->translation.setValue(-2.0f, -2.0f, 0.0f);
  root->addChild(myTrans);

  SoMaterial *myMtl = new SoMaterial;
  myMtl->diffuseColor.setValue(1.0f, 0.0f, 0.0f);
  root->addChild(myMtl);

  root->addChild(new SoCube);

  // A blue sphere translated right
  myTrans = new SoTransform;
  myTrans->translation.setValue(4.0f, 0.0f, 0.0f);
  root->addChild(myTrans);

  myMtl = new SoMaterial;
  myMtl->diffuseColor.setValue(0.0f, 0.0f, 1.0f);
  root->addChild(myMtl);

  root->addChild(new SoSphere);
}

// Build a floor that will be rendered using OpenGL.
void
buildFloor()
{
  int a = 0;

  for (float i = -5.0f; i <= 5.0f; i += 1.25f) {
    for (float j = -5.0f; j <= 5.0f; j += 1.25f, a++) {
      floorObj[a][0] = j;
      floorObj[a][1] = 0.0f;
      floorObj[a][2] = i;
    }
  }
}

#ifdef _WIN32
// Create and initialize OpenGL window.
SoGLContext*
openWindow(HWND &window)
{
  int pixelformat;
  HDC hdc;
  WNDCLASS wndclass;
  PIXELFORMATDESCRIPTOR pfd;
   
  wndclass.style = CS_HREDRAW | CS_VREDRAW;
  wndclass.lpfnWndProc   = ::DefWindowProc;
  wndclass.cbClsExtra    = 0;
  wndclass.cbWndExtra    = 0;
  wndclass.hInstance     = SoWin::getInstance();
  wndclass.hIcon         = NULL;
  wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
  wndclass.hbrBackground = NULL;
  wndclass.lpszMenuName  = NULL;
  wndclass.lpszClassName = "17_3GLFloor";
    
  if (! RegisterClass(&wndclass)) {
    SoWin::createSimpleErrorDialog(NULL, "Error",
                                   "RegisterClass failed", NULL);
    exit(1);
  }

  window = CreateWindow("17_3GLFloor",
                        "17_3GLFloor",
                        WS_OVERLAPPEDWINDOW |
                        WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
                        100, 100,               // X Y position
                        WINWIDTH, WINHEIGHT,    // Width Height
                        NULL,                   // Parent
                        NULL,                   // Menu
                        SoWin::getInstance(),   // App instance handle
                        NULL);                  // Window creation data
  if (! window) {
    SoWin::createSimpleErrorDialog(NULL, "Error",
                                   "CreateWindow failed", NULL);
    exit(2);
  }

  pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
  pfd.nVersion = 1;
  pfd.dwFlags = PFD_DRAW_TO_WINDOW
    | PFD_SUPPORT_OPENGL
    | PFD_DOUBLEBUFFER;
  pfd.iPixelType = PFD_TYPE_RGBA;
  pfd.cColorBits = 24;
  pfd.cRedBits = 0;
  pfd.cRedShift = 0;
  pfd.cGreenBits = 0;
  pfd.cGreenShift = 0;
  pfd.cBlueBits = 0;
  pfd.cBlueShift = 0;
  pfd.cAlphaBits = 0;
  pfd.cAlphaShift = 0;
  pfd.cAccumBits = 0;
  pfd.cAccumRedBits = 0;
  pfd.cAccumGreenBits = 0;
  pfd.cAccumBlueBits = 0;
  pfd.cAccumAlphaBits = 0;
  pfd.cDepthBits = 32;
  pfd.cStencilBits = 0;
  pfd.cAuxBuffers = 0;
  pfd.iLayerType = PFD_MAIN_PLANE;
  pfd.bReserved = 0;
  pfd.dwLayerMask = 0;
  pfd.dwVisibleMask = 0;
  pfd.dwDamageMask = 0;

  // Setup pixel format
  hdc = GetDC(window);
  if ((pixelformat = ChoosePixelFormat(hdc, &pfd)) == 0) {
    SoWin::createSimpleErrorDialog(NULL, "Error",
                                   "ChoosePixelFormat failed", NULL);
    exit(3);
  }
  if (SetPixelFormat(hdc, pixelformat, &pfd) == FALSE) {
    SoWin::createSimpleErrorDialog(NULL, "Error",
                                   "SetPixelFormat failed", NULL);
    exit(4);
  }

  SoGLFormat format;
  format.setDisplay( hdc );
  format.setVisualInfo( &pfd );
  
  SoGLContext* context = new SoGLContext( format );
  if (!context->isValid()) {
    SoWin::createSimpleErrorDialog(NULL, "Error",
                                   "wglCreateContext failed", NULL);
    exit(5);
  }

  ShowWindow(window, SW_SHOW);

  return context;
}
#else
// Callback used by GLX window
static Bool
waitForNotify(Display *, XEvent *e, char *arg)
{
  return (e->type == MapNotify) && 
    (e->xmap.window == (Window)arg);
}

// Create and initialize OpenGL window.
SoGLContext*
openWindow(Display *&display, Window &window)
{
  XVisualInfo *vi;
  Colormap cmap;
  XSetWindowAttributes swa;
  XEvent event;
  static int attributeList[] = {
    GLX_RGBA,
    GLX_RED_SIZE, 1,
    GLX_GREEN_SIZE, 1,
    GLX_BLUE_SIZE, 1,
    GLX_DEPTH_SIZE, 1,
    GLX_DOUBLEBUFFER,        
    None,
  };

  // Open the X display 
  display = XOpenDisplay(0);

  // Initialize the GLX visual and context
  vi = glXChooseVisual(display, 
                       DefaultScreen(display), attributeList);

  // Create the X color map
  cmap = XCreateColormap(display, 
                         RootWindow(display, vi->screen), 
                         vi->visual, AllocNone);

  // Create and map the X window
  swa.colormap = cmap;
  swa.border_pixel = 0;
  swa.event_mask = StructureNotifyMask;
  window = XCreateWindow(display, 
                         RootWindow(display, vi->screen), 100, 100, WINWIDTH,
                         WINHEIGHT, 0, vi->depth, InputOutput, vi->visual, 
                         (CWBorderPixel | CWColormap | CWEventMask), &swa);
  XMapWindow(display, window);
  XIfEvent(display, &event, waitForNotify, (char *)window);

  SoGLFormat format;
  format.setDisplay( display );
  format.setVisualInfo( vi );
  
  SoGLContext* context = new SoGLContext( format, window );

  return context;
}
#endif

// Draw the lines that make up the floor, using OpenGL
void
drawFloor()
{
  int i;

  glBegin(GL_LINES);
  for (i=0; i<4; i++) {
    glVertex3fv(floorObj[i*18]);
    glVertex3fv(floorObj[(i*18)+8]);
    glVertex3fv(floorObj[(i*18)+17]);
    glVertex3fv(floorObj[(i*18)+9]);
  }

  glVertex3fv(floorObj[i*18]);
  glVertex3fv(floorObj[(i*18)+8]);
  glEnd();

  glBegin(GL_LINES);
  for (i=0; i<4; i++) {
    glVertex3fv(floorObj[i*2]);
    glVertex3fv(floorObj[(i*2)+72]);
    glVertex3fv(floorObj[(i*2)+73]);
    glVertex3fv(floorObj[(i*2)+1]);
  }
  glVertex3fv(floorObj[i*2]);
  glVertex3fv(floorObj[(i*2)+72]);
  glEnd();
}

int
main(int, char **)
{
  // Initialize Inventor
  SoDB::init();

  // Build a simple scene graph
  SoSeparator *root = new SoSeparator;
  root->ref();
  buildScene(root);

  // Build the floor geometry
  buildFloor();

  // Create and initialize window
#ifdef _WIN32
  HWND window;
  SoRef<SoGLContext> context = openWindow(window);
#else
  Display *display;
  Window window;
  SoRef<SoGLContext> context = openWindow(display, window);
#endif

  context->bind();

  glViewport( 0, 0, WINWIDTH, WINHEIGHT );

  glEnable(GL_DEPTH_TEST);
  glClearColor(g_backgroundColor[0], g_backgroundColor[0], g_backgroundColor[0], 1.0f);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Set up the camera using OpenGL.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(g_cameraFovY, g_cameraAspect, g_cameraNear, g_cameraFar);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef(0.0f, 0.0f, -g_cameraPosZ);

  // Render the floor using OpenGL
  glPushMatrix();
  glTranslatef(0.0f, -3.0f, 0.0f);
  glColor3f(0.7f, 0.0f, 0.0f);
  glLineWidth(2.0f);
  glDisable(GL_LIGHTING);
  drawFloor();
  glEnable(GL_LIGHTING);
  glPopMatrix();

  SbViewportRegion myViewport(WINWIDTH, WINHEIGHT);

  SoSceneManager* manager = new SoSceneManager;
  //manager->setBackgroundColor(g_backgroundColor);
  manager->setSceneGraph(root);
  manager->setViewportRegion(myViewport);

  // Render the scene
  manager->render();

  context->swapBuffers();

  SbTime::sleep(2000);
   
  //Rerender the floor using OpenGL again:
  glClearColor(0.8f, 0.8f, 0.8f, 1.0f);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
  glPushMatrix();
  glTranslatef(0.0f, -3.0f, 0.0f);
  glColor3f(0.0f, 0.7f, 0.0f);
  glLineWidth(2.0f);
  glDisable(GL_LIGHTING);
  drawFloor();
  glEnable(GL_LIGHTING);
  glPopMatrix();
   
  //Redraw the rest of the scene: 
  manager->render();

  context->swapBuffers();

  SbTime::sleep(2000);

  root->unref();

  context->unbind();

  // Release context
  context = NULL;

  SoDB::finish();
  return 0;
}


