/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
/*--------------------------------------------------------------
 *  This is an example from the Inventor Toolmaker,
 *  chapter 2, example 6.
 *
 *  Source file for "Alternate" group node.
 *------------------------------------------------------------*/

#include <Inventor/misc/SoChildList.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoGetMatrixAction.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/actions/SoPickAction.h>
#include <Inventor/actions/SoSearchAction.h>

// This define causes the declaration of initClass in the node's
// header file to be declared _declspec(dllexport).  This is required
// for the entry point to be visible outside the DLL.
#define BUILDING_DLL

#include "Alternate.h"

SO_NODE_SOURCE(Alternate);

//
// This initializes the Alternate class.
//

void
Alternate::initClass()
{
   // Initialize type id variables
   getClassRenderEngineMode().setRenderMode(SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING);
   SO_NODE_INIT_CLASS(Alternate, SoGroup, "Group"); 
}

void
Alternate::exitClass()
{
   // Initialize type id variables
   SO__NODE_EXIT_CLASS(Alternate);
}


//
// Constructor
//

Alternate::Alternate()
{
   SO_NODE_CONSTRUCTOR(Alternate);
}

//
// Constructor that takes approximate number of children
//

Alternate::Alternate(int numChildren) : SoGroup(numChildren)
{
   SO_NODE_CONSTRUCTOR(Alternate);
}

//
// Destructor
//

Alternate::~Alternate()
{
}

//
// Each of these implements an action by calling the standard
// traversal method. Note that (as in the Glow node source) we prefix
// the call to doAction() with the name of the class, to avoid
// problems with derived classes.
//

void
Alternate::getBoundingBox(SoGetBoundingBoxAction *action)
{
   Alternate::doAction(action);
}

void
Alternate::GLRender(SoGLRenderAction *action)
{
   Alternate::doAction(action);
}

void
Alternate::handleEvent(SoHandleEventAction *action)
{
   Alternate::doAction(action);
}

void
Alternate::pick(SoPickAction *action)
{
   Alternate::doAction(action);
}

//
// This implements the traversal for the SoGetMatrixAction,
// which is handled a little differently: it does not traverse
// below the root node or tail of the path it is applied to.
// Therefore, we need to compute the matrix only if this group
// is in the middle of the current path chain or is off the path
// chain (since the only way this could be true is if the group
// is under a group that affects the chain).
//

void
Alternate::getMatrix(SoGetMatrixAction *action)
{
   int         numIndices;
   const int   *indices;

   // Use SoAction::getPathCode() to determine where this group
   // is in relation to the path being applied to (if any). (see
   // the comments in doAction() for details.)
   switch (action->getPathCode(numIndices, indices)) {

     case SoAction::NO_PATH:
     case SoAction::BELOW_PATH:
      // If there's no path, or we're off the end, do nothing
      break;

     case SoAction::OFF_PATH:
     case SoAction::IN_PATH:
      // If we are in the path chain or we affect nodes in the
      // path chain, traverse the children
      Alternate::doAction(action);
      break;
   }
}

//
// This implements the traversal for the SoSearchAction, which
// is also a little different. The search action is set to
// either traverse all nodes in the graph or just those that
// would be traversed during normal traversal. We need to check
// that flag before traversing our kids.
//

void
Alternate::search(SoSearchAction *action)
{
   // if the action is searching everything, then traverse all
   // of our children as SoGroup would
   if (action->isSearchingAll())
      SoGroup::search(action);

   else {
      // First, make sure this node is found if we are searching
      // for Alternate (or group) nodes
      SoNode::search(action);

      // Traverse the children in our usual way
      Alternate::doAction(action);
   }
}

//
// This implements typical action traversal for an Alternate
// node, skipping every other child.
//

void
Alternate::doAction(SoAction *action)
{
   // SoAction has a method called "getPathCode()" that returns
   // a code indicating how this node is related to the path(s)
   // the action is being applied to. This code is one of the
   // following:
   //
   // NO_PATH    = Not traversing a path (action was applied
   //                to a node) 
   // IN_PATH    = This node is in the path chain, but is not
   //                the tail node
   // BELOW_PATH = This node is the tail of the path chain or
   //                is below the tail
   // OFF_PATH   = This node is off to the left of some node in
   //                the path chain
   //
   // If getPathCode() returns IN_PATH, it returns (in its two
   // arguments) the indices of the next nodes in the paths.
   // (Remember that an action can be applied to a list of
   // paths.)

   // For the IN_PATH case, these will be set by getPathCode()
   // to contain the number of child nodes that are in paths and
   // the indices of those children, respectively. In the other
   // cases, they are not meaningful.
   int         numIndices;
   const int   *indices;

   // This will be set to the index of the last (rightmost)
   // child to traverse
   int         lastChildIndex;

   // If this node is in a path, see which of our children are
   // in paths, and traverse up to and including the rightmost
   // of these nodes (the last one in the "indices" array).
   if (action->getPathCode(numIndices, indices) ==
       SoAction::IN_PATH)
      lastChildIndex = indices[numIndices - 1];

   // Otherwise, consider all of the children
   else
      lastChildIndex = getNumChildren() - 1;

   // Now we are ready to traverse the children, skipping every
   // other one. For the SoGetBoundingBoxAction, however, we
   // have to do some extra work in between each pair of
   // children - we have to make sure the center points get
   // averaged correctly.
   if (action->isOfType(
            SoGetBoundingBoxAction::getClassTypeId())) {
      SoGetBoundingBoxAction *bba =
         (SoGetBoundingBoxAction *) action;
      SbVec3f  totalCenter(0.0, 0.0, 0.0);
      int      numCenters = 0;

      for (int i = 0; i <= lastChildIndex; i += 2) {
         children->traverse(bba, i);

         // If the traversal set a center point in the action,
         // add it to the total and reset for the next child.
         if (bba->isCenterSet()) {
            totalCenter += bba->getCenter();
            numCenters++;
            bba->resetCenter();
         }
      }
      // Now, set the center to be the average. Since the
      // centers were already transformed, there's no need to
      // transform the average.
      if (numCenters != 0)
         bba->setCenter(totalCenter / (float)numCenters, FALSE);
   }

   // For all other actions, just traverse every other child
   else
      for (int i = 0; i <= lastChildIndex; i += 2)
         children->traverse(action, i);
}


