/*=================================================================================
 ***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

//
// Source file for "Torus" shape node.
//

#include <Inventor/SoDB.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/elements/SoComplexityElement.h>
#include <Inventor/misc/SoState.h>

#include "Torus.h"

SO_NODE_SOURCE( Torus );

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif
#define TWOPI (2.0 * M_PI)

//
// This initializes the Torus class.
//
void
Torus::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );
  SO__NODE_INIT_CLASS( Torus, "Torus", SoShape );
}

//
// This exits the Torus class.
//
void
Torus::exitClass()
{
  SO__NODE_EXIT_CLASS( Torus );
}

//
// Constructor
//
Torus::Torus()
: m_internalShapeCache( new SoNodeDependencies )
{
  SO_NODE_CONSTRUCTOR( Torus );
  SO_NODE_ADD_FIELD( radius,      (2.0f) );
  SO_NODE_ADD_FIELD( minorRadius, (1.0f) );

  // Add dependency to the torus fields. This allows the cache handler
  // to automatically invalidate itself when the fields are changed.
  m_internalShapeCache->addDependency( radius );
  m_internalShapeCache->addDependency( minorRadius );

  // This is where we add the dependancy to the complexity element.
  // This allows the cache handler to automatically invalidate
  // itself when the complexity element is changed.
  m_internalShapeCache->addDependency<SoComplexityElement>();
}

//
// Destructor
//
Torus::~Torus()
{
}

// Build the internal shape depending on the elements from the current state
void
Torus::updateInternalShape( SoState* state )
{
  // Use this structure to hold info about how to draw the torus
  TorusInfo info;

  // Figure out how many polygons to use
  computeComplexityInfo( state, info );

  // Each triangle strip goes around the top view
  // Number of vertices per strip = twice the number of (major subdivisions + 1)
  // because we want to loop back to the beginning
  const int stripSize = 2 * (info.numt + 1);

  // Number of triangle strips = number of minor subdivisions
  const int numStrips = info.numc;

  // Set the numVertices field of the TriangleStripSet accordingly
  m_internalShape->numVertices.setNum( numStrips );
  int32_t* numVertices = m_internalShape->numVertices.startEditing();
  for ( int strip = 0; strip < numStrips; strip++ )
    numVertices[strip] = stripSize;
  m_internalShape->numVertices.finishEditing();

  // Set the size of the VertexProperty buffers
  const int numVerticesTotal = stripSize * numStrips;
  m_vertexProperty->vertex.setNum( numVerticesTotal );
  m_vertexProperty->normal.setNum( numVerticesTotal );
  m_vertexProperty->texCoord.setNum( numVerticesTotal );

  SbVec3f* vertices  = m_vertexProperty->vertex.startEditing();
  SbVec3f* normals   = m_vertexProperty->normal.startEditing();
  SbVec2f* texCoords = m_vertexProperty->texCoord.startEditing();

  // Now fill the buffers
  int vertexIndex = 0;
  // go around cross section
  for ( int strip = 0; strip < numStrips; strip++ )
  {
    // go around top view
    for ( int stripVertex = 0; stripVertex <= info.numt; stripVertex++ )
    {
      for ( int offset = 1; offset >= 0; offset-- )
      {
        const int crossSection = strip + offset;

        vertices[vertexIndex]  = getVertex( crossSection, info.numc, stripVertex, info.numt );
        normals[vertexIndex]   = getNormal( vertices[vertexIndex], stripVertex, info.numt );
        texCoords[vertexIndex] = getTexCoord( crossSection, info.numc, stripVertex, info.numt );

        vertexIndex++;
      }
    }
  }

  m_vertexProperty->vertex.finishEditing();
  m_vertexProperty->normal.finishEditing();
  m_vertexProperty->texCoord.finishEditing();
}

// Retrieve internal shape representing the torus
SoNode*
Torus::getInternalShape( SoState* state )
{
  // Initialize internal shape
  if ( m_internalShape.ptr() == NULL )
  {
    m_vertexProperty = new SoVertexProperty;
    m_vertexProperty->normalBinding.setValue( SoVertexProperty::PER_VERTEX );
    m_vertexProperty->materialBinding.setValue( SoVertexProperty::OVERALL );

    m_internalShape = new SoTriangleStripSet;
    m_internalShape->vertexProperty.setValue( m_vertexProperty.ptr() );
  }

  // If the cache handler is invalid, we need an update
  if ( !m_internalShapeCache->isValid(state) )
  {
    updateInternalShape( state );
    m_internalShapeCache->updateCache( state ); // make the cache valid again after the update
  }

  return m_internalShape.ptr();
}

void
Torus::doAction( SoAction* action )
{
  // Access the state from the action
  SoState* state = action->getState();

  // Do a push before modifying elements as we don't want the shape to leak.
  state->push();

  // Retrieve our implementation scenegraph
  SoNode* internalShape = getInternalShape( state );

  // Here comes the central part of the implementation. We delegate the rendering part to our internal sceneGraph.
  // The forwardTraversal() method enables the action to traverse a sub-scenegraph which is not
  // actually part of the whole inventor sceneGraph. Outside of this node, this sub-scenegraph won't
  // be visible. It won't appear in current path and won't be visible to a search action for instance.
  action->forwardTraversal( internalShape );

  // Do a pop after modifying elements as we don't want the shape to leak.
  state->pop();
}

//
// Implements the computeBBox for the Torus node.
//
void
Torus::computeBBox( SoAction* /*action*/, SbBox3f& box, SbVec3f& center )
{
  // Here, we could redirect #action to the doAction() method so that
  // all the work is done by the internal shape, but by
  // knowing the values of the radius and minorRadius, we can compute the
  // bounding box of the torus much more efficiently.

  float radiiSum = radius.getValue() + minorRadius.getValue();
  float minRad = minorRadius.getValue();
  box = SbBox3f( -radiiSum, -radiiSum, -minRad, radiiSum, radiiSum, minRad );
  center = SbVec3f( 0.0f, 0.0f, 0.0f );
}

//
// Implements the SoGLRenderAction for the Torus node.
//
void
Torus::GLRender( SoGLRenderAction* action )
{
  // First see if the object is visible and should be rendered
  // now. This is a method on SoShape that checks for INVISIBLE
  // draw style, BOUNDING_BOX complexity, and delayed
  // transparency.
  if ( !shouldGLRender(action) )
    return;

  // Redirect action to the doAction() method because
  // all the work is done by the internal shape.
  doAction( action );
}

//
// Generates the triangles for the torus.
//
void
Torus::generatePrimitives( SoAction* action )
{
  // Redirect action to the doAction() method because
  // all the work is done by the internal shape.
  doAction( action );
}

// Computes vertex position given the current torus subdivision we are working on.
SbVec3f
Torus::getVertex( int minorSubdiv, int numMinorSubdivs, int subdiv, int numSubdivs )
{
  subdiv %= numSubdivs;
  minorSubdiv %= numMinorSubdivs;

  const double angle = M_PI_2 + TWOPI * static_cast<double>( subdiv ) / static_cast<double>( numSubdivs );
  const double minorAngle = TWOPI * static_cast<double>( minorSubdiv ) / static_cast<double>( numMinorSubdivs );
  const double minorAngleCos = radius.getValue() + minorRadius.getValue() * cos(minorAngle);

  return SbVec3f( static_cast<float>(minorAngleCos * cos(angle)),
                  static_cast<float>(minorAngleCos * sin(angle)),
                  static_cast<float>(minorRadius.getValue() * sin(minorAngle)) );
}

// Computes vertex texture coordinates given the current torus subdivision we are working on.
SbVec2f
Torus::getTexCoord( int minorSubdiv, int numMinorSubdivs, int subdiv, int numSubdivs )
{
  return SbVec2f( static_cast<float>(minorSubdiv) / static_cast<float>(numMinorSubdivs),
                  1.0f - static_cast<float>(subdiv) / static_cast<float>(numSubdivs) );
}

// Computes vertex normal given the current torus subdivision we are working on.
SbVec3f
Torus::getNormal( const SbVec3f& vert, int subdiv, int numSubdivs )
{
  subdiv %= numSubdivs;

  const double angle = M_PI_2 + TWOPI * static_cast<double>( subdiv ) / static_cast<double>( numSubdivs );

  SbVec3f norm( vert[0] - radius.getValue() * static_cast<float>(cos(angle)),
                vert[1] - radius.getValue() * static_cast<float>(sin(angle)),
                vert[2] );
  norm.normalize();

  return norm;
}

//
// Fills in numc and numt in the TorusInfo.
//
void
Torus::computeComplexityInfo( SoState* state, TorusInfo& info )
{
  // Here we retrieve the complexity value from the state and clamp it between 0.0 and 1.0
  const float complexity = SbMathHelper::Clamp( SoComplexityElement::get(state), 0.0f, 1.0f );

  // Based on the complexity, find the number of verts/normals
  // to use around the cross-section (numc) and torus (numt)
  // Bounding box complexity has already been taken care of by the
  // parent class.
  static const float complexityFactor = 70.0f;
  const float numVerts = complexity * complexityFactor;

  float minorMajorRatio = 0.0f;
  if ( radius.getValue() > 0.0f )
    minorMajorRatio = minorRadius.getValue() / radius.getValue();

  info.numt = 3 + static_cast<int>( numVerts );
  info.numc = 3 + static_cast<int>( numVerts * minorMajorRatio );
}
