/*=================================================================================
 ***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

//
// Header file for "Torus" node.
//

#ifndef TORUS_H
#define TORUS_H

#include <Inventor/SbLinear.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoTriangleStripSet.h>
#include <Inventor/caches/SoNodeDependencies.h>

// Shape class representing a torus.
// The #radius field controls the major radius of the torus (from the center).
// The #minorRadius field controls the radius of the cross section of the torus.
// The torus geometry is affected by the value of the SoComplexity Inventor element :
// Higher complexity values will result in a more refined shape with more polygons.
// The default torus has radius 2.0 and minorRadius 1.0.
// The torus lies in the x-y plane and is centered about the origin.
class Torus : public SoShape
{
  SO_NODE_HEADER( Torus );

public:

  // Fields
  SoSFFloat radius;      // Radius of the torus from center
  SoSFFloat minorRadius; // Radius of cross-section

  // Initializes this class
  static void initClass();

  // Exits this class
  static void exitClass();

  // Constructor
  Torus();

protected:
  // These implement supported actions
  virtual void GLRender( SoGLRenderAction* action );
  virtual void generatePrimitives( SoAction* action );
  virtual void computeBBox( SoAction* action, SbBox3f& box, SbVec3f& center );
  virtual void doAction( SoAction* action );

  // Destructor
  virtual ~Torus();

private:

  // Structure used to pass around information about how to draw the torus
  struct TorusInfo
  {
    int numc; // number of vertices around cross-section
    int numt; // number of vertices around torus
  };

  // These methods are used to compute the different vertex properties given
  // the current torus subdivision we are working on during shape construction.
  SbVec3f getVertex( int minorSubdiv, int numMinorSubdivs, int subdiv, int numSubdivs );
  SbVec2f getTexCoord( int minorSubdiv, int numMinorSubdivs, int subdiv, int numSubdivs );
  SbVec3f getNormal( const SbVec3f& vert, int subdiv, int numSubdivs );

  // Uses the complexity element from the state to compute geometrical information
  // needed to build the internal shape.
  void computeComplexityInfo( SoState* state, TorusInfo& info );

  // Retrieve internal shape representing the torus
  SoNode* getInternalShape( SoState* state );

  // Update internal shape geometry depending on the Torus field values.
  void updateInternalShape( SoState* state );

  // vertex property node containing the geometry of the torus
  SoRef<SoVertexProperty> m_vertexProperty;

  // Internal shape representing the torus
  SoRef<SoTriangleStripSet> m_internalShape;

  // This cache handler is used as a flag to check whether or not the internal shape needs to be rebuilt.
  // It is also needed to add a dependency on one or more inventor elements, so that the cache
  // automatically invalidates itself when the element is changed.
  // In our case, we need a dependancy on the Complexity element and on both radii fields.
  SoRef<SoNodeDependencies> m_internalShapeCache;
};

#endif /* TORUS_H */
