/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
 **=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
/*--------------------------------------------------------------
 *  This is an example from the Inventor Toolmaker,
 *  chapter 2, example 3.
 *
 *  Header file for "Pyramid" shape node.
 *------------------------------------------------------------*/

#ifndef PYRAMID_H
#define PYRAMID_H

#include <Inventor/SbLinear.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoIndexedTriangleSet.h>
#include <Inventor/caches/SoNodeDependencies.h>

// SoShape.h includes SoSubNode.h; no need to include it again

// Pyramid texture coordinates are defined on the sides so that
// the seam is along the left rear edge, wrapping
// counterclockwise around the sides. The texture coordinates on
// the base are set up so the texture is right side up when the
// pyramid is tilted back.

#if defined(BUILDING_DLL)
#include <Inventor/sys/port.h>
#if defined(_WIN32)
   // When building the node as a DLL under Win32 we must explicitly
   // declare this entry point as visible outside the DLL.  The macro
   // BUILDING_DLL is defined in the node's source file.
#define CUSTOMAPI VC_DLL_EXPORT
#else
#define CUSTOMAPI GCC_DLLEXPORT
#endif
#else
#define CUSTOMAPI
#endif

class CUSTOMAPI Pyramid : public SoShape
{
  SO_NODE_HEADER( Pyramid );

public:

  enum Part
  {                  // Pyramid parts:
    SIDES = 0x01,    // The 4 side faces
    BASE = 0x02,     // The bottom square face
    ALL = 0x03       // All parts
  };

  // Fields
  SoSFBitMask parts;         // Visible parts
  SoSFFloat   baseWidth;     // Width of base
  SoSFFloat   baseDepth;     // Depth of base
  SoSFFloat   height;        // Height, base to apex

  // Initializes this class
  static void initClass();
  static void exitClass();

  // Constructor
  Pyramid();

  // Turns on/off a part of the pyramid. (Convenience)
  void addPart( Part part );
  void removePart( Part part );

  // Returns whether a given part is on or off. (Convenience)
  SbBool hasPart( Part part ) const;

protected:
  // This implements the GL rendering action. We will inherit
  // all other action behavior, including rayPick(), which is
  // defined by SoShape to pick against all of the triangles
  // created by generatePrimitives.
  virtual void GLRender( SoGLRenderAction* action );

  // Generates triangles representing a pyramid
  virtual void generatePrimitives( SoAction* action );

  // This computes the bounding box and center of a pyramid. It
  // is used by SoShape for the SoGetBoundingBoxAction and also
  // to compute the correct box to render or pick when
  // complexity is BOUNDING_BOX. Note that we do not have to
  // define a getBoundingBox() method, since SoShape already
  // takes care of that (using this method).
  virtual void computeBBox( SoAction* action, SbBox3f& box, SbVec3f& center );

  virtual void doAction( SoAction* action );

  // Destructor
  virtual ~Pyramid();

private:
  // Static data representing the geometry of the pyramid
  // Static members are used in order to minimize memory consumption.

  // Vertex positions
  static std::vector<SbVec3f> s_vertices;

  // Vertex normals
  static std::vector<SbVec3f> s_normals;

  // Texture coordinates
  static std::vector<SbVec2f> s_texCoords;

  // Vertex indices
  static std::vector<int32_t> s_vertexIndices;

  // Normal indices
  static std::vector<int32_t> s_normalIndices;

  // Texture Coordinates indices
  static std::vector<int32_t> s_texCoordIndices;

  // Material indices
  static std::vector<int32_t> s_materialIndices;

  // static vertex property node containing the geometry of the pyramid
  static SoRef<SoVertexProperty> s_vertexProperty;

  // Internal shape representing the pyramid
  SoRef<SoIndexedTriangleSet> m_triangleSet;

  // Retrieve internal shape representing the pyramid
  SoNode* getInternalShape( SoState* state );

  // Update internal shape geometry depending on the value of the #parts field.
  void updateInternalShape();

  // Computes and returns half-width, half-height, and
  // half-depth based on current field values
  void getSize( float& halfWidth, float& halfHeight, float& halfDepth ) const;

  // This cache handler is used as a flag to check whether or not the internal shape needs to be rebuilt.
  // It is needed to add a dependency on the shape's fields, so that the cache automatically invalidates
  // itself when one of the fields is changed.
  // In our case, we need a dependancy on the #parts field.
  SoRef<SoNodeDependencies> m_internalShapeCache;
};

#endif /* PYRAMID_H */
