/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

////////////////////////////////////////////////////////////////////
// 
// 3DTextureViewer Demo - View volume dataset through 3D Textures
//
// This source file is an example application implementation and
// may be used as is, modified and/or incorporated into user
// applications.  In any case this copyright statement should be
// left intact.
//
// Discussion:
// This example loads a volume dataset (basically a sequence of
// image slices). Only a small number of file formats are supported
// (file reading is left as an exercise for the user :-).
//
// What you should see:
// If your graphics board does not support 3D textures, all you will
// see is the geometry the texture would be applied to (a series of
// green translucent rectangles)!  Otherwise...
// Initially you should see the volume displayed as a series of 5 or
// 9 slices through the volume in shades of green.  You can spin this
// scene, but not much else in this example.
// Change to selection mode (press ESC or click the pointer button).
// Now press the 'T' (Toggle) key.
// You should see a single slice through the volume in shades of blue.
// There is a manipulator attached to this slice which allows you to
// move it through the volume along its perpendicular axis or rotate
// it to any orientation.
// In selection mode there are other keys available, see below.
//
// Interactive Use:
// in selection mode, the following action-key apply
//  T       Toggle between display of slice and volume
//  V       Volume display
//  S       Slice display
//  R       Restore original slice plane
//  X       X axis slice
//  Y       Y axis slice
//  Z       Z axis slice
//  C       (un)compensate unregular (non cubic) data
//
// Limitations:
//
// 1) If your graphics hardware does not support 3D textures, you
//    will only see the untextured geometry.
//    It is also possible to simulate volume rendering using 2D
//    textures (see for example the VSG VolumeRendering extension).
//
// 2) Only data with 8 bits per texel is currently handled by this
//    example program.
//
// 3) No color map is applied to the volume data in this example.
//    It creates a two component (Luminance/Alpha) texture from the
//    volume data, assigning the sample value directly to Luminance
//    and assigning zero or one to Alpha depending on whether the
//    sample value is greater than some minimum.
//    In practice you would probably create an RGBA texture.
//
//    The color you see (green in the volume, yellow on the slice)
//    is created by assigning color to the geometry that the texture
//    is applied to (using a standard SoMaterial node).
//
// Original: P.Vigneras Jan 2000
// Modified: mmh July 2001
////////////////////////////////////////////////////////////////////

#define MORE_SLICES 1

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/draggers/SoTransformBoxDragger.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/manips/SoJackManip.h>
#include <Inventor/manips/SoTransformBoxManip.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoIndexedFaceSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTextureCoordinate3.h>
#include <Inventor/nodes/SoTextureCoordinateBinding.h>
#include <Inventor/nodes/SoTexture3.h>
#include <Inventor/nodes/SoTexture3Transform.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/SbBox.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoDB.h>

#include <Inventor/helpers/SbFileHelper.h>

#ifdef _WIN32
#  include <crtdbg.h>
#endif

#define DEFAULT_FILE_NAME "Syn_64.vol"

//----- Declare external file readers
extern int ReadVolFile( const SbString&, int &, int &, int &, unsigned char *& );

//----- Declare known file types
enum fileType_t { 
  UNKNOWN, 
  VOL_FILE 
};

typedef struct {
  const char *ext;
  int type;
} extTableEntry;

static extTableEntry extTable[] = {  
  { "vol", VOL_FILE }
};

static int numExt = sizeof(extTable) / sizeof(extTableEntry);

//----- Declare internal data structures
typedef struct {
  unsigned char *data;
  int nx;
  int ny;
  int nz;
  char *name;
} volume_t;

//----- Utility functions
static void 
Output(const char *string)
{
  printf("%s", string);
#ifdef _WIN32
  OutputDebugString(string);
#endif
}

int 
extractFilename(const char *pathname, const char *&fileName, const char *&fileExt)
{
  // Extract filename and extension from (possible) pathname
  char *p1 = (char*)strrchr(pathname, '/');    // Find last forward slash (UNIX)
  char *p2 = (char*)strrchr(pathname, '\\');   // Find last backslash (Windows)
  p2 = (p2 > p1) ? p2 : p1;             // Whichever is later in string
  fileName = (p2 > pathname) ? (p2+1) : pathname;
  
  char *pDot = (char*)strrchr(fileName, '.');
  if (pDot == NULL) {
    char buffer[256];
    sprintf(buffer, "Data file '%s' has no file extension.\n", fileName);
    exit(0);
  }
  
  fileExt = pDot + 1;
  int fileType = UNKNOWN;
  for(int i = 0; i < numExt; i++) {
    if (!strcmp(fileExt, extTable[i].ext)) {
      fileType = extTable[i].type;
      break;
    }
  }
  return fileType;
}

//---------------------------------------------------------------------------

static float XMIN = -1;
static float XMAX = 1;

static float YMIN = -1;
static float YMAX = 1;

static float ZMIN = -1;
static float ZMAX = 1;
#ifdef MORE_SLICES
static float EZ = (ZMAX-ZMIN)/8.0f; // eighth of distance [ZMIN;ZMAX]
#endif // MORE_SLICES
static float QZ = (ZMAX-ZMIN)/4.0f; // quarter of distance [ZMIN;ZMAX]
static float HZ_ = (ZMAX-ZMIN)/2.0f; // half of distance [ZMIN;ZMAX]

static float TXMIN = 0;
static float TXMAX = 1;

static float TYMIN = 0;
static float TYMAX = 1;

static float TZMIN = 0;
static float TZMAX = 1;
#ifdef MORE_SLICES
static float ETZ = (TZMAX-TZMIN)/8.0f; // eighth of distance [TZMIN;TZMAX]
#endif // MORE_SLICES
static float QTZ = (TZMAX-TZMIN)/4.0f; // quarter of distance [TZMIN;TZMAX]
static float HTZ = (TZMAX-TZMIN)/2.0f; // half of distance [TZMIN;TZMAX]

//---------------------------------------------------------------------------

//#define MORE_SLICES

#ifdef MORE_SLICES
static const int NUM = 9;
#else
static const int NUM = 5;
#endif

static float xratio = 1; // ratio between longest side and x
static float yratio = 1; // ratio between longest side and y
static float zratio = 1; // ratio between longest side and z
static int useRatio = 1;  // say whether to compensate unregular (non cubic) data

static SbVec3f texVals[] = {
  SbVec3f(TXMIN, TYMIN, TZMIN), 
  SbVec3f(TXMAX, TYMIN, TZMIN), 
  SbVec3f(TXMAX, TYMAX, TZMIN), 
  SbVec3f(TXMIN, TYMAX, TZMIN), 

#ifdef MORE_SLICES
  SbVec3f(TXMIN, TYMIN, TZMIN+ETZ), 
  SbVec3f(TXMAX, TYMIN, TZMIN+ETZ), 
  SbVec3f(TXMAX, TYMAX, TZMIN+ETZ), 
  SbVec3f(TXMIN, TYMAX, TZMIN+ETZ), 
#endif

  SbVec3f(TXMIN, TYMIN, TZMIN+QTZ), 
  SbVec3f(TXMAX, TYMIN, TZMIN+QTZ), 
  SbVec3f(TXMAX, TYMAX, TZMIN+QTZ), 
  SbVec3f(TXMIN, TYMAX, TZMIN+QTZ), 

#ifdef MORE_SLICES
  SbVec3f(TXMIN, TYMIN, TZMIN+QTZ+ETZ), 
  SbVec3f(TXMAX, TYMIN, TZMIN+QTZ+ETZ), 
  SbVec3f(TXMAX, TYMAX, TZMIN+QTZ+ETZ), 
  SbVec3f(TXMIN, TYMAX, TZMIN+QTZ+ETZ), 
#endif

  SbVec3f(TXMIN, TYMIN, TZMIN+HTZ), 
  SbVec3f(TXMAX, TYMIN, TZMIN+HTZ), 
  SbVec3f(TXMAX, TYMAX, TZMIN+HTZ), 
  SbVec3f(TXMIN, TYMAX, TZMIN+HTZ), 

#ifdef MORE_SLICES
  SbVec3f(TXMIN, TYMIN, TZMIN+HTZ+ETZ), 
  SbVec3f(TXMAX, TYMIN, TZMIN+HTZ+ETZ), 
  SbVec3f(TXMAX, TYMAX, TZMIN+HTZ+ETZ), 
  SbVec3f(TXMIN, TYMAX, TZMIN+HTZ+ETZ), 
#endif

  SbVec3f(TXMIN, TYMIN, TZMAX-QTZ), 
  SbVec3f(TXMAX, TYMIN, TZMAX-QTZ), 
  SbVec3f(TXMAX, TYMAX, TZMAX-QTZ), 
  SbVec3f(TXMIN, TYMAX, TZMAX-QTZ), 

#ifdef MORE_SLICES
  SbVec3f(TXMIN, TYMIN, TZMAX-ETZ), 
  SbVec3f(TXMAX, TYMIN, TZMAX-ETZ), 
  SbVec3f(TXMAX, TYMAX, TZMAX-ETZ), 
  SbVec3f(TXMIN, TYMAX, TZMAX-ETZ), 
#endif

  SbVec3f(TXMIN, TYMIN, TZMAX), 
  SbVec3f(TXMAX, TYMIN, TZMAX), 
  SbVec3f(TXMAX, TYMAX, TZMAX), 
  SbVec3f(TXMIN, TYMAX, TZMAX) };



static SbVec3f coordsVals[] = {
  SbVec3f(XMIN, YMIN, ZMAX), 
  SbVec3f(XMAX, YMIN, ZMAX), 
  SbVec3f(XMAX, YMAX, ZMAX), 
  SbVec3f(XMIN, YMAX, ZMAX), 

#ifdef MORE_SLICES
  SbVec3f(XMIN, YMIN, ZMAX-EZ), 
  SbVec3f(XMAX, YMIN, ZMAX-EZ), 
  SbVec3f(XMAX, YMAX, ZMAX-EZ), 
  SbVec3f(XMIN, YMAX, ZMAX-EZ), 
#endif

  SbVec3f(XMIN, YMIN, ZMAX-QZ), 
  SbVec3f(XMAX, YMIN, ZMAX-QZ), 
  SbVec3f(XMAX, YMAX, ZMAX-QZ), 
  SbVec3f(XMIN, YMAX, ZMAX-QZ), 

#ifdef MORE_SLICES
  SbVec3f(XMIN, YMIN, ZMIN+HZ_+EZ), 
  SbVec3f(XMAX, YMIN, ZMIN+HZ_+EZ), 
  SbVec3f(XMAX, YMAX, ZMIN+HZ_+EZ), 
  SbVec3f(XMIN, YMAX, ZMIN+HZ_+EZ), 
#endif

  SbVec3f(XMIN, YMIN, ZMIN+HZ_), 
  SbVec3f(XMAX, YMIN, ZMIN+HZ_), 
  SbVec3f(XMAX, YMAX, ZMIN+HZ_), 
  SbVec3f(XMIN, YMAX, ZMIN+HZ_), 

#ifdef MORE_SLICES
  SbVec3f(XMIN, YMIN, ZMIN+QZ+EZ), 
  SbVec3f(XMAX, YMIN, ZMIN+QZ+EZ), 
  SbVec3f(XMAX, YMAX, ZMIN+QZ+EZ), 
  SbVec3f(XMIN, YMAX, ZMIN+QZ+EZ), 
#endif

  SbVec3f(XMIN, YMIN, ZMIN+QZ), 
  SbVec3f(XMAX, YMIN, ZMIN+QZ), 
  SbVec3f(XMAX, YMAX, ZMIN+QZ), 
  SbVec3f(XMIN, YMAX, ZMIN+QZ), 

#ifdef MORE_SLICES
  SbVec3f(XMIN, YMIN, ZMIN+EZ), 
  SbVec3f(XMAX, YMIN, ZMIN+EZ), 
  SbVec3f(XMAX, YMAX, ZMIN+EZ), 
  SbVec3f(XMIN, YMAX, ZMIN+EZ), 
#endif

  SbVec3f(XMIN, YMIN, ZMIN), 
  SbVec3f(XMAX, YMIN, ZMIN), 
  SbVec3f(XMAX, YMAX, ZMIN), 
  SbVec3f(XMIN, YMAX, ZMIN) };

//---------------------------------------------------------------------------

#define JACK_MANIP
#ifdef JACK_MANIP
#  define JACK_DRAGGER_TYPE SoJackDragger
#  define JACK_MANIP_TYPE SoJackManip
#else
#  define JACK_DRAGGER_TYPE SoTransformBoxDragger
#  define JACK_MANIP_TYPE SoTransformBoxManip
#endif

// data structure (passed to callback functions)
struct ClipStructCB {
  SoSeparator *root;
  JACK_DRAGGER_TYPE *dragger;
  SoIndexedFaceSet *crossSection;
  SoTexture3 *texture3;
  SoTextureCoordinate3 *csTexCoords;
  SoCoordinate3 *csCoords; // cross section coords
  SoCoordinate3 *coords3;  // vertical slices coords
  SbBox3f *cube;
  SoSeparator *cubeLimits;
  SoComplexity *complexity;
  SbBool revertCPlane;
  SoSwitch *crossSectionSwitch;
} csCB;

SbVec3f *findIntersection(SbBox3f *cube, SbPlane *plane, int &numPoints);
void orderIntersection(SbVec3f *points, int size);
void draggerCB(struct ClipStructCB *csCB, JACK_DRAGGER_TYPE *dragger);
void eventCB(struct ClipStructCB *csCB, SoEventCallback *node);

//---------------------------------------------------------------------------

static void usage() {
  Output("usage: 3DTextureViewer datafile\n");
  Output("commands: in selection mode, the following commands apply\n");
  Output("\tT       Toggle between display of slice and volume\n");
  Output("\tV       Volume display\n");
  Output("\tS       Slice display\n");
  Output("\tR       Restore original slice plane\n");
  Output("\tX       X axis slice\n");
  Output("\tY       Y axis slice\n");
  Output("\tZ       Z axis slice\n");
  Output("\tC       (un)compensate unregular (non cubic) data\n");
}

//---------------------------------------------------------------------------

int
main(int argc, char **argv)
{
  Widget mainWindow = SoXt::init("3D Texture Viewer");
  if (mainWindow == NULL) 
    exit(1);

  char buffer[256];
  SbString pathname;
  if (argc > 1) {
    char *cp = argv[1];
    if (*cp == '-' || *cp == '/') {
      cp++;
      if (*cp == 'h' || *cp == 'H' || *cp == '?') {
        usage();
        exit(0);
      }
    }
    pathname = argv[1] ;
  }
  else {
    pathname = "$OIVHOME/examples/data/Inventor/3DTexture/";
    pathname += DEFAULT_FILE_NAME;
  }
  FILE *fp = SbFileHelper::open (pathname, "r");
  if (fp == NULL)
  {
    pathname = DEFAULT_FILE_NAME;   // try local directory
	fp = SbFileHelper::open (pathname, "r");
  }

  if (fp == NULL) {
    sprintf(buffer, "Data file '%s' not found.\n", pathname.toLatin1());
    Output(buffer);
    exit(0);
  }
  fclose( fp );

  // Extract filename and extension from (possible) pathname
  // Also get file type from extension if possible
  const char *fileName;
  const char *fileExt;
  int fileType = extractFilename(pathname.toLatin1(), fileName, fileExt);

  if (fileType == UNKNOWN) {
    sprintf(buffer, "Data file '%s' has unknown file extension.\n", fileName);
    Output(buffer);
    exit(0);
  }

  sprintf(buffer, "Loading volume data file '%s'\n", pathname.getString());
  Output(buffer);

  usage();

  // Load the volume data
  int ok = 0, nx = -1, ny = -1, nz = -1;
  unsigned char *bytes = NULL;
  if (fileType == VOL_FILE) {
    ok = ReadVolFile(pathname, nx, ny, nz, bytes);
  }
  if (!ok || nx <= 0 || ny <= 0 || nz <= 0) {
    sprintf(buffer, "Read failed for data file '%s'.\n", fileName);
    Output(buffer);
    exit(-1);
  }

  int numValues = nx * ny * nz;
  sprintf(buffer, "Used %d bytes to load volume data\n", numValues);
  Output(buffer);

  {
    // slices are defined in a cube with all dimensions being equal.
    // redefine this cube if the volume of data has different dimensions
    int longest = nx > ny ? (nx > nz ? nx : nz) : (ny > nz ? ny : nz);
    xratio = nx / (float) longest;
    yratio = ny / (float) longest;
    zratio = nz / (float) longest;
    if (useRatio) {
      for(int i = 0; i < 4*NUM; i++) {
        SbVec3f *v = &coordsVals[i];
        v->setValue((*v)[0]*xratio, (*v)[1]*yratio, (*v)[2]*zratio);
      }
    }
  }

  // Root of scenegraph
  SoSeparator *root = new SoSeparator;
  root->ref();

  SoTexture3Transform *tex3Trans = NULL;

  // if one (or several) dimension is not a power of 2, adapt the
  // texture coordinates (the texture3 node will enlarge the
  // 3d texture without making any scale) to fill the gap.
  float xTexRatio, yTexRatio, zTexRatio;
  int xdim = 1, ydim = 1, zdim = 1;
  while(xdim < nx) xdim <<= 1;
  xTexRatio = nx / (float) xdim;
  while(ydim < ny) ydim <<= 1;
  yTexRatio = ny / (float) ydim;
  while(zdim < nz) zdim <<= 1;
  zTexRatio = nz / (float) zdim;
  if (xTexRatio != 1 || yTexRatio != 1 || zTexRatio != 1) {
    tex3Trans = new SoTexture3Transform;
    root->addChild(tex3Trans);
    tex3Trans->scaleFactor.setValue
      (SbVec3f(xTexRatio, yTexRatio, zTexRatio));
  }

  // Callback for event handling
  SoEventCallback *eCallback = new SoEventCallback;
  eCallback->addEventCallback(SoKeyboardEvent::getClassTypeId(), 
                              (SoEventCallbackCB *) eventCB, &csCB);
  root->addChild(eCallback);

  // Everything except the dragger is *not* pickable
  // (so we can grab the dragger through the other data)
  SoPickStyle *pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;
  root->addChild( pickStyle );

  // Draw a bounding box around the volume
  SoSeparator *pBoxSep = new SoSeparator;
  SoDrawStyle *pBoxStyle = new SoDrawStyle;
  pBoxStyle->style = SoDrawStyle::LINES;
  pBoxSep->addChild( pBoxStyle );
  pBoxSep->addChild( new SoCube );
  root->addChild( pBoxSep );

  // Specify overall texture quality
  SoComplexity *complexity = new SoComplexity;
  root->addChild(complexity);
  csCB.complexity = complexity;
  complexity->textureQuality.setValue(0.6f);

  // Disable lighting
  SoLightModel *lightmodel = new SoLightModel;
  lightmodel->model.setValue(SoLightModel::BASE_COLOR);
  root->addChild(lightmodel);

  // Switch to toggle visibility of cross-section or volume
  SoSwitch *crossSectionSwitch = new SoSwitch;
  crossSectionSwitch->whichChild = 0;
  csCB.crossSectionSwitch = crossSectionSwitch;
  root->addChild( crossSectionSwitch );

  // Separator for volume geometry
  SoSeparator *volumeSep = new SoSeparator;
  crossSectionSwitch->addChild( volumeSep );

  // Separator for cross-section (slice)
  SoSeparator *crossSectionSep = new SoSeparator;
  crossSectionSwitch->addChild(crossSectionSep);

  // Material for cross-section (slice)
  SoMaterial *sliceMat = new SoMaterial;
  sliceMat->diffuseColor.setValue(SbColor(0, 0, 1));
  sliceMat->transparency.setValue(0.0f);
  crossSectionSep->addChild( sliceMat );

  // Manipulator to allow moving the slice
  JACK_MANIP_TYPE *manip = new JACK_MANIP_TYPE;
  JACK_DRAGGER_TYPE *dragger = (JACK_DRAGGER_TYPE *) manip->getDragger();
  // remove the scaler and translator parts of the dragger
  dragger->setPart("scaler", NULL);
#ifdef JACK_MANIP
  dragger->setPart("translator.yzTranslator", NULL);
  dragger->setPart("translator.xzTranslator", NULL);
  dragger->setPart("translator.xyTranslator", NULL);
#endif
  crossSectionSep->addChild( new SoPickStyle ); // enable picking for dragger
  crossSectionSep->addChild(dragger);

  csCB.root = root;
  csCB.dragger = dragger;
  csCB.revertCPlane = FALSE;
  dragger->addMotionCallback((SoDraggerCB *) draggerCB, &csCB);

  // Create texture image from volume data
  int numComps = 2;
  unsigned char *newbytes = bytes;
  if (numComps > 1) {
      // Allocate memory for the texture image
      newbytes = (unsigned char*)malloc(numValues * numComps);
      unsigned char *src = bytes;
      unsigned char *dst = newbytes;
      unsigned char minval = 255;
      unsigned char maxval = 0;
      // Find min and max values in the data
      int ii;
      for (ii = 0; ii < numValues; ii++) {
          unsigned char value = *src++;
          if (value > maxval)
              maxval = value;
          if (value < minval)
              minval = value;
      }
      src = bytes;
      float range = (float)(maxval - minval);
      if (numComps == 2) {
          // Two components = Luminance+Alpha
          // Scale actual range of sample values to full range of
          // possible luminance/alpha values to enhance contrast.
          // "Discard" sample values below 32 (usually noise) by
          // setting their alpha to zero.
          for (int ii = 0; ii < numValues; ii++) {
              unsigned char value = *src++;
              unsigned char texel = (unsigned char)((value - minval)/range * 255);
              *dst++ = texel;
              if (value < 32)
                *dst++ = 0;
              else
                *dst++ = texel;
          }
      }
      if (numComps == 4) {
          // Four components = RGBA (gray scale in this case)
          for (int ii = 0; ii < numValues; ii++) {
              unsigned char value = *src++;
              unsigned char texel = (unsigned char)((value - minval)/range * 255);
              *dst++ = texel;
              *dst++ = texel;
              *dst++ = texel;
              if (value < 32)
                *dst++ = 0;
              else
                *dst++ = texel;
          }
      }
  }

  // Create texture3 node
  // Add it to both root (volume) and cross-section separators
  SoTexture3 *texture3 = new SoTexture3;
  texture3->images.setValue(SbVec3s(nx, ny, nz), numComps, newbytes, SoSFImage3::NO_COPY_AND_FREE);
  volumeSep->addChild(texture3);
  crossSectionSep->addChild(texture3);
  csCB.texture3 = texture3;

  // Create geometry for cross-section (slice)
  SoTextureCoordinate3 *csTexCoords = new SoTextureCoordinate3;
  SoCoordinate3 *csCoords = new SoCoordinate3;
  SoIndexedFaceSet *crossSection = new SoIndexedFaceSet;

  csCB.crossSection = crossSection;
  csCB.csTexCoords = csTexCoords;
  csCB.csCoords = csCoords;

  if (tex3Trans) {
    // work around (problem in cumulated texture transformations)
    SoTexture3Transform *tex3Trans2 = new SoTexture3Transform;
    tex3Trans2->scaleFactor.setValue(SbVec3f(1.0001f, 1.0001f, 1.0001f));
    crossSectionSep->addChild(tex3Trans2);
  }
  crossSectionSep->addChild(csTexCoords);
  crossSectionSep->addChild(csCoords);
  crossSectionSep->addChild(crossSection);

  // Create NUM indexed facesets to draw the volume
  // (2 boundary-slices and 3 inner slices)
  SoSeparator *limits = new SoSeparator;
  volumeSep->addChild(limits);
  csCB.cubeLimits = limits;

  // Material for volume geometry
  SoMaterial *greenMat = new SoMaterial;
  greenMat->diffuseColor.setValue(SbColor(0, 1, 0));
  greenMat->transparency.setValue(0.3f);
  limits->addChild(greenMat);

  // Volume geometry
  SoIndexedFaceSet *iFaceSets[NUM];
  SoTextureCoordinate3 *texCoords3 = new SoTextureCoordinate3;
  texCoords3->point.setValues(0, 4*NUM, texVals);
  limits->addChild(texCoords3);

  SoCoordinate3 *coords3 = new SoCoordinate3;
  coords3->point.setValues(0, 4*NUM, coordsVals);
  limits->addChild(coords3);
  csCB.coords3 = coords3;

  for(int i = 0; i < NUM; i++) {
    int index = 4 * i;
    iFaceSets[i] = new SoIndexedFaceSet;
    iFaceSets[i]->coordIndex.set1Value(3, index+3);
    iFaceSets[i]->coordIndex.set1Value(2, index+2);
    iFaceSets[i]->coordIndex.set1Value(1, index+1);
    iFaceSets[i]->coordIndex.set1Value(0, index);
    iFaceSets[i]->textureCoordIndex.set1Value(3, index+3);
    iFaceSets[i]->textureCoordIndex.set1Value(2, index+2);
    iFaceSets[i]->textureCoordIndex.set1Value(1, index+1);
    iFaceSets[i]->textureCoordIndex.set1Value(0, index);
    limits->addChild(iFaceSets[i]);
  }

  limits->ref();
  SoGetBoundingBoxAction *gbba = new SoGetBoundingBoxAction(SbViewportRegion());
  gbba->setInCameraSpace(FALSE);
  gbba->apply(limits);
  csCB.cube = new SbBox3f();
  csCB.cube->extendBy(gbba->getBoundingBox());
  delete gbba;
  limits->unrefNoDelete();
  draggerCB(&csCB, dragger);

  // Create and initialize viewer
  SoXtExaminerViewer *vwr = new SoXtExaminerViewer(mainWindow, "3D Texture Viewer");
  vwr->setSize(SbVec2s(400, 400));
  vwr->setSceneGraph(root);
  vwr->setBackgroundColor(SbColor(.7f, .7f, .7f));
  vwr->setTitle("3D Texture Viewer");
  vwr->setTransparencyType(SoGLRenderAction::SORTED_OBJECT);
  vwr->show();

  SoXt::show(mainWindow);
  SoXt::mainLoop();

  root->unref();
  delete vwr;
  SoXt::finish();

  return 0;
}

/*---------------------------------------------------------------------------*/

void 
draggerCB(struct ClipStructCB *csCB, JACK_DRAGGER_TYPE *dragger)
{
  csCB->root->enableNotify(FALSE);

  SbMatrix transform;
  SbVec3f trans = dragger->translation.getValue();
  SbRotation rot = dragger->rotation.getValue();
  SbVec3f scale = dragger->scaleFactor.getValue();

  transform.setTransform(trans, rot, scale);
  SbVec3f normal(0, 1, 0);
  if (csCB->revertCPlane)
    normal[1] = -normal[1];
  SbPlane plane(normal, SbVec3f(0, 0, 0));
  plane.transform(transform);

  int numPoints = 0;
  SbVec3f *points = findIntersection(csCB->cube, &plane, numPoints);
  orderIntersection(points, numPoints);

  csCB->crossSection->coordIndex.deleteValues(0);
  csCB->crossSection->textureCoordIndex.deleteValues(0);

  for(int i = 0; i < numPoints; i++) {
    SbVec3f coord = points[i];
    csCB->csCoords->point.set1Value(i, coord);
           
    float xpos, ypos, zpos;

    if (useRatio) {
      xpos = ((coord[0]/xratio)-XMIN) / (XMAX - XMIN);
      ypos = ((coord[1]/yratio)-YMIN) / (YMAX - YMIN);
      zpos = ((coord[2]/zratio)-ZMIN) / (ZMAX - ZMIN);
    }
    else {
      xpos = (coord[0]-XMIN) / (XMAX - XMIN);
      ypos = (coord[1]-YMIN) / (YMAX - YMIN);
      zpos = (coord[2]-ZMIN) / (ZMAX - ZMIN);
    }
           
    SbVec3f tcoord(TXMIN+xpos*(TXMAX-TXMIN), 
                   TYMIN+ypos*(TYMAX-TYMIN), 
                   TZMAX-zpos*(TZMAX-TZMIN));
    csCB->csTexCoords->point.set1Value(i, tcoord);
    csCB->crossSection->coordIndex.set1Value(i, i);
    csCB->crossSection->textureCoordIndex.set1Value(i, i);
  }
  csCB->root->enableNotify(TRUE);
  csCB->root->touch();
}

void 
eventCB(struct ClipStructCB *csCB, SoEventCallback *node)
{
  if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::T)) {
    int whichChild = csCB->crossSectionSwitch->whichChild.getValue();
    whichChild = 1 - whichChild;
    csCB->crossSectionSwitch->whichChild = whichChild;
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::S)) {
    csCB->crossSectionSwitch->whichChild = 1;
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::V)) {
    csCB->crossSectionSwitch->whichChild = 0;
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::X)) {
    // X plane
    csCB->dragger->translation.setValue(SbVec3f(0, 0, 0));
    csCB->dragger->rotation.setValue(SbVec3f(0, 0, 1), 1.5707963f); // PI/2
    draggerCB(csCB, csCB->dragger);
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::Y)) {
    // Y plane
    csCB->dragger->translation.setValue(SbVec3f(0, 0, 0));
    csCB->dragger->rotation.setValue(SbVec3f(1, 0, 0), 0);
    draggerCB(csCB, csCB->dragger);
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::Z)) {
    // Z plane
    csCB->dragger->translation.setValue(SbVec3f(0, 0, 0));
    csCB->dragger->rotation.setValue(SbVec3f(1, 0, 0), 1.5707963f); // PI/2
    draggerCB(csCB, csCB->dragger);
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::R)) {
    // Restore original cross-section
    csCB->dragger->translation.setValue(SbVec3f(0, 0, 0));
    csCB->dragger->rotation.setValue(SbVec3f(1, 0, 0), 0);
    csCB->revertCPlane = FALSE;
    draggerCB(csCB, csCB->dragger);
  }
  else if (SoKeyboardEvent::isKeyPressEvent(node->getEvent(), 
                                            SoKeyboardEvent::C)) {
    float ratioX, ratioY, ratioZ;
    if (useRatio) {
      useRatio = 0;
      ratioX = 1/xratio;
      ratioY = 1/yratio;
      ratioZ = 1/zratio;
    }
    else {
      useRatio = 1;
      ratioX = xratio;
      ratioY = yratio;
      ratioZ = zratio;
    }
    for(int i = 0; i < 4*NUM; i++) {
      SbVec3f *v = &coordsVals[i];
      v->setValue((*v)[0]*ratioX, (*v)[1]*ratioY, (*v)[2]*ratioZ);
    }
    csCB->coords3->point.setValues(0, 4*NUM, coordsVals);
    csCB->cubeLimits->ref();

    SoGetBoundingBoxAction *gbba = new SoGetBoundingBoxAction(SbViewportRegion());
    gbba->setInCameraSpace(FALSE);
    gbba->apply(csCB->cubeLimits);
    csCB->cube->makeEmpty();
    csCB->cube->extendBy(gbba->getBoundingBox());
    csCB->cubeLimits->unrefNoDelete();
    draggerCB(csCB, csCB->dragger);
    delete gbba;
  }
}

//---------------------------------------------------------------------------

// returns a list of points that represent the intersection
// between the cube and the plane. Note that this list of
// points is not ordered.
SbVec3f *
findIntersection(SbBox3f *cube, SbPlane *plane, int &numPoints)
{
  // for each edge of the cube, check if there is a point
  // in this edge that belongs to the plane :
  // if [AB] is the edge, let P, a point of [AB], be the
  // intersection with the plane.
  // from a vector standpoint, we can say that
  // P = A + t * AB
  // if the plane is defined by the point N and the normal 
  // vector n, then
  // NP * n = 0
  // we can then prove that if n is not normal to AB, ie
  // n * AB != 0 then t = -(n * NA) / (n * AB)
  // then if 0 <= t <= 1, P is inside the segment [AB] and
  // is a valid intersection
  // If n * AB = 0, there are two cases :
  // . n * NA = 0 => the segment [AB] is included in the plane
  // . n * NA != 0 => there is no intersection between the
  //  plane and the line (AB)

  // 6 is the max number of intersecting points
  static SbVec3f *points = new SbVec3f[6];
  static SbVec3f edges[2][12];

  SbVec3f min = cube->getMin();
  SbVec3f max = cube->getMax();
  SbVec3f normal = plane->getNormal();
  SbVec3f npoint = normal *plane->getDistanceFromOrigin();

  edges[0][0].setValue(min[0], min[1], min[2]);
  edges[1][0].setValue(min[0], min[1], max[2]);
  edges[0][1].setValue(min[0], min[1], min[2]);
  edges[1][1].setValue(min[0], max[1], min[2]);
  edges[0][2].setValue(min[0], min[1], min[2]);
  edges[1][2].setValue(max[0], min[1], min[2]);
  edges[0][3].setValue(max[0], min[1], min[2]);
  edges[1][3].setValue(max[0], min[1], max[2]);
  edges[0][4].setValue(max[0], min[1], min[2]);
  edges[1][4].setValue(max[0], max[1], min[2]);
  edges[0][5].setValue(min[0], min[1], max[2]);
  edges[1][5].setValue(max[0], min[1], max[2]);
  edges[0][6].setValue(min[0], min[1], max[2]);
  edges[1][6].setValue(min[0], max[1], max[2]);
  edges[0][7].setValue(max[0], min[1], max[2]);
  edges[1][7].setValue(max[0], max[1], max[2]);
  edges[0][8].setValue(max[0], max[1], min[2]);
  edges[1][8].setValue(max[0], max[1], max[2]);
  edges[0][9].setValue(min[0], max[1], max[2]);
  edges[1][9].setValue(max[0], max[1], max[2]);
  edges[0][10].setValue(min[0], max[1], min[2]);
  edges[1][10].setValue(max[0], max[1], min[2]);
  edges[0][11].setValue(min[0], max[1], min[2]);
  edges[1][11].setValue(min[0], max[1], max[2]);

  // index in the table of intersecting points
  int index = 0;

  for(int i = 0; i < 12; i++) {
    float Q = normal.dot(edges[1][i]-edges[0][i]);
    float R = normal.dot(edges[0][i]-npoint);
    if (Q == 0) {
      if (R == 0) {
        // this edge is included in the plane
        SbBool foundFirst = FALSE, foundSecond = FALSE;
        for(int j = 0; j < index; j++) {
          if (points[j] == edges[0][i]) foundFirst = TRUE;
          else if (points[j] == edges[1][i]) foundSecond = TRUE;
        }
        if (!foundFirst) points[index++] = edges[0][i];
        if (!foundSecond) points[index++] = edges[1][i];
      }
      else continue;
    }
    else {
      float t = - R / Q;
      if (t >= 0 && t <= 1) {
        // the intersection is inside the edge
        SbVec3f newPoint = edges[0][i] + t * (edges[1][i]-edges[0][i]);
        SbBool found = FALSE;
        for(int j = 0; j < index; j++) {
          if (points[j] == newPoint) found = TRUE;
        }
        if (!found) points[index++] = newPoint;
      }
      else continue;
    }
  }
  numPoints = index;
  return points;
}

// this function takes a list of points that are the intersecting points
// between a plane and a cube, in an undefined order and put them
// in an order that allows to construct a triangulation of them.
// The list passed as an argument is modified to reflect the new order.
void 
orderIntersection(SbVec3f *points, int size)
{
  SbVec3f avg(0, 0, 0);
  int i;

  // find out what is the average point and order the points
  // according to an increasing angle between the average point and
  // one fixed point in one hand, and each of the other points.
  if (size <= 3 || size > 6) 
    return;

  for(i = 0; i < size; i++)
    avg.setValue(points[i][0]+avg[0], points[i][1]+avg[1], points[i][2]+avg[2]);
  avg.setValue(avg[0]/size, avg[1]/size, avg[2]/size);
  SbRotation rot;
  int num = 0;
  SbVec3f vec1 = points[0] - avg;
  SbVec3f vec2 = points[1] - avg;
  // take first cross vector as reference
  SbVec3f cross = vec1.cross(vec2);
  float lvec1 = vec1.length();
  float lvec2 = 0;

  // we don't actually need the angles: instead, the cosine
  // value will do the job. The cross vectors will allow us
  // to know if the angle is greater than PI, in which case
  // the cosine value is substracted from -2 (note that 2 is
  // the period of the cosine function [-1, 1]). We then avoid
  // arc functions, which are pretty expensive.
  static float angles[6]; // 6: max number of points in the intersection
  angles[0] = 1;
  for(i = 1; i < size; i++) {
    vec2 = points[i] - avg;
    lvec2 = vec2.length();
    angles[i] = vec1.dot(vec2) / (lvec1 * lvec2);
    if (cross.dot(vec1.cross(vec2)) < 0)
      angles[i] = -2 - angles[i];
  }

  // sort the points according to their cosine values. Since there's
  // a few points, the bubble-sort algo is ok.
  num = size-1;
  SbVec3f dummy;
  for(i = 0; i < size; i++) {
    for(int j = 0; j < num; j++) {
      if (angles[j] > angles[j+1]) {
        float tmp = angles[j+1];
        angles[j+1] = angles[j];
        angles[j] = tmp;
        dummy = points[j+1];
        points[j+1] = points[j];
        points[j] = dummy;
      }
    }
    num--;
  }
}


