/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : MMH (Jul 1999)
**=======================================================================*/

/* Slightly modified for use in 3DMS demos.
 */

/*+-------------------------------------------------------------------
  |
  |  File:      VOL_IO.C
  |  Voxelator example program.
  |  Copyright (C) 1996 Hewlett-Packard Co.
  |
  |  This source file is an example application implementation and
  |  may be used as is, modified and/or incorporated into user
  |  applications.  In any case this copyright statement should be
  |  left intact.
  |
  |  Description:   Contains procedures to read a volume dataset in
  |                 the Voxelator format.
  |
  +-------------------------------------------------------------------*/

typedef unsigned long              GLenum;
typedef unsigned char              GLboolean;
typedef unsigned long              GLbitfield;
typedef signed char                GLbyte;
typedef unsigned char              GLubyte;
typedef short                      GLshort;
typedef unsigned short             GLushort;
typedef long                       GLint;
typedef unsigned long              GLuint;
typedef float                      GLfloat;
typedef float                      GLclampf;
typedef double                     GLdouble;
typedef double                     GLclampd;
typedef long                       GLsizei;
typedef void                       GLvoid;

#define GL_BYTE                    0x1400
#define GL_UNSIGNED_BYTE           0x1401
#define GL_SHORT                   0x1402
#define GL_UNSIGNED_SHORT          0x1403

#define GL_ALPHA4_EXT              0x803B
#define GL_ALPHA8_EXT              0x803C
#define GL_ALPHA12_EXT             0x803D
#define GL_ALPHA16_EXT             0x803E
#define GL_LUMINANCE4_EXT          0x803F
#define GL_LUMINANCE8_EXT          0x8040
#define GL_LUMINANCE12_EXT         0x8041
#define GL_LUMINANCE16_EXT         0x8042
#define GL_LUMINANCE4_ALPHA4_EXT   0x8043
#define GL_LUMINANCE6_ALPHA2_EXT   0x8044
#define GL_LUMINANCE8_ALPHA8_EXT   0x8045
#define GL_LUMINANCE12_ALPHA4_EXT  0x8046
#define GL_LUMINANCE12_ALPHA12_EXT 0x8047
#define GL_LUMINANCE16_ALPHA16_EXT 0x8048
#define GL_INTENSITY_EXT           0x8049
#define GL_INTENSITY4_EXT          0x804A
#define GL_INTENSITY8_EXT          0x804B
#define GL_INTENSITY12_EXT         0x804C
#define GL_INTENSITY16_EXT         0x804D
#define GL_RGB2_EXT                0x804E
#define GL_RGB4_EXT                0x804F
#define GL_RGB5_EXT                0x8050
#define GL_RGB8_EXT                0x8051
#define GL_RGB10_EXT               0x8052
#define GL_RGB12_EXT               0x8053
#define GL_RGB16_EXT               0x8054
#define GL_RGBA2_EXT               0x8055
#define GL_RGBA4_EXT               0x8056
#define GL_RGB5_A1_EXT             0x8057
#define GL_RGBA8_EXT               0x8058
#define GL_RGB10_A2_EXT            0x8059
#define GL_RGBA12_EXT              0x805A
#define GL_RGBA16_EXT              0x805B

#define GL_INDEX4_INTENSITY12_HP   0x816E
#define GL_INDEX8_INTENSITY8_HP    0x816F
#define GL_INDEX16_INTENSITY16_HP  0x8170

#ifndef TRUE
#  define TRUE 1
#endif
#ifndef FALSE
#  define FALSE 0
#endif
#ifndef TRACE
#  define TRACE printf
#endif

//////////////////////////////////////////////////////////////////////////

#include <fcntl.h>

#if defined(_WIN32)
#  include <io.h>
#  define OPEN_FLAG _O_RDONLY | _O_BINARY
#else
#  include <unistd.h>
#  define _open  open
#  define _close close
#  define _read  read
#  define _lseek lseek
#  define OPEN_FLAG O_RDONLY
#endif

#include <stdlib.h>
#include <stdio.h>
#ifdef __APPLE__
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif
#include "defines.h"

#include <Inventor/SbString.h>
#include <Inventor/helpers/SbFileHelper.h>

/* Forward decls */
int ReadHPVolFile(const SbString&, int &, int &, int &, unsigned char *&);
int ReadSunVolFile(const SbString&, int &, int &, int &, unsigned char *&);

/*+-------------------------------------------------------------------
  |
  |  Global variables
  |
  +-------------------------------------------------------------------*/

/*----- info from data file header... -----*/

int         _MagicNumber;
int         _HeaderLength;

C_IntTuple  _VolumeDimensions;  /* size of volume   */
int         _PixelDepth;
GLenum      _PixelFormat;       /* GL format */
GLenum      _PixelType;         /* GL type */
int         _IndexBits;         /* number of "object" index bits */
                                /* in each pixel                 */

float       _ScaleX;            /* base scale factors */
float       _ScaleY;
float       _ScaleZ;
float       _RotX;              /* base rotation */
float       _RotY;
float       _RotZ;

char        _Manufacturer[C_MANUFACTURER_STRING_LENGTH];
char        _OriginalFileName[C_ORIG_FILENAME_STRING_LENGTH];
int         _VolumeName = 0;

/*+---------------+
  |   SwapBytes   |
  +---------------+*/ 

GLvoid 
SwapBytes(void *ptr,     /* pointer of words requiring byte swapping */
          int sizeBytes) /* number of bytes in array */
{
#if defined(_WIN32) || defined(__linux__)  || defined(__APPLE__)/* all LITTLE ENDIAN systems */
  int sizeInts;
  GLubyte *bytePtr;
  int *intPtr = (int *)ptr;
  
  sizeInts = sizeBytes / 4;
  for (int i = 0; i < sizeInts; i++) {
    bytePtr = (GLubyte *)intPtr;
    *intPtr++ = ((bytePtr[0] << 24) | 
		 (bytePtr[1] << 16) | 
		 (bytePtr[2] << 8)  | 
		 (bytePtr[3] << 0));
  }
#endif
}

/*+----------------+
  |   ReadVolume   |
  +----------------+*/ 

int 
ReadVolFile( const SbString& lpszPathName, int &width, int &height, 
            int &depth, unsigned char *&bytes )
{
  /*----- open the volume file -----*/
  FILE* input_fd;

  if ((input_fd = SbFileHelper::open( lpszPathName, "rb" )) == NULL) {
    TRACE("Can't open volume file: %s\n", lpszPathName.toLatin1());
    return FALSE;
  }
  fread( &_MagicNumber, sizeof(int), 1, input_fd );
  SwapBytes(&_MagicNumber, sizeof(int));
  SbFileHelper::close( input_fd );

  if ((_MagicNumber != C_VR_MAGIC_NUMBER) &&
      (_MagicNumber != C_VR_MAGIC_NUMBER_OLD)) {
    return ReadSunVolFile(lpszPathName, width, height, depth, bytes);
  }
  else { /* HP Voxelator format file */
    return ReadHPVolFile(lpszPathName, width, height, depth, bytes);
  }
}


/*-------------------------------------------------------------------*/
int 
ReadSunVolFile( const SbString& lpszPathName, int &width, int &height, 
               int &depth, unsigned char *&bytes)
{
  width = height = depth = 0;
  bytes = NULL;

  /*----- open the volume file -----*/
  FILE* input_fd;

  if ((input_fd = SbFileHelper::open( lpszPathName, "rb" )) == NULL) {
    TRACE("Can't open volume file: %s\n", lpszPathName.toLatin1());
    return FALSE;
  }

  char magic[4];
  fread( (void *)magic, sizeof(magic), 1, input_fd );
  if (magic[0] != 'v' || magic[1] != 'o' ||
      magic[2] != 'l' || magic[3] != '3'  ) {
    /* Not a valid file */
        SbFileHelper::close(input_fd);
    return FALSE;
  }

  int xDim, yDim, zDim;
  fread( (void *) &xDim, sizeof(int), 1, input_fd );
  fread( (void *) &yDim, sizeof(int), 1, input_fd );
  fread( (void *) &zDim, sizeof(int), 1, input_fd );
  SwapBytes(&xDim, sizeof(int));
  SwapBytes(&yDim, sizeof(int));
  SwapBytes(&zDim, sizeof(int));

  float xSpace, ySpace, zSpace;
  fread( (void *) &xSpace, sizeof(float), 1, input_fd );
  fread( (void *) &ySpace, sizeof(float), 1, input_fd );
  fread( (void *) &zSpace, sizeof(float), 1, input_fd );
  SwapBytes(&xSpace, sizeof(float));
  SwapBytes(&ySpace, sizeof(float));
  SwapBytes(&zSpace, sizeof(float));

  int dataOffset;
  short minVal, maxVal;
  unsigned char bytesPerVoxel;
  fread( (void *)&dataOffset, sizeof(int), 1, input_fd );
  fread( (void *)&minVal, 2, 1, input_fd );
  fread( (void *)&maxVal, 2, 1, input_fd );
  fread( (void *)&bytesPerVoxel, 1, 1, input_fd );
  SwapBytes(&dataOffset, sizeof(int));
  SwapBytes(&minVal, sizeof(short));
  SwapBytes(&maxVal, sizeof(short));

  fseek(input_fd, 64, SEEK_SET);
  unsigned char id[64];
  fread( (void *)id, 1, sizeof(id), input_fd );

  size_t voxelsPerSlice = xDim * yDim;
  size_t bytesPerSlice  = voxelsPerSlice * bytesPerVoxel;
  size_t totalBytes = bytesPerSlice * zDim;

  /*----- create a buffer big enough for entire image -----*/

  unsigned char *image = (unsigned char *) malloc(totalBytes);
  if (image == (unsigned char *) NULL) {
    TRACE("***Error allocating %llu bytes for voxel data\n", (unsigned long long)totalBytes);
    return FALSE;
  }

  if (bytesPerVoxel == 1) {
    unsigned char *pSlice = image;
    fseek( input_fd, dataOffset, SEEK_SET );
    for (int i = 0; i < zDim; i++) {
      if (fread( pSlice, 1, bytesPerSlice, input_fd ) != bytesPerSlice) {
        TRACE("Reading image #%d failed\n", i);
        SbFileHelper::close(input_fd);
        free(image);
        return FALSE;
      }
      pSlice += bytesPerSlice;
    }
  }
  else {
    /* Don't handle 16 bit data yet... */
    TRACE("*** 16 bit data not currently handled\n");
    SbFileHelper::close(input_fd);
    return FALSE;
  }

  width  = xDim;
  height = yDim;
  depth  = zDim;
  bytes  = image;
  SbFileHelper::close(input_fd);
  return TRUE;
}

/*-------------------------------------------------------------------*/
int 
ReadHPVolFile( const SbString& lpszPathName, int &width, int &height, 
              int &depth, unsigned char *&bytes )
{
  int i;
  FILE* input_fd;
  int totalDepth;
  size_t bytesPerPixel;
  size_t pixelsPerSlice;
  size_t bytesPerSlice;
  unsigned char *image;

  /*----- open the volume file -----*/

  if ((input_fd = SbFileHelper::open(lpszPathName, "rb")) == NULL) {
    TRACE("Can't open volume file: %s\n", lpszPathName.toLatin1());
    return FALSE;
  }

  /*----- read header info from the volume file -----*/
  fread( &_MagicNumber, sizeof(int), 1, input_fd );
  SwapBytes(&_MagicNumber, sizeof(int));
  if ((_MagicNumber != C_VR_MAGIC_NUMBER) &&
      (_MagicNumber != C_VR_MAGIC_NUMBER_OLD)) {
    TRACE("This is not a properly identified volume file\n");
    return FALSE;
  }
  fread( (void *) &_HeaderLength, sizeof(int), 1, input_fd );
  fread( (void *) &_VolumeDimensions, sizeof(C_IntTuple), 1, input_fd );
  fread( (void *) &_PixelDepth, sizeof(int), 1, input_fd );
  fread( (void *) &_IndexBits, sizeof(int), 1, input_fd );
  SwapBytes(&_HeaderLength, sizeof(int) );
  SwapBytes(&_VolumeDimensions, sizeof(C_IntTuple) );
  SwapBytes(&_PixelDepth, sizeof(int) );
  SwapBytes(&_IndexBits, sizeof(int) );
  _PixelDepth -= _IndexBits;
  if (_MagicNumber == C_VR_MAGIC_NUMBER) {
    fread( (void *) &_ScaleX, sizeof(float), 1, input_fd );
    fread( (void *) &_ScaleY, sizeof(float), 1, input_fd );
    fread( (void *) &_ScaleZ, sizeof(float), 1, input_fd );
    fread( (void *) &_RotX, sizeof(float), 1, input_fd );
    fread( (void *) &_RotY, sizeof(float), 1, input_fd );
    fread( (void *) &_RotZ, sizeof(float), 1, input_fd );
    SwapBytes(&_ScaleX, sizeof(float));
    SwapBytes(&_ScaleY, sizeof(float));
    SwapBytes(&_ScaleZ, sizeof(float));
    SwapBytes(&_RotX, sizeof(float));
    SwapBytes(&_RotY, sizeof(float));
    SwapBytes(&_RotZ, sizeof(float));
  }
  else {
    _ScaleX = 1.0;
    _ScaleY = 1.0;
    _ScaleZ = 1.0;
    _RotX = 0.0;
    _RotY = 0.0;
    _RotZ = 0.0;
  }

  /*----- move the file pointer to the first voxel -----*/
  fseek( input_fd, _HeaderLength, SEEK_SET );        

  /*----- first, let's figure out the GL pixel format -----*/
  totalDepth = _PixelDepth + _IndexBits;
  switch (totalDepth) {
  case 8:
    _PixelFormat = GL_INTENSITY8_EXT;
    _PixelType = GL_UNSIGNED_BYTE;
    break;
  case 16:
    if (_IndexBits == 0) {
      _PixelFormat = GL_INTENSITY16_EXT;
      _PixelType = GL_UNSIGNED_SHORT;
    }
    else if (_IndexBits == 4) {
      _PixelFormat = GL_INDEX4_INTENSITY12_HP;
      _PixelType = GL_UNSIGNED_SHORT;
    }
    else if (_IndexBits == 8) {
      _PixelFormat = GL_INDEX8_INTENSITY8_HP;
      _PixelType = GL_UNSIGNED_SHORT;
    }
    else {
      TRACE("This combination of index (%d bits) & pixel (%d bits)\n", 
            _IndexBits, _PixelDepth);
      TRACE("depths is not supported\n");
      return FALSE;
    }
    break;
  default:
    TRACE("This combination of index (%d bits) & pixel (%d bits)\n", 
          _IndexBits, _PixelDepth);
    TRACE("depths is not supported\n");
    return FALSE;
  }

  /*----- determine the size of the dataset -----*/
  if (_PixelType == GL_UNSIGNED_BYTE)
    bytesPerPixel = sizeof(unsigned char);
  else 
    bytesPerPixel = sizeof(unsigned short);

  /* MMH - Demo software can only handle 8-bit data for now */
  if (_PixelType != GL_UNSIGNED_BYTE) {
    printf("Sorry!  Can only handle 8-bit data.\n");
    return FALSE;
  }

  pixelsPerSlice = _VolumeDimensions.x * _VolumeDimensions.y;
  bytesPerSlice = pixelsPerSlice * bytesPerPixel;
  size_t totalBytes = bytesPerSlice * _VolumeDimensions.z;

  /*----- create a buffer big enough for entire image -----*/
  image = (unsigned char *) malloc(totalBytes);
  if (image == (unsigned char *) NULL) {
    TRACE("An error occurred allocating image buffer\n");
    return FALSE;
  }

  /*----- read slices from the file and add to the volume -----*/
  unsigned char *pSlice = image;

  for (i = 0; i < _VolumeDimensions.z; i++) {
    if (fread( pSlice, 1, bytesPerSlice, input_fd ) != bytesPerSlice) {
      TRACE("Reading image #%d failed\n", i);
      SbFileHelper::close( input_fd );
      free(image);
      return FALSE;
    }
    pSlice += bytesPerSlice;
  }

  /*----- Create new volume -----*/
#ifdef VOLPRO
  VLIVolume *newvolume = VLIVolume::Create(kVLIVoxelFormatUINT8, 
                                           image,   // ptr to voxels
                                           _VolumeDimensions.x, 
                                           _VolumeDimensions.y, 
                                           _VolumeDimensions.z, 
                                           VLIVolume::kSliced);
  
  if (!newvolume) {
    AfxMessageBox("ReadVolFile: VLIVolume::Create failed");
    return FALSE;
  }

  /*----- replace old volume -----*/
  if(m_volume)
    m_volume->Release();
  
  m_volume = newvolume;
#else // VOLPRO
  width  = _VolumeDimensions.x;
  height = _VolumeDimensions.y;
  depth  = _VolumeDimensions.z;
  bytes  = image;
#endif // VOLPRO

  /*----- close file and free buffer -----*/
  SbFileHelper::close(input_fd);
  //? free(image);
  if (_Manufacturer[0])
    TRACE("Original manufacturer is %s\n", _Manufacturer);
  if (_OriginalFileName[0])
    TRACE("Original file name is %s\n", _OriginalFileName);
  
  return TRUE;
}

/*+----------------+
  |   ReadVolume   |
  +----------------+*/ 

int 
ReadVoxFile(const SbString& lpszPathName, int &width, int &height, 
            int &depth, unsigned char *&bytes)
{
  int i;
  FILE* input_fd;
  int totalDepth;
  size_t bytesPerPixel;
  size_t pixelsPerSlice;
  size_t bytesPerSlice;
  unsigned char *image;

  /*----- open the volume file -----*/

  if ((input_fd = SbFileHelper::open( lpszPathName, "rb" )) == NULL) {
    TRACE("Can't open volume file: %s\n", lpszPathName.toLatin1());
    return FALSE;
  }

  /*----- read header info from the volume file -----*/
  fread( &_MagicNumber, sizeof(int), 1, input_fd );
  SwapBytes(&_MagicNumber, sizeof(int));
  if ((_MagicNumber != C_VR_MAGIC_NUMBER) &&
      (_MagicNumber != C_VR_MAGIC_NUMBER_OLD)) {
    TRACE("This is not a properly identified volume file\n");
    return FALSE;
  }
  
  fread( (void *) &_HeaderLength, sizeof(int), 1, input_fd );
  fread( (void *) &_VolumeDimensions, sizeof(C_IntTuple), 1, input_fd );
  fread( (void *) &_PixelDepth, sizeof(int), 1, input_fd );
  fread( (void *) &_IndexBits, sizeof(int), 1, input_fd );
  SwapBytes(&_HeaderLength, sizeof(int));
  SwapBytes(&_VolumeDimensions, sizeof(C_IntTuple));
  SwapBytes(&_PixelDepth, sizeof(int));
  SwapBytes(&_IndexBits, sizeof(int));
  _PixelDepth -= _IndexBits;
  if (_MagicNumber == C_VR_MAGIC_NUMBER) {
    fread( (void *) &_ScaleX, sizeof(float), 1, input_fd );
    fread( (void *) &_ScaleY, sizeof(float), 1, input_fd );
    fread( (void *) &_ScaleZ, sizeof(float), 1, input_fd );
    fread( (void *) &_RotX, sizeof(float), 1, input_fd );
    fread( (void *) &_RotY, sizeof(float), 1, input_fd );
    fread( (void *) &_RotZ, sizeof(float), 1, input_fd );
    SwapBytes(&_ScaleX, sizeof(float));
    SwapBytes(&_ScaleY, sizeof(float));
    SwapBytes(&_ScaleZ, sizeof(float));
    SwapBytes(&_RotX, sizeof(float));
    SwapBytes(&_RotY, sizeof(float));
    SwapBytes(&_RotZ, sizeof(float));
  }
  else {
    _ScaleX = 1.0;
    _ScaleY = 1.0;
    _ScaleZ = 1.0;
    _RotX = 0.0;
    _RotY = 0.0;
    _RotZ = 0.0;
  }

  /*----- move the file pointer to the first voxel -----*/
  fseek( input_fd, _HeaderLength, SEEK_SET );        

  /*----- first, let's figure out the GL pixel format -----*/
  totalDepth = _PixelDepth + _IndexBits;
  switch (totalDepth) {
  case 8:
    _PixelFormat = GL_INTENSITY8_EXT;
    _PixelType = GL_UNSIGNED_BYTE;
    break;
  case 16:
    if (_IndexBits == 0) {
      _PixelFormat = GL_INTENSITY16_EXT;
      _PixelType = GL_UNSIGNED_SHORT;
    }
    else if (_IndexBits == 4) {
      _PixelFormat = GL_INDEX4_INTENSITY12_HP;
      _PixelType = GL_UNSIGNED_SHORT;
    }
    else if (_IndexBits == 8) {
      _PixelFormat = GL_INDEX8_INTENSITY8_HP;
      _PixelType = GL_UNSIGNED_SHORT;
    }
    else {
      TRACE("This combination of index (%d bits) & pixel (%d bits)\n", 
            _IndexBits, _PixelDepth);
      TRACE("depths is not supported\n");
      return FALSE;
    }
    break;
  default:
    TRACE("This combination of index (%d bits) & pixel (%d bits)\n", 
          _IndexBits, _PixelDepth);
    TRACE("depths is not supported\n");
    return FALSE;
  }

  /*----- determine the size of the dataset -----*/
  if (_PixelType == GL_UNSIGNED_BYTE)
    bytesPerPixel = sizeof(unsigned char);
  else 
    bytesPerPixel = sizeof(unsigned short);
  
  pixelsPerSlice = _VolumeDimensions.x * _VolumeDimensions.y;
  bytesPerSlice = pixelsPerSlice * bytesPerPixel;
  size_t totalBytes = bytesPerSlice * _VolumeDimensions.z;

  /*----- create a buffer big enough for entire image -----*/
  image = (unsigned char *) malloc(totalBytes);
  if (image == (unsigned char *) NULL) {
    TRACE("An error occurred allocating image buffer\n");
    return FALSE;
  }

  /*----- read slices from the file and add to the volume -----*/
  unsigned char *pSlice = image;

  for (i = 0; i < _VolumeDimensions.z; i++) {
    if (fread( pSlice, bytesPerSlice, 1, input_fd ) != bytesPerSlice) {
      TRACE("Reading image #%d failed\n", i);
      SbFileHelper::close(input_fd);
      free(image);
      return FALSE;
    }
    pSlice += bytesPerSlice;
  }

  /*----- Create new volume -----*/
#ifdef VOLPRO
  VLIVolume *newvolume = VLIVolume::Create(kVLIVoxelFormatUINT8, 
                                           image,   // ptr to voxels
                                           _VolumeDimensions.x, 
                                           _VolumeDimensions.y, 
                                           _VolumeDimensions.z, 
                                           VLIVolume::kSliced);

  if (!newvolume) {
    AfxMessageBox("ReadVolFile: VLIVolume::Create failed");
    return FALSE;
  }

  /*----- replace old volume -----*/
  if(m_volume)
    m_volume->Release();
  m_volume = newvolume;
#else // VOLPRO
  width  = _VolumeDimensions.x;
  height = _VolumeDimensions.y;
  depth  = _VolumeDimensions.z;
  bytes  = image;
#endif // VOLPRO

  /*----- close file and free buffer -----*/
  SbFileHelper::close(input_fd);
  //? free(image);
  if (_Manufacturer[0])
    TRACE("Original manufacturer is %s\n", _Manufacturer);
  if (_OriginalFileName[0])
    TRACE("Original file name is %s\n", _OriginalFileName);

  return TRUE;
}


