/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

// Example - Background nodes
//
// Usage: backgroundNode <filename.iv>
//
// Notes:
//   Requires Open Inventor version 5.0 or higher.

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtColorEditor.h> 

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoSwitch.h>

#include <Inventor/SoInput.h>

#include <DialogViz/SoDialogVizAll.h>

// Headers for the background nodes
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoImageBackground.h>



#include <Inventor/helpers/SbFileHelper.h>

// Global variables
static SoSwitch *pGradientSwitch = NULL;
static SoSwitch *pImgBackgSwitch = NULL;
static SoGradientBackground *pGradient = NULL;
static SoImageBackground *pImgBackg = NULL;
static SoXtColorEditor *pColEdit  = NULL;
static SoTopLevelDialog *myTopLevelDialog = NULL;

// Constants
SbString ivFilename = SbFileHelper::expandString( "$OIVHOME" ) + "/examples/source/Inventor/Features/BigImageRender/test.iv";             // model to display
const char *uiFilename = "$OIVHOME/examples/source/Inventor/Features/BackgroundNode/BackgroundNodeUI.iv"; // user interface

// This image demonstrates that partially transparent images are
// correctly rendered (other backgrounds show through)
const char *imgFilename = "$OIVHOME/examples/source/Inventor/Features/BackgroundNode/cypress.png";

/////////////////////////////////////////////////////////////
//
// Auditor class to handle events in our user interface

class testAuditor : public SoDialogAuditor
{
public:
  testAuditor();
  
private:
  void dialogCheckBox(SoDialogCheckBox* cpt);
  void dialogPushButton(SoDialogPushButton* cpt);
  void dialogComboBox(SoDialogComboBox* cpt);
  void dialogEditText(SoDialogEditText* cpt);
};

Widget buildInterface(Widget window);

/////////////////////////////////////////////////////////////

int
main(int argc, char **argv)
{
  char *filename = (char *)ivFilename.toLatin1 ();

  if (argc > 1) {
    filename = argv[1];
  }

  FILE *fp = fopen(filename, "r");

  if (fp == NULL) {
    printf("Unable to open model file '%s'\n", filename);
    filename = NULL;
  }
  else {
    fclose(fp);
  }
  
  // Initialize Inventor
  Widget myWindow = SoXt::init(argv[0]);
  
  // Initialize DialogViz for user interface
  SoDialogViz::init();
  
  // Try to read the .iv file (if any)
  SoSeparator *pRoot = NULL;
  if (filename != NULL) {
    SoInput in;
    in.openFile(filename);
    pRoot = SoDB::readAll(&in);
    in.closeFile();
  }

  // If that failed, create our favorite geometry
  if (pRoot == NULL) {
    SoMaterial *pMatl = new SoMaterial;
    SoCone *pCone = new SoCone;

    pMatl->diffuseColor.setValue(1,0,0);

    pRoot = new SoSeparator;
    pRoot->addChild(pMatl);
    pRoot->addChild(pCone);
  }
  pRoot->ref();

  // Create background nodes
  //
  // Image background node will be traversed after gradient background
  // node to show how we can put an image on top of a gradient...
  // Initially the image will be displayed in the lower left corner.
  pGradient = new SoGradientBackground;
  pImgBackg = new SoImageBackground;
  pImgBackg->filename = imgFilename;
  pImgBackg->style = SoImageBackground::LOWER_LEFT;

  // Create some switch nodes so we can turn the backgrounds on and off
  pGradientSwitch = new SoSwitch;
  pImgBackgSwitch = new SoSwitch;
  pGradientSwitch->whichChild = 0;
  pImgBackgSwitch->whichChild = 0;
  pGradientSwitch->addChild(pGradient);
  pImgBackgSwitch->addChild(pImgBackg);
  SoSeparator *pBkgSep = new SoSeparator;
  pBkgSep->addChild(pGradientSwitch);
  pBkgSep->addChild(pImgBackgSwitch);

  // Insert the background nodes at the top of the scene graph
  pRoot->insertChild(pBkgSep, 0);

  // Create a viewer
  Widget parent = buildInterface(myWindow);
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);

  // attach and show viewer
  myViewer->setBackgroundColor(SbColor(0,.65f,.65f));
  myViewer->setSceneGraph(pRoot);
  myViewer->setTitle("Background Test");
  myViewer->show();

  // Loop forever
  SoXt::show(myWindow);
  SoXt::mainLoop();

  pRoot->unref();
  delete myViewer;

  myTopLevelDialog->close();

  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}

/////////////////////////////////////////////////////////////
//
// Load simple interface to experiment with background options

Widget
buildInterface(Widget window)
{
  SoInput myInput;
  if (! myInput.openFile(uiFilename)) {
    printf("Unable to open user interface file '%s'\n", uiFilename);
    return window;
  }
  
  SoGroup *myDialogGroup = SoDB::readAll(&myInput);
  if (! myDialogGroup) {
    return window;
  }
  myTopLevelDialog = (SoTopLevelDialog *)myDialogGroup->getChild(0);
  
  testAuditor *myAuditor = new testAuditor;
  myTopLevelDialog->addAuditor(myAuditor);
  
  SoDialogCustom *customNode = (SoDialogCustom *)
    myTopLevelDialog->searchForAuditorId(SbString("Viewer"));
  
  SoDialogCheckBox *checkbox = (SoDialogCheckBox *)
    myTopLevelDialog->searchForAuditorId(SbString("swapColors"));
  if (checkbox) {
    pGradient->swapColors.connectFrom(&(checkbox->state));
  }
  
  SoDialogEditText *editText = (SoDialogEditText *)
    myTopLevelDialog->searchForAuditorId(SbString("filename"));
  if (editText) {
    editText->editText = imgFilename;
  }
  
  myTopLevelDialog->buildDialog(window, customNode != NULL);
  myTopLevelDialog->show();
  
  return customNode ? customNode->getWidget() : window;
}

///////////////////////////////////////////////////////////////////////

testAuditor::testAuditor()
{
}

/*----------------------------- dialogCheckBox -----------------------------------------*/
void
testAuditor::dialogCheckBox(SoDialogCheckBox* cpt)
{
  SbBool state = cpt->state.getValue();

  if (cpt->auditorID.getValue() == "gradientVisible") {
    pGradientSwitch->whichChild = state ? 0 : -1;
  }

  if (cpt->auditorID.getValue() == "imageVisible") {
    pImgBackgSwitch->whichChild = state ? 0 : -1;
  }
}

/*----------------------------- fileSelection ---------------------------------------*/

#include <Inventor/Xt/SoXtFileSelectionDialog.h> 

static SoXtFileSelectionDialog *pFileDialog = NULL;

void
fileSelectionDialogCB(void *, SoXtFileSelectionDialog *dialog)
{
  char *filename = dialog->getFileName();
  pImgBackg->filename = filename;

  SoDialogEditText *editText =
    (SoDialogEditText *)myTopLevelDialog->searchForAuditorId(SbString("filename"));
  if (editText) {
    editText->editText = filename;
  }
}

/*----------------------------- dialogPushButton ---------------------------------------*/

void
testAuditor::dialogPushButton(SoDialogPushButton *cpt)
{
  if (cpt->auditorID.getValue() == "color0") {
    if (pColEdit == NULL) {
      pColEdit = new SoXtColorEditor;
      pColEdit->setCurrentSliders(SoXtColorEditor::RGB_V);
    }
    pColEdit->attach(&(pGradient->color0), pGradient);
    pColEdit->show();
  }
  if (cpt->auditorID.getValue() == "color1") {
    if (pColEdit == NULL) {
      pColEdit = new SoXtColorEditor;
      pColEdit->setCurrentSliders(SoXtColorEditor::RGB_V);
    }
    pColEdit->attach(&(pGradient->color1), pGradient);
    pColEdit->show();
  }
  if (cpt->auditorID.getValue() == "browse") {
    if (pFileDialog == NULL) {
      pFileDialog = new SoXtFileSelectionDialog;
      pFileDialog->setTitle((char*)"Image file");
    }
    pFileDialog->setAcceptCallback(fileSelectionDialogCB, NULL);
    pFileDialog->show();
  }
}

/*----------------------------- dialogComboBox ---------------------------------------*/

void
testAuditor::dialogComboBox(SoDialogComboBox *cpt)
{
  if (cpt->auditorID.getValue() == "style") {
    int choice = cpt->selectedItem.getValue();
    switch (choice) {
    case 0:
      pImgBackg->style = SoImageBackground::NONE;
      break;
    case 1:
      pImgBackg->style = SoImageBackground::CENTER;
      break;
    case 2:
      pImgBackg->style = SoImageBackground::LOWER_LEFT;
      break;
    case 3:
      pImgBackg->style = SoImageBackground::UPPER_LEFT;
      break;
    case 4:
      pImgBackg->style = SoImageBackground::UPPER_RIGHT;
      break;
    case 5:
      pImgBackg->style = SoImageBackground::LOWER_RIGHT;
      break;
    case 6:
      pImgBackg->style = SoImageBackground::STRETCH;
      break;
    case 7:
      pImgBackg->style = SoImageBackground::TILE;
      break;
    }
  }
}

/*----------------------------- dialogEditText ---------------------------------------*/

void
testAuditor::dialogEditText(SoDialogEditText *cpt)
{
  if (cpt->auditorID.getValue() == "filename") {
    SbString filename = cpt->editText.getValue();
    pImgBackg->filename = filename;
  }
}


