// OffscreenRenderer Test
//
// Copyright 2023 FEI SAS
//
// Author: Federico Gamba

#include <Inventor/SoInput.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/Xt/SoXtRenderArea.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/SoOffscreenRenderArea.h>

#include <DialogViz/SoDialogVizAll.h>

Widget buildInterface(Widget);

SoXtRenderArea*  renderArea = NULL;

SoTopLevelDialog* topLevelDialog0 = NULL;
SoDialogCustom*   customNode0     = NULL;

SbVec2i32 myOffscreenSize(2500, 2500);
SoSeparator* myRoot;
SbString ImageFileName="image";


class AuditorClass : public SoDialogAuditor
{
public:

  AuditorClass();

  void dialogPushButton(SoDialogPushButton* cpt);
  void dialogComboBox(SoDialogComboBox* combobox);

  void refreshTransparency ()
  {
	  SoDialogComboBox* combobox = (SoDialogComboBox*)
		  topLevelDialog0->searchForAuditorId(SbString("transparencytype"));
	  this->setTransparencyType(combobox->items[combobox->selectedItem.getValue()]);
  }

protected :

  void setTransparencyType(SbString name)
  {
	  renderArea->getGLRenderAction()->setTransparencyType(transparencyTypes[name]);
  }

private :
	std::map<SbString, SoGLRenderAction::TransparencyType> transparencyTypes;

};

//give a select list of all possible OIV transparencies
AuditorClass::AuditorClass()
{
  transparencyTypes[SbString("NO_SORT")] = SoGLRenderAction::NO_SORT;
  transparencyTypes[SbString("OPAQUE_FIRST")] = SoGLRenderAction::OPAQUE_FIRST;
  transparencyTypes[SbString("SORTED_OBJECT")] = SoGLRenderAction::SORTED_OBJECT;
  transparencyTypes[SbString("SORTED_PIXEL")] = SoGLRenderAction::SORTED_PIXEL;
}

//transparency type of export should be the same of the viewer
void
AuditorClass::dialogComboBox(SoDialogComboBox* combobox)
{
  if (combobox->auditorID.getValue() == "transparencytype")
	  this->setTransparencyType(combobox->items[combobox->selectedItem.getValue()]);
}

// this is where off-screen rendering occurs
void
AuditorClass::dialogPushButton(SoDialogPushButton* cpt)
{
  if (cpt->auditorID.getValue() == "generate")
  {
    // get image size
    SoDialogIntegerSlider* cptw = (SoDialogIntegerSlider*)
      topLevelDialog0->searchForAuditorId(SbString("width"));

    SoDialogIntegerSlider* cpth = (SoDialogIntegerSlider*)
      topLevelDialog0->searchForAuditorId(SbString("height"));

    myOffscreenSize[0] = cptw->value.getValue();
	  myOffscreenSize[1] = cpth->value.getValue();

    // off-screen renderer setup
    SoRef<SoOffscreenRenderArea> myOffscreen = new SoOffscreenRenderArea;
    myOffscreen->setSize(myOffscreenSize);
    myOffscreen->setSceneGraph(myRoot);

    // set up tiling if required
    SoDialogCheckBox* cpt3 = (SoDialogCheckBox*)
      topLevelDialog0->searchForAuditorId(SbString("subTiles"));

    if (cpt3->state.getValue() == TRUE)
    {
      SoDialogIntegerSlider* cpt4 = (SoDialogIntegerSlider*)
        topLevelDialog0->searchForAuditorId(SbString("tileWidth"));
      SoDialogIntegerSlider* cpt5 = (SoDialogIntegerSlider*)
        topLevelDialog0->searchForAuditorId(SbString("tileHeight"));

      myOffscreen->setTile(SbVec2i32(cpt4->value.getValue(), cpt5->value.getValue()), 2);
    }

    // off-screen use same background as the SoRenderArea
    SbColor bkgRGB = renderArea->getBackgroundColor();
    myOffscreen->setClearColor( SbColorRGBA(bkgRGB[0], bkgRGB[1], bkgRGB[2], 1.f) );

    // off-screen use same transparency type as the SoRenderArea
    myOffscreen->setTransparencyType(renderArea->getGLRenderAction()->getTransparencyType());

    // serialize using desired file format
    SoDialogComboBox* cpt2 = (SoDialogComboBox*)
      topLevelDialog0->searchForAuditorId(SbString("fileFormat"));

    //dump on disk in different file formats
    SbString fileExtension;
    switch (cpt2->selectedItem.getValue())
    {
    case 1: fileExtension = ".png"; break;
    case 2: fileExtension = ".ps"; break;
    case 3: fileExtension = ".rgb"; break;
    case 4: fileExtension = ".tiff"; break;
    case 5: fileExtension = ".jpg"; break;
    case 6: fileExtension = ".png"; break;
    case 7: fileExtension = ".bmp"; break;
    case 8: fileExtension = ".png"; break;
    case 9: fileExtension = ".png"; break;
    }
    myOffscreen->renderToFile(ImageFileName + fileExtension);

  }
}

AuditorClass* myAuditorClass;

int
main(int argc, char **argv)
{
  // Initialize Inventor. This returns a main window to use.
  // If unsuccessful, exit.
  Widget mainWindow = SoXt::init(argv[0]); // pass the app name
  
  SoDialogViz::init();
  
  if (mainWindow == NULL)
    exit(1);

  // Set up viewer
  Widget parent = buildInterface(mainWindow);
  renderArea = new SoXtRenderArea(parent);
  renderArea->setBackgroundColor(SbColor(0.7f,0.7f,0.7f));

  // setup scene-graph
  //test.iv contains a simple scene with a transparent cube and a cone
  SbString ivFilename = SbFileHelper::expandString( "$OIVHOME" ) + "/examples/source/Inventor/Features/BigImageRender/test.iv";

  //eventually load a file given as parameter
  if (argc > 1)
    ivFilename = argv[1];

  FILE *fp = fopen(ivFilename.toLatin1 (), "r");
  if (fp == NULL)
  {
    printf("Unable to open '%s'\n", ivFilename.toLatin1());
    ivFilename.makeNull();
  }
  else
    fclose(fp);

  // Read Inventor file or create simple scene
  myRoot = NULL;
  if (! ivFilename.isEmpty())
  {
    SoInput in;
    in.openFile(ivFilename.toLatin1 ());

    myRoot = SoDB::readAll(&in);
    in.closeFile();
	ImageFileName = ivFilename;
  }
  else
  {
    SoMaterial *pMatl = new SoMaterial;
    SoCone *pCone = new SoCone;
    SoCube *pCube = new SoCube;

    pMatl->diffuseColor.setValue(1,0,0);
	  pMatl->transparency.setValue(0.5);

    myRoot = new SoSeparator;
    myRoot->addChild(pMatl);
    myRoot->addChild(pCone);
    myRoot->addChild(pCube);
  }

  if (!myRoot)
  {
    printf("*** Unable to read file...\n");
    exit(1);
  }

  myRoot->ref();

  // Add camera and light if needed
  SoSearchAction sa;
  sa.setType(SoCamera::getClassTypeId());
  sa.apply(myRoot);
  SoPath *pCamPath = sa.getPath();

  if ( !pCamPath )
  {
    SoSeparator *pScene = myRoot;
    myRoot = new SoSeparator;
    SoPerspectiveCamera *pCam = new SoPerspectiveCamera;
    myRoot->addChild( pCam );
    myRoot->addChild( new SoDirectionalLight );
    myRoot->addChild( pScene );
    myRoot->ref();
    pScene->unref();
    SbViewportRegion vport( 200, 200 );
    pCam->viewAll(myRoot, vport);
  }

  renderArea->setSceneGraph(myRoot);

  //get transparency type from GUI
  myAuditorClass->refreshTransparency();
  
  renderArea->show();        
  SoXt::show(mainWindow);  // Display main window   
  SoXt::mainLoop();      // Main Inventor event loop

  myRoot->unref();
  delete renderArea;
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}

Widget
buildInterface(Widget window)
{
  SbString interfaceFilename = SbFileHelper::expandString( "$OIVHOME" ) + "/examples/source/Inventor/Features/BigImageRender/bigImageDialog.iv";
  SoInput myInput;

  if (! myInput.openFile(interfaceFilename)) return window;
  SoGroup* myGroup = SoDB::readAll(&myInput);
  if (! myGroup) return window;
  
  topLevelDialog0 = (SoTopLevelDialog*) myGroup->getChild(0); 

  if (topLevelDialog0) { 
    topLevelDialog0->buildDialog( window, TRUE );
    myAuditorClass = new AuditorClass;
    topLevelDialog0->addAuditor(myAuditorClass);
    customNode0 = (SoDialogCustom*)
    topLevelDialog0->searchForAuditorId(SbString("renderArea"));
    topLevelDialog0->show();
  }
 
  return customNode0 ? customNode0->getWidget() : window;
}
