#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/devices/SoGpuBufferObject.h>

#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoIndexedTriangleStripSet.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoDrawStyle.h>

#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/sensors/SoOneShotSensor.h>

#include <DialogViz/SoDialogVizAll.h>

/////////////////////////////////////////////////////////////////////////
// define/undefined this macro to allow to switch to this type of buffers
/////////////////////////////////////////////////////////////////////////


/// OpenGL specific object
//#include <Inventor/devices/SoGLDevice.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/devices/SoGpuBufferObject.h>


/// CPU specific object
#include <Inventor/devices/SoCpuDevice.h>
#include <Inventor/devices/SoCpuContext.h>
#include <Inventor/devices/SoCpuBufferObject.h>


// This enum defined the different type of buffer we can use
// with SoBufferedShape node.
typedef enum _TBufferType{
    BUFFER_OPENGL = 0,
    BUFFER_CPU,
    BUFFER_TYPE_MAX
} TBufferType;

SbString bufferTypeName(const TBufferType type)
{
  switch(type)
  {
    case BUFFER_OPENGL: return SbString("GL");     break;
    case BUFFER_CPU:    return SbString("CPU");    break;
  default: return SbString("Unknown");
  }// switch
}

#define SO_DEFAULT_MESH_SIZE  400
#define SO_MIN_MESH_SIZE      50
#define SO_MESH_SIZE_STEP     50
#define SO_ANIM_STEP          0.05f
#define SO_PERIOD             12.5653f
#define SO_DEFAULT_AMPLITUDE  10.0f
#define SO_TITLE_VBO          "SoBufferedShape"
#define SO_STRIP_COLOR_WIDTH  0.125f

int                        MeshSize=SO_DEFAULT_MESH_SIZE;
float                      MeshAmplitude=SO_DEFAULT_AMPLITUDE*((float)MeshSize/(float)SO_DEFAULT_MESH_SIZE) ;
int                        NumTriangles=MeshSize*MeshSize*2;
int                        AnimStep = (int)(MeshSize * SO_ANIM_STEP);
int                        StripColorWidth = (int)(MeshSize * SO_STRIP_COLOR_WIDTH);
SoVertexProperty          *MeshVtxProp ;
SoDrawStyle               *DrawStyle ;
SoSwitch                  *InfoSwitch ;
SoOneShotSensor           *OneShotSensor ;
uint32_t                   AnimIndex=0 ;
SbBool                     IsColorAnimation=TRUE ;
SbBool                     IsGeomAnimation=FALSE ;
SbBool                     IsTexAnimation=FALSE ;
SbBool                     IsVBO=TRUE ;
SbString                   Title(SO_TITLE_VBO) ;

SoXtExaminerViewer        *MyViewer ;
SoSeparator               *Scene3D ;
SoPerspectiveCamera       *Camera ;

// global variables to switch from different buffer type
TBufferType                g_bufferType = BUFFER_OPENGL;
SoDevice*                  g_device[BUFFER_TYPE_MAX];
SoRef<SoDeviceContext>     g_context[BUFFER_TYPE_MAX];

SoBufferedShape*          g_bufferedShape = NULL;
SoBufferObject*         g_normalsBuffer = NULL;
SoBufferObject*         g_verticesBuffer = NULL;
SoBufferObject*         g_colorsBuffer = NULL;
SoBufferObject*         g_texCoordsBuffer = NULL;
SoBufferObject*         g_indicesBuffer = NULL;

SoBBox*                   g_boundingBoxNode = NULL;
SoTexture2*               g_texture = NULL;

long                       TimeColor, TimeVtx, TimeNormal, TimeTexCoord ;
bool                       fullDemo=true;
/*---------------------------------------------------------------------------*/

void
updateColors()
{
  SbTime t1 (SbTime::getTimeOfDay()) ;

  SoCpuBufferObject* cpuBufferObject = new SoCpuBufferObject;
  cpuBufferObject->ref();

  g_colorsBuffer->map(cpuBufferObject,SoBufferObject::SET);
  uint32_t *colors    = (uint32_t *)cpuBufferObject->map( SoBufferObject::SET );
  if ( colors )
  {
    uint32_t defaultColor = SbColor(1,1,0).getPackedValue() ;
    uint32_t stripColor   = SbColor(0,0,1).getPackedValue() ;

    int minStripColor = AnimIndex ;
    int maxStripColor = AnimIndex + StripColorWidth ;
    if(maxStripColor >= MeshSize) {
      maxStripColor = MeshSize-1 ;
    }

    for(int i=0; i < MeshSize; i++) {
      int lineInd = i*MeshSize ;
      for(int j=0; j < MeshSize; j++) {
        if(j >= minStripColor && j <= maxStripColor)
          colors[lineInd+j] = stripColor ;
        else
          colors[lineInd+j] = defaultColor ;
      }
    }

    SbTime t2 (SbTime::getTimeOfDay()) ;
    TimeColor = (t2 - t1).getMsecValue() ;

  }
  cpuBufferObject->unmap();
  g_colorsBuffer->unmap(cpuBufferObject);
  cpuBufferObject->unref();

  if (g_bufferedShape)
    g_bufferedShape->colorBuffer.touch();
}/*---------------------------------------------------------------------------*/

void
updateTexCoords()
{
  SbTime t1 (SbTime::getTimeOfDay()) ;

  SoCpuBufferObject* cpuBufferObject = new SoCpuBufferObject;
  cpuBufferObject->ref();

  g_texCoordsBuffer->map(cpuBufferObject,SoBufferObject::SET);
  float* texCoords  = (float*)cpuBufferObject->map( SoBufferObject::SET );
  if (texCoords)
  {
    float animStep = (float)AnimIndex / (float) MeshSize ;

    for(int i=0; i < MeshSize; i++) 
    {
      int lineInd = i*MeshSize;

      for(int j=0; j < MeshSize; j++) 
      {
        texCoords[ 2 * (lineInd+j) ] = (float)i/(float)MeshSize;
        texCoords[ 2 * (lineInd+j) + 1 ] = (float)j / (float)MeshSize + animStep;
      }
    }

    SbTime t2 (SbTime::getTimeOfDay()) ;
    TimeTexCoord = (t2 - t1).getMsecValue() ;
  }
  cpuBufferObject->unmap();
  g_texCoordsBuffer->unmap(cpuBufferObject);
  cpuBufferObject->unref();

  if (g_bufferedShape)
    g_bufferedShape->texCoordsBuffer.touch();
}/*---------------------------------------------------------------------------*/

void updateNormals()
{
  SbTime t1 (SbTime::getTimeOfDay()) ;


  SoCpuBufferObject* cpuVerticesBufferObject = new SoCpuBufferObject;
  cpuVerticesBufferObject->ref();

  g_verticesBuffer->map(cpuVerticesBufferObject,SoBufferObject::READ_ONLY);
  float* vertices  = (float*)cpuVerticesBufferObject->map( SoBufferObject::READ_ONLY );

  SoCpuBufferObject* cpuNormalsBufferObject = new SoCpuBufferObject;
  cpuNormalsBufferObject->ref();

  g_normalsBuffer->map(cpuNormalsBufferObject,SoBufferObject::SET);
  float* normals  = (float*)cpuNormalsBufferObject->map( SoBufferObject::SET );


  if (vertices && normals)
  {
    float dx, dy ;

    for ( int i=0; i < MeshSize; i++ ) 
    {
      int lineInd  = i*MeshSize;
      int line1Ind = (i+1)*MeshSize;

      for( int j=0; j < MeshSize; j++ ) 
      {
        dx = dy = 0.0f ;
        if(j+1 < MeshSize)
          dx = vertices[ 3 * (lineInd+j+1) + 2] - vertices[ 3 * (lineInd+j) + 2 ];

        if(i+1 < MeshSize)
          dy = vertices[ 3 * (line1Ind+j) + 2 ]  - vertices[ 3 * (lineInd+j) + 2 ];

        normals[ 3 * (lineInd+j) ] = -dx;
        normals[ 3 * (lineInd+j) + 1 ] = -dy;
        normals[ 3 * (lineInd+j) + 2 ] = 1.f;
      }
    }

    SbTime t2 (SbTime::getTimeOfDay()) ;
    TimeNormal = (t2 - t1).getMsecValue() ;
  }
  cpuNormalsBufferObject->unmap();
  g_normalsBuffer->unmap(cpuNormalsBufferObject);
  cpuNormalsBufferObject->unref();

  cpuVerticesBufferObject->unmap();
  g_verticesBuffer->unmap(cpuVerticesBufferObject);
  cpuVerticesBufferObject->unref();

  if (g_bufferedShape)
    g_bufferedShape->normalBuffer.touch();
}/*---------------------------------------------------------------------------*/

void
updateVertices()
{
  SbTime t1 (SbTime::getTimeOfDay()) ;

  SoCpuBufferObject* cpuBufferObject = new SoCpuBufferObject;
  cpuBufferObject->ref();
  g_verticesBuffer->map(cpuBufferObject,SoBufferObject::SET);
  float* vertices  = (float*)cpuBufferObject->map( SoBufferObject::SET );

  if ( vertices )
  {
    float coef = (float)MeshSize/SO_PERIOD ;
    int size = MeshSize*MeshSize ;

    for(int x=0; x < MeshSize; x++) 
    {
      float sinAlpha = sin( ((float)x+AnimIndex)/coef ) * MeshAmplitude ;

      for(int i=0,y=0; i < size; i += MeshSize, y++)
      {
        vertices[ 3 * (i+x) ] = (float)x;
        vertices[ 3 * (i+x) + 1 ] = (float)y;
        vertices[ 3 * (i+x) + 2 ] = sinAlpha;
      }
    }


    g_boundingBoxNode->boundingBox.setValue( 0.f, 0.f, -1.f * float( MeshAmplitude ), 
      float( MeshSize ), float( size ) / float( MeshSize ), float( MeshAmplitude ) );

    SbTime t2 (SbTime::getTimeOfDay()) ;
    TimeVtx = (t2 - t1).getMsecValue() ;
  }
  cpuBufferObject->unmap();
  g_verticesBuffer->unmap(cpuBufferObject);
  cpuBufferObject->unref();

  if (g_bufferedShape)
    g_bufferedShape->vertexBuffer.touch();
  updateNormals() ;

}/*---------------------------------------------------------------------------*/

template <typename TBuffer>
TBuffer* allocBuffer()
{
  TBuffer* tmp = new TBuffer;
  tmp->ref();
  return tmp;
}

template <>
SoGpuBufferObject* allocBuffer<SoGpuBufferObject>()
{
  SoGpuBufferObject* oglBuffer = new SoGpuBufferObject(SoGpuBufferObject::DYNAMIC);
  oglBuffer->ref();
  return oglBuffer;
}

template <typename TBuffer>
TBuffer* allocBufferIndices()
{
  return allocBuffer<TBuffer>();
}

template <>
SoGpuBufferObject* allocBufferIndices<SoGpuBufferObject>()
{
  SoGpuBufferObject* oglBuffer = new SoGpuBufferObject( SoGpuBufferObject::DYNAMIC );
  oglBuffer->ref();
  return oglBuffer;
}

template <typename TContext,typename TDevice>
TContext* allocContext(TDevice * device)
{
  TContext *ctx = new TContext(device);
  ctx->ref();
  return ctx;
}

template <>
SoGLContext* allocContext<SoGLContext,SoGLContext>(SoGLContext* sharedContext)
{
  SoGLContext* ctx = new SoGLContext( sharedContext, true );
  return ctx;
}


template <typename TBuffer>
void buildMeshTemplate()
{
  // Some cleanup first
  if (g_normalsBuffer)
    g_normalsBuffer->unref();

  if (g_verticesBuffer)
    g_verticesBuffer->unref();

  if (g_colorsBuffer)
    g_colorsBuffer->unref();

  if( g_texCoordsBuffer)
    g_texCoordsBuffer->unref();

  if (g_indicesBuffer)
    g_indicesBuffer->unref();

  NumTriangles=MeshSize*MeshSize*2 ;
  AnimStep = (int)(MeshSize * SO_ANIM_STEP) ;
  StripColorWidth = (int)(MeshSize * SO_STRIP_COLOR_WIDTH) ;
  MeshAmplitude=SO_DEFAULT_AMPLITUDE*((float)MeshSize/(float)SO_DEFAULT_MESH_SIZE) ;

  g_context[g_bufferType]->bind();

  g_normalsBuffer = allocBuffer<TBuffer>();
  g_normalsBuffer->setSize( MeshSize * MeshSize * 3 * sizeof( float ) );
  
  g_verticesBuffer = allocBuffer<TBuffer>();
  g_verticesBuffer->setSize( MeshSize * MeshSize * 3 * sizeof( float ) );
  
  g_colorsBuffer = allocBuffer<TBuffer>();
  g_colorsBuffer->setSize( MeshSize * MeshSize * sizeof( uint32_t ) );
  

  g_texCoordsBuffer = allocBuffer<TBuffer>();
  g_texCoordsBuffer->setSize( MeshSize * MeshSize * 2 * sizeof( float ) );

  g_indicesBuffer = allocBufferIndices<TBuffer>();
  g_indicesBuffer->setSize( MeshSize * MeshSize * 2 * sizeof( int32_t ) );

  SoCpuBufferObject* cpuBufferObject = new SoCpuBufferObject;
  cpuBufferObject->ref();
  g_indicesBuffer->map(cpuBufferObject,SoBufferObject::SET );
  int32_t *coordInd  = (int32_t *)cpuBufferObject->map( SoBufferObject::SET );
  
  int* primitives = new int[ MeshSize ];
  int p = 0;

  int k=0;
  int accum = 0;
  for(int i=0; i < MeshSize-1; i++) {
    int lineInd = i*MeshSize ;
    for(int j=0; j < MeshSize; j++) {
      coordInd[k++] = lineInd+j ;
      coordInd[k++] = lineInd+j+MeshSize ;
    }
    primitives[ p ] = k - accum;
    accum = k;
    p++;
  }

  cpuBufferObject->unmap();
  g_indicesBuffer->unmap(cpuBufferObject);
  cpuBufferObject->unref();

  updateColors();
  updateTexCoords();
  updateVertices();
  g_context[g_bufferType]->unbind();

  if ( !g_bufferedShape )
  {
    g_bufferedShape = new SoBufferedShape;
  }
  
  g_bufferedShape->shapeType = SoBufferedShape::TRIANGLE_STRIP;

  g_bufferedShape->numVertices = MeshSize * MeshSize;

  g_bufferedShape->vertexBuffer = g_verticesBuffer;
  
  g_bufferedShape->colorBuffer = g_colorsBuffer;
  g_bufferedShape->colorComponentsCount = 4;
  g_bufferedShape->colorComponentsType = SbDataType::UNSIGNED_BYTE;
  
  g_bufferedShape->indexBuffer = g_indicesBuffer;
  g_bufferedShape->indexType = SbDataType::UNSIGNED_INT32;
  g_bufferedShape->numVertices.setNum( p );

  for ( int i = 0; i < p; i++ )
    g_bufferedShape->numVertices.set1Value( i, primitives[ i ] );

  delete[] primitives;

  g_bufferedShape->normalBuffer = g_normalsBuffer;

  g_bufferedShape->texCoordsBuffer = g_texCoordsBuffer;
}


/*---------------------------------------------------------------------------*/
void
animateCB(void *, SoSensor *sensor) 
{
  if (g_context[g_bufferType].ptr() && MyViewer->isVisible())
  {
    if (IsGeomAnimation || IsColorAnimation || IsTexAnimation)
    {
      g_context[g_bufferType]->bind();

      if (fullDemo) AnimIndex = (AnimIndex + AnimStep)%MeshSize ;
      if(IsGeomAnimation)
        updateVertices() ;
      if(IsColorAnimation)
        updateColors() ;
      if(IsTexAnimation)
        updateTexCoords() ;

      g_context[g_bufferType]->unbind();
    }
  }
  sensor->schedule() ;
}/*---------------------------------------------------------------------------*/

void
fpsCB(float fps, void *, SoXtViewer *viewer) {
  SbString titleStr(Title) ;
  SbString fpsStr((int)fps) ;
  SbString timeVtxStr(TimeVtx) ;
  SbString timeColorStr(TimeColor) ;
  SbString timeNormalStr(TimeNormal) ;
  SbString timeTexCoordStr(TimeTexCoord) ;

  titleStr += " (";
  titleStr += bufferTypeName(g_bufferType);
  titleStr += ")";
  titleStr += " : " ;
  titleStr += fpsStr ;
  titleStr += " fps" ;
  titleStr += " / " ;
  titleStr += NumTriangles ;
  titleStr += " triangles" ;
  titleStr += " / " ;
  titleStr += timeVtxStr ;
  titleStr += " ms (V)" ;
  titleStr += " / " ;
  titleStr += timeColorStr ;
  titleStr += " ms (C)" ;
  titleStr += " / " ;
  titleStr += timeNormalStr ;
  titleStr += " ms (N)" ;
  titleStr += " / " ;
  titleStr += timeTexCoordStr ;
  titleStr += " ms (T)" ;

  
  TimeColor = TimeVtx = TimeNormal = TimeTexCoord = 0 ;

  viewer->setTitle(titleStr.getString()) ;
}/*----------------------------------------------------------------------------*/


void buildMesh()
{
  switch(g_bufferType)
  {
    case BUFFER_OPENGL: buildMeshTemplate<SoGpuBufferObject>();     break;
    case BUFFER_CPU:    buildMeshTemplate<SoCpuBufferObject>();    break;
  default:
    SoDebugError::post("buildMesh()","Unknown buffer type");
  }// switch
}


void changeBufferType()
{
  // change g_bufferType
  g_bufferType = TBufferType( (int)(g_bufferType) + 1 );
  if (g_bufferType==BUFFER_TYPE_MAX)
    g_bufferType = BUFFER_OPENGL;
}

void 
myKeyPressCB (void *, SoEventCallback *eventCB) 
{
  const SoEvent *event = eventCB->getEvent();
  
  // Check for the keys being pressed
  if (SO_KEY_PRESS_EVENT(event, H)) { // Toggle help display
    if(InfoSwitch->whichChild.getValue() == SO_SWITCH_ALL)
      InfoSwitch->whichChild = SO_SWITCH_NONE ;
    else
      InfoSwitch->whichChild = SO_SWITCH_ALL ;   
  }
  else if(SO_KEY_PRESS_EVENT(event, F1)) { // Toggle material colors
    if ( g_texture->model.getValue() == SoTexture::BLEND )
      g_texture->model = SoTexture::REPLACE;
    else
      g_texture->model = SoTexture::BLEND;
  }
  else if(SO_KEY_PRESS_EVENT(event, F2)) { // Toggle wireframe/filled
    if(DrawStyle->style.getValue()==SoDrawStyle::FILLED)
      DrawStyle->style = SoDrawStyle::LINES ;
    else
      DrawStyle->style = SoDrawStyle::FILLED ;
  }

  else if(SO_KEY_PRESS_EVENT(event, F3)) { // Toggle geometry animation
    IsGeomAnimation = (IsGeomAnimation ? FALSE : TRUE) ;
  }

  else if(SO_KEY_PRESS_EVENT(event, F4)) { // Toggle color animation
    IsColorAnimation = (IsColorAnimation ? FALSE : TRUE) ;
  }

  else if(SO_KEY_PRESS_EVENT(event, F5)) { // Toggle texture coordinate  animation
    IsTexAnimation = (IsTexAnimation ? FALSE : TRUE) ;
  }
  else if(SO_KEY_PRESS_EVENT(event, F6)) { // change g_bufferType
    changeBufferType();
    buildMesh();
  }

  else if(SO_KEY_PRESS_EVENT(event, PAD_ADD)) { // Increase mesh size
    MeshSize += SO_MESH_SIZE_STEP ;
    buildMesh() ;
    Camera->viewAll(Scene3D, MyViewer->getViewportRegion()) ;
  }

  else if(SO_KEY_PRESS_EVENT(event, PAD_SUBTRACT)) { // Decrease mesh size
    if(MeshSize - SO_MESH_SIZE_STEP > SO_MIN_MESH_SIZE) {
      MeshSize -= SO_MESH_SIZE_STEP ;
      buildMesh() ;
      Camera->viewAll(Scene3D, MyViewer->getViewportRegion()) ;
    }
  }
}/*----------------------------------------------------------------------------*/


SoSwitch*
displayInfo() 
{
  // Informations
  SoSwitch *infoSwitch = new SoSwitch ;
  infoSwitch->ref() ;
  infoSwitch->whichChild = SO_SWITCH_ALL ;
  
  SoSeparator *infoSep = new SoSeparator ;
  
  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;
  infoSep->addChild(pickStyle) ;
  
  infoSwitch->addChild(infoSep) ;
  
  SoLightModel *lModel = new SoLightModel ;
  lModel->model = SoLightModel::BASE_COLOR ;
  infoSep->addChild(lModel) ;
  
  SoFont *fontInfo = new SoFont ;
  fontInfo->name = "Courier New" ;
  fontInfo->size = 12 ;
  infoSep->addChild(fontInfo) ;
  
  SoBaseColor *infoColor = new SoBaseColor ;
  infoColor->rgb.setValue(SbColor(1, 1, 0.f)) ;
  infoSep->addChild(infoColor) ;
  
  SoTranslation *transInfo = new SoTranslation ;
  transInfo->translation.setValue(-0.95f, 0.95f, 0.) ;
  infoSep->addChild(transInfo) ;
  
  SoText2 *infoText = new SoText2 ;
  infoText->string.set1Value(0, "H   : Toggle this display") ;
  infoText->string.set1Value(1, "F1  : Toggle material colors") ;
  infoText->string.set1Value(2, "F2  : Toggle wireframe/filled") ;
  infoText->string.set1Value(3, "F3  : Toggle geometry animation") ;
  infoText->string.set1Value(4, "F4  : Toggle color animation") ;
  infoText->string.set1Value(5, "F5  : Toggle texture animation") ;
  infoText->string.set1Value(6, "F6  : Change buffer type (GL/CPU)") ;
  infoText->string.set1Value(7, "+/- : Increase/Decrease mesh refinement") ;
  infoSep->addChild(infoText) ;
  
  infoSwitch->unrefNoDelete() ;
  return infoSwitch ;
}/*---------------------------------------------------------------------------*/


int main( int argc, char **argv )
{
  Widget myWindow = SoXt::init(argv[0]);
  if(myWindow == NULL) exit(1);

  fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);

  SoDialogViz::init();

  // We create the viewer at the very begining to reuse its GL context for later GL buffer allocation
  MyViewer = new SoXtExaminerViewer(myWindow);
  MyViewer->show();
  SoXt::show( myWindow );

  g_device[BUFFER_CPU] = SoCpuDevice::findFirstAvailableDevice();
  g_context[BUFFER_CPU]= allocContext<SoCpuContext,SoCpuDevice>((SoCpuDevice*)g_device[BUFFER_CPU]);
  
  g_device[BUFFER_OPENGL] = NULL;
  g_context[BUFFER_OPENGL]= allocContext<SoGLContext,SoGLContext>( MyViewer->getNormalSoContext()  );

  SoSeparator *root = new SoSeparator;
  root->ref();

  SoMessageDialog *errorDialog = NULL ;

  // Display Info
  InfoSwitch = displayInfo() ;
  root->addChild(InfoSwitch) ;

   // Track the keyboard events
  SoEventCallback *myEventCB = new SoEventCallback;
  myEventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), myKeyPressCB, NULL);
  root->addChild(myEventCB);

  g_texture = new SoTexture2;
  g_texture->filename = "$OIVHOME/examples/data/Inventor/Textures/MarbleTiles.jpg";
  g_texture->model = SoTexture2::BLEND;

  Scene3D = new SoSeparator ;
  root->addChild(Scene3D) ;

  Camera = new SoPerspectiveCamera ;
  Scene3D->addChild(Camera) ;

  DrawStyle = new SoDrawStyle ;
  Scene3D->addChild(DrawStyle) ;

  SoShapeHints* shaphint= new SoShapeHints;
  shaphint->neighborTolerance.setValue(0.0);
  Scene3D->addChild(shaphint);

  Scene3D->addChild(g_texture) ;

  g_boundingBoxNode = new SoBBox;
  g_boundingBoxNode->mode = SoBBox::USER_DEFINED;
  //g_boundingBoxNode->mode = SoBBox::DISABLE;

  buildMesh();

  Scene3D->addChild( g_boundingBoxNode );
  Scene3D->addChild( g_bufferedShape );


  Camera->viewAll(Scene3D, MyViewer->getViewportRegion()) ;
  MyViewer->setSceneGraph(root);
  MyViewer->setBackgroundColor(SbColor(0.,0.,0.5));
  if (fullDemo) MyViewer->setFramesPerSecondCallback(fpsCB) ;
  MyViewer->setSize(SbVec2s(800,800)) ;
  MyViewer->setTitle(Title.getString());

  // Dynamic adjustment of camera clipping planes 
  // is desactivated because it is time consuming
  // to compute them with a not cached scene graph
  MyViewer->setAutoClipping(FALSE) ; 
  MyViewer->viewAll() ;
  

  OneShotSensor = new SoOneShotSensor(animateCB, NULL) ;
  OneShotSensor->schedule() ;

  SoXt::mainLoop();

  // release all allocated context
  for (int i=0;i<BUFFER_TYPE_MAX;i++)
    g_context[i]=NULL;

  if(errorDialog)
    errorDialog->destroy() ;

  root->unref();
  delete OneShotSensor;
  delete MyViewer;

  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


