
#include <Inventor/SoDB.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/SoPrimitiveVertex.h>
#include <Inventor/details/SoDetail.h>
#include <Inventor/details/SoFaceDetail.h>
#include <Inventor/details/SoLineDetail.h>
#include <Inventor/details/SoPointDetail.h>


#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoRayPickAction.h>

#include <Inventor/devices/SoGpuBufferObject.h>

#include <Inventor/nodes/SoDepthOffset.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoLightModel.h> 
#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderObject.h>
#include <Inventor/nodes/SoGradientBackground.h> 
#include <Inventor/nodes/SoShapeHints.h> 
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoDrawStyle.h>

#include <Inventor/nodes/SoPointLight.h> 
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

#include <Inventor/actions/SoWriteAction.h> 
#include <Inventor/actions/SoGetPrimitiveCountAction.h>

#include <DialogViz/SoDialogVizAll.h>

#include <cmath>

//------------------------------------------------------------------------------
// Default values for the super shape

#define DEFAULT_WIDTH 100
#define DEFAULT_HEIGHT 100

#define DEFAULT_A1 1.f
#define DEFAULT_B1 1.f
#define DEFAULT_M1 7.f
#define DEFAULT_N11 20.45f
#define DEFAULT_N12 -0.96f
#define DEFAULT_N13 -0.33f

#define DEFAULT_A2 1.f
#define DEFAULT_B2 1.f
#define DEFAULT_M2 6.f
#define DEFAULT_N21 4.46f
#define DEFAULT_N22 -3.54f
#define DEFAULT_N23 0.52f

//------------------------------------------------------------------------------
// GUI global data
SbString g_guiFilename = 
  "$OIVHOME/examples/source/Inventor/Features/BufferObjects/BufferedShapePicking/gui.iv";

class MyAuditorClass;

SoTopLevelDialog* g_topLevelDialog = NULL;
MyAuditorClass *g_dialogAuditor = NULL;
SoDialogLabel* g_sizeLabel = NULL;

SoDialogLabel* g_screenPosLabel = NULL;
SoDialogLabel* g_worldCoordLabel = NULL;
SoDialogLabel* g_normalLabel = NULL;
SoDialogLabel* g_vertexIndexLabel = NULL;
SoDialogLabel* g_normalIndexLabel = NULL;

//------------------------------------------------------------------------------
// Scene Graph global data modified on gui events.

// Scene graph root separator needed for tunning in the event callback
SoSeparator* g_sceneGraph = NULL;

SoXtExaminerViewer* g_examinerViewer = NULL;

SoSwitch* g_generatePrimitivesSwitch = NULL;

SoBufferedShape* g_bufferedShape = NULL;
bool g_indexedShape = true;
// Do we use TRIANGLE_STRIP or QUAD_STRIP instead of TRIANGLES and QUADS...
bool g_stripPrimitive = true;

SoFaceSet* g_generatedPrimitivesShape = NULL;
SoLineSet* g_generatedPrimitivesLineSet = NULL;
SoPointSet* g_generatedPrimitivesPointSet = NULL;
bool g_generatePrimitivesDisplayed = true;

SoPointSet* g_pickedPoint = NULL;
int g_pickedIndex = -1;
bool g_pointPickingMode = true;
SoFaceSet* g_pickedGeometry = NULL;
SoLineSet* g_pickedLinesGeometry = NULL;

int g_shapeWidth = DEFAULT_WIDTH;
int g_shapeHeight = DEFAULT_HEIGHT;
SoBufferedShape::Type g_shapeType = SoBufferedShape::QUADS;

//------------------------------------------------------------------------------

void createGeometry();
void performGeneratePrimitives();

//------------------------------------------------------------------------------
void 
eventCallback( void* /*userData*/, SoEventCallback *eventCB ) 
{
  char buffer[4096];

  const SoEvent *event = eventCB->getEvent();

  //if (event->getClassTypeId()
  if (SO_MOUSE_RELEASE_EVENT(event, BUTTON1) && g_examinerViewer->isViewing() == false)
  {
    SoMouseButtonEvent* buttonEvent = (SoMouseButtonEvent*)event;
    SbVec2s iMousePosition = buttonEvent->getPosition();

    SoRayPickAction rayPickAction(g_examinerViewer->getViewportRegion());

    rayPickAction.setPoint(iMousePosition);
    rayPickAction.setPickAll(FALSE);
    rayPickAction.setPickingMode(g_pointPickingMode?SoRayPickAction::POINT_PICKING:SoRayPickAction::DEFAULT);

    rayPickAction.apply(g_examinerViewer->getSceneGraph());

    // Reset labels
    sprintf(buffer, "Screen: %d, %d", iMousePosition[0], iMousePosition[1]);
    g_screenPosLabel->label = buffer;
    g_worldCoordLabel->label = "World coordinate: ---";
    g_normalLabel->label = "Normal: ---";
    g_vertexIndexLabel->label = "Vertex coordinate index: ---";
    g_normalIndexLabel->label = "Normal index: ---";
    g_pickedIndex = -1;

    g_pickedLinesGeometry->numVertices.setNum(0);
    g_pickedGeometry->numVertices.setNum(0);

    SoPickedPoint* pickedPoint = rayPickAction.getPickedPoint();
    if (pickedPoint)
    {
      SbVec3f point = pickedPoint->getPoint();
      SbVec3f normal = pickedPoint->getNormal();

      SoVertexProperty* vp = (SoVertexProperty*)g_pickedPoint->vertexProperty.getValue();
      vp->vertex.setNum(1);
      vp->orderedRGBA.setNum(1);
      vp->normal.setNum(0);

      vp->vertex.set1Value(0, point);
      vp->orderedRGBA.set1Value(0, 0x00FF00FF);

      g_pickedPoint->numPoints.setValue(1);
      
      const SoDetail* detail = pickedPoint->getDetail();
      
      sprintf(buffer, "World coordinate: %f, %f, %f", point[0], point[1], point[2]);
      g_worldCoordLabel->label = buffer;
      sprintf(buffer, "Normal: %f, %f, %f", normal[0], normal[1], normal[2]);
      g_normalLabel->label = buffer;

      if (g_pointPickingMode && dynamic_cast<const SoPointDetail*>(detail))
      {
        SoPointDetail* pointDetail = (SoPointDetail*)detail;

        sprintf(buffer, "Vertex coordinate index: %d", pointDetail->getCoordinateIndex());
        g_vertexIndexLabel->label = buffer;

        sprintf(buffer, "Normal index: %d", pointDetail->getNormalIndex());
        g_normalIndexLabel->label = buffer;

        g_pickedIndex = pointDetail->getCoordinateIndex();
        g_pickedGeometry->numVertices.setNum(0);
      }
      else
        if (!g_pointPickingMode && dynamic_cast<const SoFaceDetail*>(detail))
        {
          SoFaceDetail* faceDetail = (SoFaceDetail*)detail;
          int numPoints = faceDetail->getNumPoints();

          SoVertexProperty* vp = (SoVertexProperty*)g_pickedGeometry->vertexProperty.getValue();
          vp->vertex.setNum(numPoints);
          vp->orderedRGBA.setNum(numPoints);
          vp->normal.setNum(numPoints);
          g_pickedGeometry->numVertices.setNum(1);
          g_pickedGeometry->numVertices.set1Value(0, numPoints);

          g_examinerViewer->bindNormalContext();
          SoGpuBufferObject* verticesBuffer = (SoGpuBufferObject*)g_bufferedShape->vertexBuffer.getValue();

          float* vertices = (float*)verticesBuffer->map(SoBufferObject::READ_ONLY);

          for (int i = 0; i < numPoints; ++i)
          {
            int ptCoordIndex = faceDetail->getPoint(i)->getCoordinateIndex();
            point = SbVec3f(vertices[ptCoordIndex * 3], vertices[ptCoordIndex * 3+1], vertices[ptCoordIndex * 3+2]);
            vp->vertex.set1Value(i, point);
            vp->orderedRGBA.set1Value(i, 0xFF0000FF);
          }

          verticesBuffer->unmap();

          g_examinerViewer->unbindNormalContext();
        }
        else
          if (!g_pointPickingMode && dynamic_cast<const SoLineDetail*>(detail))
          {
            SoLineDetail* lineDetail = (SoLineDetail*)detail;

            SoVertexProperty* vp = (SoVertexProperty*)g_pickedLinesGeometry->vertexProperty.getValue();
            vp->vertex.setNum(2);
            vp->orderedRGBA.setNum(2);
            vp->normal.setNum(2);
            g_pickedLinesGeometry->numVertices.setNum(1);
            g_pickedLinesGeometry->numVertices.set1Value(0, 2);

            g_examinerViewer->bindNormalContext();
            SoGpuBufferObject* verticesBuffer = (SoGpuBufferObject*)g_bufferedShape->vertexBuffer.getValue();

            float* vertices = (float*)verticesBuffer->map(SoBufferObject::READ_ONLY);

            for (int i = 0; i < 2; ++i)
            {
              int ptCoordIndex;
              if (i&1)
                ptCoordIndex = lineDetail->getPoint1()->getCoordinateIndex();
              else
                ptCoordIndex = lineDetail->getPoint0()->getCoordinateIndex();

              point = SbVec3f(vertices[ptCoordIndex * 3], vertices[ptCoordIndex * 3+1], vertices[ptCoordIndex * 3+2]);
              vp->vertex.set1Value(i, point);
              vp->orderedRGBA.set1Value(i, 0xFF0000FF);
            }

            verticesBuffer->unmap();

            g_examinerViewer->unbindNormalContext();
          }
    }
    else
    {
      g_pickedPoint->numPoints.setValue(0);
      g_pickedGeometry->numVertices.setNum(0);
    }
  }

  // View all.
  if (SO_KEY_PRESS_EVENT(event, SPACE) )
  {
    // Ok we create our shape.
    g_examinerViewer->viewAll();
  }
  // We write the content of the scene graph to a file.
  if (SO_KEY_PRESS_EVENT(event, RETURN) )
  {
    //writeIvFile( "output.iv", g_sceneGraph );
  }
}


////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle inputs

class MyAuditorClass : public SoDialogAuditor
{
  void dialogPushButton( SoDialogPushButton* cpt )
  {
    if (cpt->auditorID.getValue() == SbString("extrudeButton"))
    {
      if (g_pickedIndex < 0)
        return;

      g_examinerViewer->bindNormalContext();
      SoGpuBufferObject* verticesBuffer = (SoGpuBufferObject*)g_bufferedShape->vertexBuffer.getValue();
      SoGpuBufferObject* normalsBuffer = (SoGpuBufferObject*)g_bufferedShape->normalBuffer.getValue();

      float* normals = (float*)normalsBuffer->map(SoGpuBufferObject::READ_ONLY);
      float* vertices = (float*)verticesBuffer->map(SoGpuBufferObject::READ_WRITE);

      for (int c = 0; c < 3; c++)
        vertices[g_pickedIndex*3 + c] += 0.5f * normals[g_pickedIndex*3 + c];
      
      normalsBuffer->unmap();
      verticesBuffer->unmap();
      
      g_examinerViewer->unbindNormalContext();

      g_examinerViewer->scheduleRedraw();
    }

  }

  void dialogIntegerSlider( SoDialogIntegerSlider* slider )
  {
    if (slider->auditorID.getValue() == SbString("geometrySize"))
    {
      int value = slider->value.getValue();
      if (value > 0)
      {
        g_shapeWidth = g_shapeHeight = value;
        createGeometry();
      }
    }
  }

  void dialogComboBox( SoDialogComboBox* cbox )
  {
    if (cbox->auditorID.getValue() == SbString("shapeType"))
    {
      switch(cbox->selectedItem.getValue())
      {
        case 0: g_shapeType = SoBufferedShape::QUADS; break;
        case 1: g_shapeType = SoBufferedShape::TRIANGLES; break;
        case 2: g_shapeType = SoBufferedShape::LINES; break;
        case 3: g_shapeType = SoBufferedShape::POINTS; break;
      }
      createGeometry();
    }
  }

  void dialogCheckBox( SoDialogCheckBox* checkbox )
  {
    if (checkbox->auditorID.getValue() == SbString("showGeneratePrimnitivesShape"))
    {
      g_generatePrimitivesDisplayed = checkbox->state.getValue();

      performGeneratePrimitives();

      g_generatePrimitivesSwitch->whichChild = checkbox->state.getValue()?SO_SWITCH_ALL:SO_SWITCH_NONE;
    }

    if (checkbox->auditorID.getValue() == SbString("indexedShape"))
    {
      g_indexedShape = checkbox->state.getValue();
      createGeometry();
    }

    if (checkbox->auditorID.getValue() == SbString("pointPicking"))
    {
      g_pointPickingMode = checkbox->state.getValue();
    }

    if (checkbox->auditorID.getValue() == SbString("stripShape"))
    {
      g_stripPrimitive = checkbox->state.getValue();
      createGeometry();
    }
  }
};


//------------------------------------------------------------------------------
Widget
buildInterface( Widget window )
{
  SoInput myInput;
  if (! myInput.openFile( g_guiFilename ))
    return NULL;

  SoRef<SoGroup> myGroup = SoDB::readAll( &myInput );
  if (! myGroup.ptr() )
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );
  g_topLevelDialog->ref();

  g_dialogAuditor = new MyAuditorClass;
  g_topLevelDialog->addAuditor( g_dialogAuditor );

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, customNode != NULL );
  g_topLevelDialog->show();

  g_sizeLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("label_size") );

  g_screenPosLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("ScreenPos") );
  g_worldCoordLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("WorldCoord") );
  g_normalLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("Normal") );
  g_vertexIndexLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("VertexIndex") );
  g_normalIndexLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("NormalIndex") );
  
  return customNode ? customNode->getWidget() : window;
}

void releaseInterface()
{
  g_topLevelDialog->unref();
  delete g_dialogAuditor;
}


void updateLabel(int quadsCount)
{
  char buffer[4096];

  sprintf(buffer, "Buffered shape primitives count: %d", quadsCount);

  g_sizeLabel->label.setValue( buffer );
}

std::vector<SbVec3f> g_triangleVertex;
std::vector<SbVec3f> g_triangleNormal;
void triangleCallback(void* /*userData*/, SoCallbackAction* /*action*/, const SoPrimitiveVertex *v1, const SoPrimitiveVertex *v2, const SoPrimitiveVertex *v3)
{
  g_triangleVertex.push_back(v1->getPoint());
  g_triangleVertex.push_back(v2->getPoint());
  g_triangleVertex.push_back(v3->getPoint());

  g_triangleNormal.push_back(v1->getNormal());
  g_triangleNormal.push_back(v2->getNormal());
  g_triangleNormal.push_back(v3->getNormal());
}

std::vector<SbVec3f> g_lineVertex;
std::vector<SbVec3f> g_lineNormal;
void lineCallback(void* /*userData*/, SoCallbackAction* /*action*/, const SoPrimitiveVertex *v1, const SoPrimitiveVertex *v2)
{
  g_lineVertex.push_back(v1->getPoint());
  g_lineVertex.push_back(v2->getPoint());

  g_lineNormal.push_back(v1->getNormal());
  g_lineNormal.push_back(v2->getNormal());
}

std::vector<SbVec3f> g_pointVertex;
std::vector<SbVec3f> g_pointNormal;
void pointCallback(void* /*userData*/, SoCallbackAction* /*action*/, const SoPrimitiveVertex *v1)
{
  g_pointVertex.push_back(v1->getPoint());
  g_pointNormal.push_back(v1->getNormal());
}


double r(double angle, const SbVec4d& param, const SbVec2d& ab)
{
  double value = param[0] * angle / 4.0;

  if (value < 0)
    value += 2.0 * 3.141592;
  if (value >= 2.0 * 3.141592)
    value -= 2.0 * 3.141592;
  
  double a = std::pow(std::abs(std::cos(value)/ab[0]), param[2]);
  double b = std::pow(std::abs(std::sin(value)/ab[1]), param[3]);
  
  return 1.0 / std::pow(a+b, 1.0 / param[1]);
}

SbVec3d computePos(double theta, double phi, const SbVec4d& param1, const SbVec4d& param2, const SbVec2d& ab1, const SbVec2d& ab2)
{
  double r1 = r(theta, param1, ab1);
  double r2 = r(phi, param2, ab2);

  SbVec3d pos;
  pos[0] = r1 * std::cos(theta) * r2 * std::cos(phi);
  pos[1] = r1 * std::sin(theta) * r2 * std::cos(phi);
  pos[2] = r2 * std::sin(phi);

  return pos;
}

void createGeometry()
{
#define PRIMITIVE_RESTART_VALUE (1<<30)

  g_examinerViewer->bindNormalContext();

  int width = g_shapeWidth;
  int height = g_shapeHeight;

  if (!g_bufferedShape)
  {
    g_bufferedShape = new SoBufferedShape();
    SoSeparator* bufferedShapeSep = new SoSeparator();
    bufferedShapeSep->addChild(g_bufferedShape);
    g_sceneGraph->addChild(bufferedShapeSep);
  }

  SbVec4d param1 = SbVec4d(DEFAULT_M1, DEFAULT_N11, DEFAULT_N12, DEFAULT_N13);
  SbVec4d param2 = SbVec4d(DEFAULT_M2, DEFAULT_N21, DEFAULT_N22, DEFAULT_N23);
  SbVec2d ab1 = SbVec2d(DEFAULT_A1, DEFAULT_B1);
  SbVec2d ab2 = SbVec2d(DEFAULT_A2, DEFAULT_B2);

  int primitivesCount = width*height;
  int primitiveSize = 4;
  int stepDivider = 1;

  if (g_shapeType == SoBufferedShape::TRIANGLES)
  {
    primitivesCount *= 2;
    primitiveSize = 3;

    // Output two triangles for one step.
    stepDivider = 2;
  }
  if (g_shapeType == SoBufferedShape::LINES)
  {
    primitivesCount *= 2;

    // We use triangles mode
    if (g_stripPrimitive)
      primitiveSize = 3;
    else
      primitiveSize = 2;

    // Output two lines for one step.
    stepDivider = 2;
  }
  if (g_shapeType == SoBufferedShape::POINTS)
  {
    // Output one point for one step.
    primitiveSize = 1;
  }

  // Compute the size of the buffer objects.
  // If we use indexed mode we generate one vertex for each point in the grid and much more indices.
  // Otherwise we repeat the vertices for each primitive.
  int buffersEntries = 0;
  
  if (g_indexedShape)
    buffersEntries = width*(height+1);
  else
    buffersEntries = primitivesCount*primitiveSize;

  updateLabel(primitivesCount);

  // The buffer objects for the resulting data
  SoGpuBufferObject* verticesBufferObject = (SoGpuBufferObject*)g_bufferedShape->vertexBuffer.getValue();
  SoGpuBufferObject* colorsBufferObject = (SoGpuBufferObject*)g_bufferedShape->colorBuffer.getValue();
  SoGpuBufferObject* normalsBufferObject = (SoGpuBufferObject*)g_bufferedShape->normalBuffer.getValue();
  SoGpuBufferObject* indicesBuffer = (SoGpuBufferObject*)g_bufferedShape->indexBuffer.getValue();

  if (indicesBuffer && !g_indexedShape)
  {
    g_bufferedShape->indexBuffer = NULL;
    indicesBuffer = NULL;
  }
  else
    if (!indicesBuffer && g_indexedShape)
    {
      g_bufferedShape->indexBuffer = indicesBuffer = new SoGpuBufferObject(SoGpuBufferObject::DYNAMIC);
    }


  if (!verticesBufferObject)
    verticesBufferObject = new SoGpuBufferObject(SoGpuBufferObject::DYNAMIC);
  if (!colorsBufferObject)
    colorsBufferObject = new SoGpuBufferObject(SoGpuBufferObject::DYNAMIC);
  if (!normalsBufferObject)
    normalsBufferObject = new SoGpuBufferObject(SoGpuBufferObject::DYNAMIC);

  verticesBufferObject->setSize(buffersEntries * 3 * sizeof(float));
  colorsBufferObject->setSize(buffersEntries * 3 * sizeof(float));
  normalsBufferObject->setSize(buffersEntries * 3 * sizeof(float));

  float* verticesPtr = (float*)verticesBufferObject->map(SoGpuBufferObject::READ_WRITE);
  float* colorsPtr = (float*)colorsBufferObject->map(SoGpuBufferObject::READ_WRITE);
  float* normalsPtr = (float*)normalsBufferObject->map(SoGpuBufferObject::READ_WRITE);
  unsigned int* indicesPtr = NULL;
  unsigned int indexBufferSize = primitivesCount * primitiveSize;

  if (indicesBuffer)
  {
    // Quad strips
    // Stips divides the indices count by two but we must provide last 2 vertices of each row, so we add height*2.
    // We use primitive restart to switch the strip (See primitive restart in SoBufferedShape documentation).
    if (g_stripPrimitive && primitiveSize == 4)
      indexBufferSize = indexBufferSize / 2 + 
          /* triangle strip extra memory */ 2 * height + 
          /* and one value per row to start a new strip */ height;

    // Triangle strips
    // Stips divides the indices count by three but we must provide last 2 vertices of each row, so we add height*2.
    // We use primitive restart to switch the strip (See primitive restart in SoBufferedShape documentation).
    if (g_stripPrimitive && (primitiveSize == 3 || primitiveSize == 2))
      indexBufferSize = indexBufferSize / 3 + 
          /* triangle strip extra memory */ 2 * height + 
          /* and one value per row to start a new strip */ height;


    indicesBuffer->setSize(indexBufferSize * sizeof(unsigned int));
    indicesPtr = (unsigned int *)indicesBuffer->map(SoBufferObject::READ_WRITE);
  }


  // 1 - Create the grid of points, we have width * (height+1) points to compute (vert[width+1] = vert[0] so no need for the extra column).
  float* vertices = new float[width * (height+1) * 3];
  float* colors = new float[width * (height+1) * 3];
  float* normals = new float[width * (height+1) * 3];

  // Divide xStep by width because we want last vertex = first...
  float xStep = 2.0f * 3.141592f / float(width-1);
  float yStep = 3.141592f / float(height+1);

  // - Compute vertices position
  for (int i = 0; i < width * (height+1); i++)
  {
    int xId = i % width;
    int yId = i / width;

    double a = xId * xStep - 3.141592f;
    double b = yId * yStep - 3.141592f * 0.5f;

    SbVec3d position = computePos(a, b, param1, param2, ab1, ab2);

    for (int c = 0; c < 3; c++)
      vertices[3*i+c] = (float)position[c];
  }

  // - Compute vertices normal / color
  for (int i = 0; i < width * (height+1); i++)
  {
    // Last row copy previous one.
    if ((i/width) == height)
    {
      for (int c = 0; c < 3; c++)
      {
          normals[3*i+c] = normals[3*(i-width)+c];
          colors[3*i+c] = colors[3*(i-width)+c];
      }
    }
    else
      // Last column copy previous one.
      if ((i%width) == width-1)
      {
        for (int c = 0; c < 3; c++)
        {
          normals[3*i+c] = normals[3*(i-1)+c];
          colors[3*i+c] = colors[3*(i-1)+c];
        }
      }
      // Let's compute the normal
      else
      {
        SbVec3f v[3];
        v[0] = SbVec3f(vertices[3*i], vertices[3*i+1], vertices[3*i+2]);
        v[1] = SbVec3f(vertices[3*(i+1)], vertices[3*(i+1)+1], vertices[3*(i+1)+2]);
        v[2] = SbVec3f(vertices[3*(i+width)], vertices[3*(i+width)+1], vertices[3*(i+width)+2]);

        SbVec3f n = (v[1] - v[0]).cross(v[2] - v[0]);
        n.normalize();

        for (int c = 0; c < 3; c++)
        {
          normals[3*i+c] = n[c];
          colors[3*i+c] = std::abs(n[c]);
        }
      }
  }

  // 2 - If indexed, just copy the previous arrays and fill the index buffer
  if (indicesBuffer)
  {
    // Copy the buffers...
    memcpy(verticesPtr, vertices, buffersEntries * 3 * sizeof(float));
    memcpy(normalsPtr, normals, buffersEntries * 3 * sizeof(float));
    memcpy(colorsPtr, colors, buffersEntries * 3 * sizeof(float));

    for(int i = 0; i < primitivesCount; ++i)
    {
      int xId = (i/stepDivider) % width;
      int yId = (i/stepDivider) / width;

      if (g_shapeType == SoBufferedShape::QUADS)
      {
        // Strip mode
        if (g_stripPrimitive)
          {
            *(indicesPtr++) = yId * width + xId;
            *(indicesPtr++) = (yId+1) * width + xId;

            // Adds last two indices and the primitive restart value
            if ((i % width) == width - 1)
            {
              *(indicesPtr++) = yId * width + ((xId + 1)%width);
              *(indicesPtr++) = (yId+1) * width + ((xId + 1)%width);
              *(indicesPtr++) = PRIMITIVE_RESTART_VALUE;
            }
          }
        // No strip here
        else
        {
          *(indicesPtr++) = yId * width + xId;
          *(indicesPtr++) = yId * width + ((xId + 1)%width);
          *(indicesPtr++) = (yId+1) * width + ((xId + 1)%width);
          *(indicesPtr++) = (yId+1) * width + xId;
        }
      }
      else
        if (g_shapeType == SoBufferedShape::TRIANGLES)
        {
          // Strip mode
          if (g_stripPrimitive)
          {
            if ((i&1) == 1) 
              *(indicesPtr++) = yId * width + xId;
            else
              *(indicesPtr++) = (yId+1) * width + xId;

            // Adds last two indices and the primitive restart value
            if ((i % (stepDivider * width)) == stepDivider * width - 1)
            {
              *(indicesPtr++) = (yId+1) * width + ((xId + 1)%width);
              *(indicesPtr++) = yId * width + ((xId + 1)%width);
              *(indicesPtr++) = PRIMITIVE_RESTART_VALUE;
            }
          }
          // No strip here
          else
          {
            if ((i&1) == 0)
            {
              *(indicesPtr++) = yId * width + xId;
              *(indicesPtr++) = yId * width + ((xId + 1)%width);
              *(indicesPtr++) = (yId+1) * width + xId;
            }
            else
            {
              *(indicesPtr++) = yId * width + xId + 1;
              *(indicesPtr++) = (yId+1) * width + ((xId + 1)%width);
              *(indicesPtr++) = (yId+1) * width + xId;
            }
          }
        }
        else
          if (g_shapeType == SoBufferedShape::LINES)
          {
            // Strip mode, uses triangle stip in line mode ...
            if (g_stripPrimitive)
            {
              if ((i&1) == 1) 
                *(indicesPtr++) = yId * width + xId;
              else
                *(indicesPtr++) = (yId+1) * width + xId;

              // Adds last two indices and the primitive restart value
              if ((i % (stepDivider * width)) == stepDivider * width - 1)
              {
                *(indicesPtr++) = (yId+1) * width + ((xId + 1)%width);
                *(indicesPtr++) = yId * width + ((xId + 1)%width);
                *(indicesPtr++) = PRIMITIVE_RESTART_VALUE;
              }
            }
            // No strip here
            else
            {
              if ((i&1) == 0)
              {
                *(indicesPtr++) = yId * width + xId;
                *(indicesPtr++) = yId * width + ((xId + 1)%width);
              }
              else
              {
                *(indicesPtr++) = yId * width + xId;
                *(indicesPtr++) = (yId+1) * width + xId;
              }
            }
          }
          else
            if (g_shapeType == SoBufferedShape::POINTS)
            {
              *(indicesPtr++) = yId * width + xId;
            }
    }
  }
  // 3 - Else duplicate vertex attributes.
  else
  {
    for(int i = 0; i < primitivesCount; ++i)
    {
      int xId = (i/stepDivider) % width;
      int yId = (i/stepDivider) / width;

      if (g_shapeType == SoBufferedShape::QUADS)
      {
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[12*i+c] = vertices[3*(yId * width + xId)+c];
              colorsPtr[12*i+c] = colors[3*(yId * width + xId)+c];
              normalsPtr[12*i+c] = normals[3*(yId * width + xId)+c];
            }

            for (int c = 0; c < 3; c++)
            {
              verticesPtr[12*i+c+3] = vertices[3*(yId * width + ((xId + 1)%width))+c];
              colorsPtr[12*i+c+3] = colors[3*(yId * width + ((xId + 1)%width))+c];
              normalsPtr[12*i+c+3] = normals[3*(yId * width + ((xId + 1)%width))+c];
            }

            for (int c = 0; c < 3; c++)
            {
              verticesPtr[12*i+c+6] = vertices[3*((yId+1) * width + ((xId + 1)%width))+c];
              colorsPtr[12*i+c+6] = colors[3*((yId+1) * width + ((xId + 1)%width))+c];
              normalsPtr[12*i+c+6] = normals[3*((yId+1) * width + ((xId + 1)%width))+c];
            }

            for (int c = 0; c < 3; c++)
            {
              verticesPtr[12*i+c+9] = vertices[3*((yId+1) * width + xId)+c];
              colorsPtr[12*i+c+9] = colors[3*((yId+1) * width + xId)+c];
              normalsPtr[12*i+c+9] = normals[3*((yId+1) * width + xId)+c];
            }
      }
      else
        if (g_shapeType == SoBufferedShape::TRIANGLES)
        {
          if ((i&1) == 0)
          {
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[9*i+c] = vertices[3*(yId * width + xId)+c];
              colorsPtr[9*i+c] = colors[3*(yId * width + xId)+c];
              normalsPtr[9*i+c] = normals[3*(yId * width + xId)+c];
            }
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[9*i+c+3] = vertices[3*(yId * width + ((xId + 1)%width))+c];
              colorsPtr[9*i+c+3] = colors[3*(yId * width + ((xId + 1)%width))+c];
              normalsPtr[9*i+c+3] = normals[3*(yId * width + ((xId + 1)%width))+c];
            }
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[9*i+c+6] = vertices[3*((yId+1) * width + xId)+c];
              colorsPtr[9*i+c+6] = colors[3*((yId+1) * width + xId)+c];
              normalsPtr[9*i+c+6] = normals[3*((yId+1) * width + xId)+c];
            }
          }
          else
          {
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[9*i+c] = vertices[3*(yId * width + xId + 1)+c];
              colorsPtr[9*i+c] = colors[3*(yId * width + xId + 1)+c];
              normalsPtr[9*i+c] = normals[3*(yId * width + xId + 1)+c];
            }
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[9*i+c+3] = vertices[3*((yId+1) * width + ((xId + 1)%width))+c];
              colorsPtr[9*i+c+3] = colors[3*((yId+1) * width + ((xId + 1)%width))+c];
              normalsPtr[9*i+c+3] = normals[3*((yId+1) * width + ((xId + 1)%width))+c];
            }
            for (int c = 0; c < 3; c++)
            {
              verticesPtr[9*i+c+6] = vertices[3*((yId+1) * width + xId)+c];
              colorsPtr[9*i+c+6] = colors[3*((yId+1) * width + xId)+c];
              normalsPtr[9*i+c+6] = normals[3*((yId+1) * width + xId)+c];
            }
          }
        }
        else
          if (g_shapeType == SoBufferedShape::LINES)
          {
            if ((i&1) == 0)
            {
              for (int c = 0; c < 3; c++)
              {
                verticesPtr[6*i+c] = vertices[3*(yId * width + xId)+c];
                colorsPtr[6*i+c] = colors[3*(yId * width + xId)+c];
                normalsPtr[6*i+c] = normals[3*(yId * width + xId)+c];
              }
              for (int c = 0; c < 3; c++)
              {
                verticesPtr[6*i+c+3] = vertices[3*(yId * width + ((xId + 1)%width))+c];
                colorsPtr[6*i+c+3] = colors[3*(yId * width + ((xId + 1)%width))+c];
                normalsPtr[6*i+c+3] = normals[3*(yId * width + ((xId + 1)%width))+c];
              }
            }
            else
            {
              for (int c = 0; c < 3; c++)
              {
                verticesPtr[6*i+c] = vertices[3*(yId * width + xId)+c];
                colorsPtr[6*i+c] = colors[3*(yId * width + xId)+c];
                normalsPtr[6*i+c] = normals[3*(yId * width + xId)+c];
              }
              for (int c = 0; c < 3; c++)
              {
                verticesPtr[6*i+c+3] = vertices[3*((yId+1) * width + xId)+c];
                colorsPtr[6*i+c+3] = colors[3*((yId+1) * width + xId)+c];
                normalsPtr[6*i+c+3] = normals[3*((yId+1) * width + xId)+c];
              }
            }
          }
          else
            if (g_shapeType == SoBufferedShape::POINTS)
            {
              for (int c = 0; c < 3; c++)
              {
                verticesPtr[3*i+c] = vertices[3*(yId * width + xId)+c];
                colorsPtr[3*i+c] = colors[3*(yId * width + xId)+c];
                normalsPtr[3*i+c] = normals[3*(yId * width + xId)+c];
              }
            }
    }
  }


  // 4 - Housekeeping
  delete[] vertices;
  delete[] normals;
  delete[] colors;

  verticesBufferObject->unmap();
  colorsBufferObject->unmap();
  normalsBufferObject->unmap();

  if (indicesBuffer)
    indicesBuffer->unmap();

  g_bufferedShape->vertexBuffer = verticesBufferObject;
  g_bufferedShape->colorBuffer = colorsBufferObject;
  g_bufferedShape->normalBuffer = normalsBufferObject;
  g_bufferedShape->shapeType = g_shapeType;
  g_bufferedShape->primitiveRestartEnabled = FALSE;
  
  if (g_indexedShape)
  {
    g_bufferedShape->numVertices = indexBufferSize;

    if (g_stripPrimitive)
    {
      g_bufferedShape->primitiveRestartEnabled = TRUE;
      g_bufferedShape->primitiveRestartValue = PRIMITIVE_RESTART_VALUE;

      if (g_shapeType == SoBufferedShape::TRIANGLES)
        g_bufferedShape->shapeType = SoBufferedShape::TRIANGLE_STRIP;
      if (g_shapeType == SoBufferedShape::QUADS)
        g_bufferedShape->shapeType = SoBufferedShape::QUAD_STRIP;
      if (g_shapeType == SoBufferedShape::LINES)
        g_bufferedShape->shapeType = SoBufferedShape::LINE_STRIP;
    }
  }
  else
    g_bufferedShape->numVertices = primitivesCount*primitiveSize;

  g_examinerViewer->unbindNormalContext();

  // Compute again the new shape
  performGeneratePrimitives();
}

//------------------------------------------------------------------------------
void performGeneratePrimitives()
{
  SoVertexProperty* vpFaceSet = NULL;
  SoVertexProperty* vpLineSet = NULL;
  SoVertexProperty* vpPointSet = NULL;

  if (!g_generatedPrimitivesShape)
  {
    g_generatedPrimitivesShape = new SoFaceSet();
    g_generatedPrimitivesPointSet = new SoPointSet;
    g_generatedPrimitivesLineSet = new SoLineSet;

    SoSeparator* shapeSep = new SoSeparator();
    SoTranslation* translation = new SoTranslation();
    translation->translation.setValue(3.f, 0.f, 0.f);

    g_generatePrimitivesSwitch = new SoSwitch;
    g_generatePrimitivesSwitch->whichChild = SO_SWITCH_ALL;
    g_generatePrimitivesSwitch->addChild(shapeSep);

    shapeSep->addChild(translation);
    shapeSep->addChild(g_generatedPrimitivesShape);
    shapeSep->addChild(g_generatedPrimitivesLineSet);
    shapeSep->addChild(g_generatedPrimitivesPointSet);
    g_sceneGraph->addChild(g_generatePrimitivesSwitch);

    vpFaceSet = new SoVertexProperty();
    vpFaceSet->normalBinding = SoVertexProperty::PER_VERTEX;
    g_generatedPrimitivesShape->vertexProperty = vpFaceSet;

    vpLineSet = new SoVertexProperty();
    vpLineSet->normalBinding = SoVertexProperty::PER_VERTEX;
    g_generatedPrimitivesLineSet->vertexProperty = vpLineSet;

    vpPointSet = new SoVertexProperty();
    vpPointSet->normalBinding = SoVertexProperty::PER_VERTEX;
    g_generatedPrimitivesPointSet->vertexProperty = vpPointSet;
  }
  else
  {
    vpFaceSet = (SoVertexProperty*)g_generatedPrimitivesShape->vertexProperty.getValue();
    vpLineSet = (SoVertexProperty*)g_generatedPrimitivesLineSet->vertexProperty.getValue();
    vpPointSet = (SoVertexProperty*)g_generatedPrimitivesPointSet->vertexProperty.getValue();
  }

  if (!g_generatePrimitivesDisplayed)
    return;

  vpFaceSet->vertex.setNum(0);
  vpFaceSet->orderedRGBA.setNum(0);
  vpFaceSet->normal.setNum(0);
  vpLineSet->vertex.setNum(0);
  vpLineSet->orderedRGBA.setNum(0);
  vpLineSet->normal.setNum(0);
  vpPointSet->vertex.setNum(0);
  vpPointSet->orderedRGBA.setNum(0);
  vpPointSet->normal.setNum(0);

  // Generates the list of triangles/point/lines and put them in a separate faceset/pointSet/linset
  SoCallbackAction* cb = new SoCallbackAction();
  cb->addTriangleCallback(SoShape::getClassTypeId(), triangleCallback, NULL);
  cb->addLineSegmentCallback(SoShape::getClassTypeId(), lineCallback, NULL);
  cb->addPointCallback(SoShape::getClassTypeId(), pointCallback, NULL);
  // as each callback setup vertex and normal of each generatedPrimitive<Shape> for each primitive
  // we can speedup the process by preallocating the array to avoid continous resize when growing
  {
    SoGetPrimitiveCountAction gpca;
    gpca.apply( g_bufferedShape );

    g_triangleVertex.clear();
    int numTris   = gpca.getTriangleCount();
    if ( numTris>0)
    {
      g_triangleVertex.reserve(numTris*3);
      g_triangleNormal.reserve(numTris*3);
    }

    g_pointVertex.clear();
    int numPoints = gpca.getPointCount();
    if (numPoints>0)
    {
      g_pointVertex.reserve(numPoints);
      g_pointNormal.reserve(numPoints);
    }

    g_lineVertex.clear();
    int numLines  = gpca.getLineCount();
    if( numLines )
    {
      g_lineVertex.reserve(numLines);
      g_lineNormal.reserve(numLines);
    }
  }

  cb->apply(g_bufferedShape);

  // update g_generatedPrimitives with the arrays generated by the callbacks
  if ( g_triangleVertex.size()>0 )
  {
    ((SoVertexProperty*)g_generatedPrimitivesShape->vertexProperty.getValue())->vertex.setValuesPointer(static_cast<int>(g_triangleVertex.size()), &g_triangleVertex[0]);
	((SoVertexProperty*)g_generatedPrimitivesShape->vertexProperty.getValue())->normal.setValuesPointer(static_cast<int>(g_triangleNormal.size()), &g_triangleNormal[0]);
  }
  else
  {
    ((SoVertexProperty*)g_generatedPrimitivesShape->vertexProperty.getValue())->vertex.setNum(0);
    ((SoVertexProperty*)g_generatedPrimitivesShape->vertexProperty.getValue())->normal.setNum(0);
  }

  if ( g_pointVertex.size()>0 )
  {
	((SoVertexProperty*)g_generatedPrimitivesPointSet->vertexProperty.getValue())->vertex.setValuesPointer(static_cast<int>(g_pointVertex.size()), &g_pointVertex[0]);
	((SoVertexProperty*)g_generatedPrimitivesPointSet->vertexProperty.getValue())->normal.setValuesPointer(static_cast<int>(g_pointNormal.size()), &g_pointNormal[0]);
  }
  else
  {
    ((SoVertexProperty*)g_generatedPrimitivesPointSet->vertexProperty.getValue())->vertex.setNum(0);
    ((SoVertexProperty*)g_generatedPrimitivesPointSet->vertexProperty.getValue())->normal.setNum(0);
  }

  if( g_lineVertex.size()>0)
  {
	((SoVertexProperty*)g_generatedPrimitivesLineSet->vertexProperty.getValue())->vertex.setValuesPointer(static_cast<int>(g_lineVertex.size()), &g_lineVertex[0]);
	((SoVertexProperty*)g_generatedPrimitivesLineSet->vertexProperty.getValue())->normal.setValuesPointer(static_cast<int>(g_lineNormal.size()), &g_lineNormal[0]);
  }
  else
  {
    ((SoVertexProperty*)g_generatedPrimitivesLineSet->vertexProperty.getValue())->vertex.setNum(0);
    ((SoVertexProperty*)g_generatedPrimitivesLineSet->vertexProperty.getValue())->normal.setNum(0);
  }

  // Triangles
  int count = vpFaceSet->vertex.getNum()/3;
  g_generatedPrimitivesShape->numVertices.setNum(count);

  for (int i = 0; i < count; i++)
    g_generatedPrimitivesShape->numVertices.set1Value(i, 3);

  // Lines
  count = vpLineSet->vertex.getNum()/2;
  g_generatedPrimitivesLineSet->numVertices.setNum(count);

  for (int i = 0; i < count; i++)
    g_generatedPrimitivesLineSet->numVertices.set1Value(i, 2);

  // Points
  count = vpPointSet->vertex.getNum();
  g_generatedPrimitivesPointSet->numPoints = count;

  delete cb;
}

//------------------------------------------------------------------------------
int 
main( int , char** )
{
  // We init OIV
  Widget mainWindow = SoXt::init("BufferedShapePicking");
  if (mainWindow == NULL)
    exit (1);

  SoDialogViz::init();

  // The gui
  Widget mainWidget = buildInterface( mainWindow );

  if (!mainWidget)
  {
    SoDialogViz::finish();
    SoXt::finish();
    exit (1);
  }


  // The background
  SoGradientBackground* background = new SoGradientBackground;

  SoShapeHints* shapeHints = new SoShapeHints;

  // The default material
  SoMaterial* material = new SoMaterial;
  {
    material->diffuseColor.setValue( 1.0, 1.0, 1.0 );
    material->ambientColor.setValue( 1.0, 1.0, 1.0 );
  }

  SoPerspectiveCamera* camera = new SoPerspectiveCamera();
  camera->orientation.setValue(SbRotation(SbVec3f(1.0f, 0.0f, 0.0f), 3.141592f/2.0f));

  g_examinerViewer = new SoXtExaminerViewer( mainWidget );

  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), eventCallback ,g_examinerViewer  );
  eventCB->addEventCallback( SoMouseButtonEvent::getClassTypeId(), eventCallback ,g_examinerViewer  );

  // Let's display the viewer
  {
    g_examinerViewer->setTitle("BufferedShape picking");

    g_examinerViewer->show();

    // Ok here we go...
    SoXt::show( mainWindow );
  }

  // Assembly
  g_sceneGraph = new SoSeparator;
  g_sceneGraph->ref();
  {
    g_sceneGraph->addChild(camera);
    g_sceneGraph->addChild(eventCB);

    g_sceneGraph->addChild(background);

    g_sceneGraph->addChild(material);

    g_sceneGraph->addChild(shapeHints);

    createGeometry();

    // Add a scene graph for the picking geometry...
    // - We don't want it to be pickable.
    SoPickStyle* pickStyle = new SoPickStyle;
    pickStyle->style = SoPickStyle::UNPICKABLE;

    g_sceneGraph->addChild(pickStyle);

    // - We don't want lighting.
    SoLightModel* lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    g_sceneGraph->addChild(lightModel);

    // - We want a large point...
    SoDrawStyle* drawStyle = new SoDrawStyle();
    drawStyle->pointSize = 5.0f;
    g_sceneGraph->addChild(drawStyle);

    SoDepthOffset*  depthOffset = NULL;
    SoVertexProperty* vp = NULL;
    
    // - The geometry must be shift to be always on top...
    depthOffset = new SoDepthOffset();
    depthOffset->offset = 0.1f;
    g_sceneGraph->addChild(depthOffset);
    
    // - Add an empty faceset...
    g_pickedGeometry = new SoFaceSet();
    vp = new SoVertexProperty();
    g_pickedGeometry->vertexProperty = vp;

    g_sceneGraph->addChild(g_pickedGeometry);

    g_pickedLinesGeometry = new SoLineSet();
    vp = new SoVertexProperty();
    g_pickedLinesGeometry->vertexProperty = vp;
    
    g_sceneGraph->addChild(g_pickedLinesGeometry);

    depthOffset = new SoDepthOffset();
    depthOffset->offset = 0.1f;
    g_sceneGraph->addChild(depthOffset);

    // - Add an empty point set to display the picked point.
    g_pickedPoint = new SoPointSet();
    vp = new SoVertexProperty();
    g_pickedPoint->vertexProperty = vp;

    g_sceneGraph->addChild(g_pickedPoint);
  }

  g_examinerViewer->setSceneGraph( g_sceneGraph );

  g_examinerViewer->getCamera()->viewAll( g_sceneGraph, g_examinerViewer->getViewportRegion() );

  SoXt::mainLoop();

  g_sceneGraph->unref();

  delete g_examinerViewer;
  
  releaseInterface();

  SoDialogViz::finish();

  SoXt::finish();

  return 0;
}

//------------------------------------------------------------------------------

