
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoGpuBufferObject.h>

#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderObject.h>
#include <Inventor/nodes/SoVertexAttribFeedback.h>
#include <Inventor/nodes/SoTranslation.h>

const char* g_vertexShaderStr =
  "#version 410 core\n"
	"//!oiv_include <Inventor/oivShapeAttribute.h>\n"
	"\n"
	"void main()\n"
	"{\n"
	"   // Directly provide the vertex coordinates to the geometry shader, no transformation here\n"
	"	gl_Position = OivVertexPosition();\n"
	"}\n";

const char* g_geometryShaderStr =
  "#version 410 core\n"
  "\n"
  "layout(triangles) in;\n"
  "layout(triangle_strip, max_vertices = 16) out;\n"
	"\n"
  "out vec3 outVertex;\n"
  "out vec3 outColor;\n"
  "\n"
  "layout(triangles) in;\n"
  "layout(triangle_strip, max_vertices = 16) out;\n"
	"void main()\n"
	"{\n"
  "   vec4 b = (gl_in[0].gl_Position + gl_in[1].gl_Position + gl_in[2].gl_Position) / vec4(3.0);\n"
	"   // Triangle 1\n"
  "   gl_Position = gl_in[0].gl_Position;\n"
  "   outVertex = gl_in[0].gl_Position.xyz;\n"
  "   outColor = vec3(1.0, 0.0, 0.0);\n"
	"   EmitVertex();\n"
  "   gl_Position = gl_in[1].gl_Position;\n"
  "   outVertex = gl_in[1].gl_Position.xyz;\n"
  "   outColor = vec3(1.0, 0.0, 0.0);\n"
	"   EmitVertex();\n"
  "   gl_Position = b;\n"
  "   outVertex = b.xyz;\n"
  "   outColor = vec3(0.0, 1.0, 0.0);\n"
	"   EmitVertex();\n"
	"   EndPrimitive();\n"
	"   // Triangle 2\n"
  "   gl_Position = gl_in[1].gl_Position;\n"
  "   outVertex = gl_in[1].gl_Position.xyz;\n"
  "   outColor = vec3(1.0, 0.0, 0.0);\n"
	"   EmitVertex();\n"
  "   gl_Position = gl_in[2].gl_Position;\n"
  "   outVertex = gl_in[2].gl_Position.xyz;\n"
  "   outColor = vec3(1.0, 0.0, 0.0);\n"
	"   EmitVertex();\n"
  "   gl_Position = b;\n"
  "   outVertex = b.xyz;\n"
  "   outColor = vec3(0.0, 1.0, 0.0);\n"
	"   EmitVertex();\n"
	"   EndPrimitive();\n"
	"   // Triangle 3\n"
  "   gl_Position = gl_in[2].gl_Position;\n"
  "   outVertex = gl_in[2].gl_Position.xyz;\n"
  "   outColor = vec3(1.0, 0.0, 0.0);\n"
	"   EmitVertex();\n"
  "   gl_Position = gl_in[0].gl_Position;\n"
  "   outVertex = gl_in[0].gl_Position.xyz;\n"
  "   outColor = vec3(1.0, 0.0, 0.0);\n"
	"   EmitVertex();\n"
  "   gl_Position = b;\n"
  "   outVertex = b.xyz;\n"
  "   outColor = vec3(0.0, 1.0, 0.0);\n"
	"   EmitVertex();\n"
	"   EndPrimitive();\n"
	"}\n";


//------------------------------------------------------------------------------
int 
main( int , char** )
{
  // We init OIV
  Widget mainWindow = SoXt::init("InterleavedVertexAttribFeedback");
  if (mainWindow == NULL)
    exit (1);

   // Build and initialize the fake Inventor render area widget
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(mainWindow);
  viewer->show();
  SoXt::show( mainWindow );


  // We need a simple scene graph with a shape to provide some data to the geometry shader.
  // This scene graph will be added to the vertex attributes feedback list of children and to the viewer scene graph for reference.
  SoFaceSet* simpleFaceSet = new SoFaceSet;
  SoVertexProperty* vp = new SoVertexProperty;

  // - First triangle
  vp->vertex.set1Value(0, SbVec3f(0.f, 0.f, 0.f));
  vp->vertex.set1Value(1, SbVec3f(1.f, 0.f, 0.f));
  vp->vertex.set1Value(2, SbVec3f(0.f, 1.f, 0.f));

  // - Second triangle
  vp->vertex.set1Value(3, SbVec3f(1.f, 0.f, 0.f));
  vp->vertex.set1Value(4, SbVec3f(1.f, 1.f, 0.f));
  vp->vertex.set1Value(5, SbVec3f(0.f, 1.f, 0.f));

  simpleFaceSet->vertexProperty = vp;
  simpleFaceSet->numVertices.set1Value(0, 3);
  simpleFaceSet->numVertices.set1Value(1, 3);

  /********************************************************************************************/
  // Create the vertex attributes feedback scene graph in order to generate the a shape from the
  // the previously declared simpleFaceSet.
  /********************************************************************************************/

  // We need the OpenGL context from the viewer to create the bufferobjects
  viewer->bindNormalContext();

  // This is the root of the whole vertex attributes feedback scene graph
  SoSeparator* feedbackScene = new SoSeparator;
  feedbackScene->ref();
  
  // We need to add a vertex shader and a geometry shader, they must be added before the 
  // vertex attributes feedback node because the node needs an access to the shader to re-link it,
  // so it uses the shaders on the state.
  // - We don't care about any fragment shader because we disable the rasterizer.
  SoVertexShader* vertexShader = new SoVertexShader();
  SoGeometryShader* geometryShader = new SoGeometryShader();
  
  vertexShader->sourceProgram = g_vertexShaderStr;
  vertexShader->sourceType = SoShaderObject::GLSL_PROGRAM;
  geometryShader->sourceProgram = g_geometryShaderStr;
  geometryShader->sourceType = SoShaderObject::GLSL_PROGRAM;
  
  SoShaderProgram* shaderProgram = new SoShaderProgram;
  shaderProgram->shaderObject.set1Value(0, vertexShader);
  shaderProgram->shaderObject.set1Value(1, geometryShader);
  shaderProgram->geometryInputType = SoShaderProgram::TRIANGLES_INPUT;
  shaderProgram->geometryOutputType = SoShaderProgram::TRIANGLE_STRIP_OUTPUT;
  shaderProgram->maxGeometryOutputVertices = 16;

  // We need a buffer shape which will receive the data from the vertex attributes feedback.
  // - First we create the buffer object which 
  SoGpuBufferObject* tfOutputBufferObject = new SoGpuBufferObject(SoGpuBufferObject::DYNAMIC);

  // feedback node
  SoVertexAttribFeedback* feedbackNode = new SoVertexAttribFeedback;
  feedbackNode->autoResizeBuffers = TRUE;
  feedbackNode->queryGeneratedPrimitivesCount = TRUE;
  feedbackNode->buffersMode = SoVertexAttribFeedback::INTERLEAVED;

  int primitiveSize = /* 3 position values of 3 components of size float */ 3 * 3 * sizeof(float) + 
    /* 3 color values of 3 components of size float */ 3 * 3 * sizeof(float);

  feedbackNode->registerFeedback("outVertex", tfOutputBufferObject, primitiveSize);
  feedbackNode->registerFeedback("outColor", NULL);

  feedbackScene->addChild(shaderProgram);
  feedbackScene->addChild(feedbackNode);
  feedbackNode->addChild(simpleFaceSet);

  /********************************************************************************************/
  // Compute the new geometry using vertex attributes feedback.
  /********************************************************************************************/
  viewer->getGLRenderAction()->apply(feedbackScene);

  int generatedPrimitives = feedbackNode->getGeneratedPrimitivesCount();

  printf("Generated primitives: %d\n", generatedPrimitives);

  /********************************************************************************************/
  // Create a scene graph to display both the input shape and the resulting shape.
  /********************************************************************************************/

  // We create a scene graph to show the result in the viewer
  SoSeparator* viewerScene = new SoSeparator;
  viewerScene->ref();

  // Disable lighting
  SoLightModel* lightModel = new SoLightModel();
  lightModel->model = SoLightModel::BASE_COLOR;

  viewerScene->addChild(lightModel);

  // - We want to render using lines, so we add an SoDrawStyle
  SoDrawStyle* style = new SoDrawStyle;
  style->style = SoDrawStyle::LINES;

  viewerScene->addChild(style);

  // - Add the simple faceset used as input of the vertex attributes feedback node
  viewerScene->addChild(simpleFaceSet);
  
  // - Then the buffered shape
  SoBufferedShape* bufferedShape = new SoBufferedShape;
  bufferedShape->vertexBuffer = tfOutputBufferObject;
  bufferedShape->colorBuffer = tfOutputBufferObject;
  bufferedShape->vertexStride = 2 * 3 * sizeof(float);
  bufferedShape->colorOffset = 3 * sizeof(float);
  bufferedShape->colorStride = 2 * 3 * sizeof(float);

  // - "generatedPrimitives" triangles of 3 vertices each...
  bufferedShape->numVertices = generatedPrimitives * 3;

  // - Add a simple translation to show the two shapes at different positions
  SoTranslation* translation = new SoTranslation;
  translation->translation.setValue(2.f, 0.f, 0.f);
  viewerScene->addChild(translation);
  viewerScene->addChild(bufferedShape);
 

  /********************************************************************************************/
  // Vertex attributes feedback house keeping.
  // We cannot unref it before because the faceset node is shared both by the viewer
  // and by the vertex attributes feedback scene graph.
  /********************************************************************************************/

  // We can release the vertex attributes feedback scene graph
  feedbackScene->unref();

  // We don't need the context anymore
  viewer->unbindNormalContext();

  /********************************************************************************************/
  // Demo main loop.
  /********************************************************************************************/

  viewer->setSceneGraph(viewerScene);

  SoXt::mainLoop();

  // Housekeeping
  viewerScene->unref();
  delete viewer;
  SoXt::finish();

  return 0;
}

//------------------------------------------------------------------------------


