/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Damien DALLA ROSA (Jul 2008)
**=======================================================================*/

#include <Inventor/SbLinear.h>

#include <Inventor/devices/SoGpuBufferObject.h>
#include <Inventor/devices/SoCpuBufferObject.h>

#include <Inventor/image/SbRasterImage.h>
#include <Inventor/image/SoRasterReaderSet.h>
#include <Inventor/image/SoRasterImageFile.h>

#include <Inventor/nodes/SoBufferedShape.h>

#include "geometry.h"

SoBufferObject* g_heightMapBufferObject = NULL;
int g_heightMapWidth = 0;
int g_heightMapHeight = 0;
int g_heightMapComponentsCount = 0;

SoBufferObject* g_colorMapBufferObject = NULL;
int g_colorMapWidth = 0;
int g_colorMapHeight = 0;
int g_colorMapComponentsCount = 0;

#define USE_CPU_BUFFERS 0

//------------------------------------------------------------------------------
bool readRasterImage( std::string filename, SoBufferObject* bufferObject, int& width, int& height, int& nc, bool convertToGray );

float computeElevation( int i, int j, int width, int height, int heightMapWidth, int heightMapHeight, unsigned char* heightMapBufferPtr );

//------------------------------------------------------------------------------
SoBufferedShape* buildGeometry( std::string heightMapFile, std::string colorMap, float zScale, bool computeNormals )
{
  SoBufferedShape* bufferedShape = new SoBufferedShape;

  SO_UNREF_RESET(g_heightMapBufferObject);

  SO_UNREF_RESET(g_colorMapBufferObject);

  /*******************************************/
  /*    1 - We read the images               */
  /*******************************************/

  g_heightMapBufferObject = new SoCpuBufferObject;
  g_heightMapBufferObject->ref();

  g_colorMapBufferObject = new SoCpuBufferObject;
  g_colorMapBufferObject->ref();

  readRasterImage( heightMapFile, g_heightMapBufferObject, g_heightMapWidth, g_heightMapHeight, g_heightMapComponentsCount, true );
  readRasterImage( colorMap, g_colorMapBufferObject, g_colorMapWidth, g_colorMapHeight, g_colorMapComponentsCount, false );

  updateGeometry( bufferedShape, zScale, computeNormals );

  return bufferedShape;
}


//------------------------------------------------------------------------------
bool updateGeometry( SoBufferedShape* bufferedShape, float zScale, bool computeNormals )
{
  if ( !g_heightMapBufferObject || !g_colorMapBufferObject )
    return false;

  int scale = 1;

  int width = scale * g_heightMapWidth;
  int height = scale * g_heightMapHeight;

  /*******************************************/
  /*    2 - We check the input data          */
  /*******************************************/

  SoCpuBufferObject* heightMapBuffer = new SoCpuBufferObject;
  heightMapBuffer->ref();

  SoCpuBufferObject* colorMapBuffer=new SoCpuBufferObject;
  colorMapBuffer->ref();

  g_heightMapBufferObject->map( heightMapBuffer, SoBufferObject::READ_ONLY );
  g_colorMapBufferObject->map( colorMapBuffer, SoBufferObject::READ_ONLY );

  unsigned char* heightMapBufferPtr = (unsigned char*)heightMapBuffer->map( SoBufferObject::READ_ONLY );
  unsigned char* colorMapBufferPtr = (unsigned char*)colorMapBuffer->map( SoBufferObject::READ_ONLY );

  // We check the size
  int dataSize = width * height;

  // The last point in a row and the last row don't produce any triangles.
  int indicesCount = (width - 1) * (height - 1);

  /*******************************************/
  /*    3 - Build the geometry               */
  /*******************************************/

  // We build the geometry
  SoBufferedShape* shape = bufferedShape;
  {
    // We create a buffer for the indices. 
    // - The size must be indicesCount * sizeof( unsigned int ) * 2 * 3 (3 indices per triangle, 2 triangles for each vertex)
#if USE_CPU_BUFFERS
    SoCpuBufferObject* indices = (SoCpuBufferObject *)shape->indexBuffer.getValue();
#else
    SoGpuBufferObject* indices = (SoGpuBufferObject *)shape->indexBuffer.getValue();
#endif
    
    if ( !indices )
    {
#if USE_CPU_BUFFERS
      indices = new SoCpuBufferObject();
#else
      indices = new SoGpuBufferObject( SoGpuBufferObject::DYNAMIC );
#endif
      shape->indexBuffer = indices;
    }

    indices->setSize( indicesCount * sizeof( unsigned int ) * 2 * 3 );

    // We create the buffer for the vertices: 3 float components.
#if USE_CPU_BUFFERS
    SoCpuBufferObject* vertices = (SoCpuBufferObject *)shape->vertexBuffer.getValue();
#else
    SoGpuBufferObject* vertices = (SoGpuBufferObject *)shape->vertexBuffer.getValue();
#endif
    
    if ( !vertices )
    {
#if USE_CPU_BUFFERS
      vertices = new SoCpuBufferObject();
#else
      vertices = new SoGpuBufferObject( SoGpuBufferObject::DYNAMIC );
#endif
      shape->vertexBuffer = vertices;
    }

    vertices->setSize( 3 * dataSize * sizeof( float ) );

    // We create the buffer for the colors: 3 float components.
#if USE_CPU_BUFFERS
    SoCpuBufferObject* colors = (SoCpuBufferObject *)shape->colorBuffer.getValue();
#else
    SoGpuBufferObject* colors = (SoGpuBufferObject *)shape->colorBuffer.getValue();
#endif

    if ( !colors )
    {
#if USE_CPU_BUFFERS
      colors = new SoCpuBufferObject();
#else
      colors = new SoGpuBufferObject( SoGpuBufferObject::DYNAMIC );
#endif
      shape->colorBuffer = colors;
    }

    colors->setSize( 3 * dataSize * sizeof( float ) );

    // We create the buffer for the tex coords: 2 float components.
#if USE_CPU_BUFFERS
    SoCpuBufferObject* texCoords = (SoCpuBufferObject *)shape->texCoordsBuffer[0];
#else
    SoGpuBufferObject* texCoords = (SoGpuBufferObject *)shape->texCoordsBuffer[0];
#endif
    
    if ( !texCoords )
    {
#if USE_CPU_BUFFERS
      texCoords = new SoCpuBufferObject();
#else
      texCoords = new SoGpuBufferObject( SoGpuBufferObject::DYNAMIC );
#endif


      shape->texCoordsBuffer = texCoords;
    }

    texCoords->setSize( 2 * dataSize * sizeof( float ) );

    // We create the buffer for the normals: 3 float components.
#if USE_CPU_BUFFERS
    SoCpuBufferObject* normals = (SoCpuBufferObject *)shape->normalBuffer.getValue();
#else
    SoGpuBufferObject* normals = (SoGpuBufferObject *)shape->normalBuffer.getValue();
#endif

    if ( computeNormals )
    {
      if ( !normals )
      {
#if USE_CPU_BUFFERS
        normals = new SoCpuBufferObject();
#else
        normals = new SoGpuBufferObject( SoGpuBufferObject::DYNAMIC );
#endif
        shape->normalBuffer = normals;
      }

      normals->setSize( 3 * dataSize * sizeof( float ) );      
    }
    else
      shape->useNormalsGenerator = TRUE;

    // The shape has a collection of triangles so we set the mode to triangles.
    shape->shapeType = SoBufferedShape::TRIANGLES;

    // Ok we setup the shape...
    shape->numVertices.set1Value( 0, indicesCount * 3 );
    shape->numVertices.set1Value( 1, indicesCount * 3 );

    shape->indexType = SbDataType::UNSIGNED_INT32;

    // We must map the vertices buffer, in order to write it.

    float* verticesPtr = (float *)vertices->map( SoBufferObject::READ_WRITE );

    // We must map the normals buffer, in order to write it.
    float* normalsPtr = NULL;

    if ( computeNormals ) 
    {
      normalsPtr = (float *)normals->map( SoBufferObject::READ_WRITE );
    }

    // We must map the indices buffer, in order to write it.
    unsigned int* indicesPtr = (unsigned int *)indices->map( SoBufferObject::READ_WRITE );

    // We must map the colors buffer, in order to write it.
    float* colorsPtr = (float *)colors->map( SoBufferObject::READ_WRITE );

    // We must map the colors buffer, in order to write it.
    float* texCoordsPtr = (float *)texCoords->map( SoBufferObject::READ_WRITE );

    // We must compute the image ratio for the vertices coordinates.
    float ratio = float( g_heightMapHeight ) / float( g_heightMapWidth );

    float maxHeight = -100000.0;
    float minHeight = 100000.0;

    // We compute the vertices (positions, colors and texCoords) and the indices to reduce the number of vertices
    for ( int j = 0; j<height; j++ )
      for ( int i = 0; i<width; i++ )
      {
        // We compute the index in the coordinates buffers.
        int index = ( j * width + i);

        // We compute the index in the indices buffer.
        int indicesIndex = (j*(width-1) + i);

        float computedElevation = computeElevation( i, j, width, height, g_heightMapWidth, g_heightMapHeight, heightMapBufferPtr ); 
          
        float elevation = zScale * computedElevation;

        if ( elevation > maxHeight )
          maxHeight = elevation;

        if ( elevation < minHeight )
          minHeight = elevation;

        // We compute the vertex coordinates.
        //  - For the elevation:
        //    - We convert to the range [0; 1.0]
        //    - We multiply by 8.0, because the everest is almost 8kms
        //    - We devide by the earch length at the equator to get a normalized elevation.
        //    - We multiply by 50.0 in order to see the g_heightMap in the viewer.
        verticesPtr[ 3 * index ] = float(i) / float(width);
        verticesPtr[ 3 * index + 1 ] = ratio * (float(j) / float(height));
        verticesPtr[ 3 * index + 2 ] = float( elevation );

        texCoordsPtr[ 2 * index ] = float(i) / float(width);
        texCoordsPtr[ 2 * index + 1 ] = float(j) / float(height);

         // If we need some indices we compute them.
        if ( indicesPtr )
        {
          if ( i < width - 1 )
            if ( j < height - 1 )
            {
              indicesPtr[ 2 * 3 * indicesIndex + 0 ] = index;
              indicesPtr[ 2 * 3 * indicesIndex + 1 ] = index + width + 1;
              indicesPtr[ 2 * 3 * indicesIndex + 2 ] = index + width;

              indicesPtr[ 2 * 3 * indicesIndex + 3 ] = index;
              indicesPtr[ 2 * 3 * indicesIndex + 4 ] = index + 1;
              indicesPtr[ 2 * 3 * indicesIndex + 5 ] = index + width + 1;
            }
        }
      }


    // We compute the colors
    for ( int j = 0; j<height; j++ )
      for ( int i = 0; i<width; i++ )
      {
       int index = (j*width + i);

       float elevation = verticesPtr[ 3 * index + 2 ];

       float heightInMap = float( g_colorMapWidth ) * (( elevation - minHeight ) / maxHeight );
       int colormapIndex = int( heightInMap );
       
       // For the colors we directly take the colors from the colors image file.
        if ( g_colorMapComponentsCount == 3 )
        {
          colorsPtr[ 3 * index + 0 ] = float( colorMapBufferPtr[ 3 * colormapIndex + 0 ] ) / 255.f;
          colorsPtr[ 3 * index + 1 ] = float( colorMapBufferPtr[ 3 * colormapIndex + 1 ] ) / 255.f;
          colorsPtr[ 3 * index + 2 ] = float( colorMapBufferPtr[ 3 * colormapIndex + 2 ] ) / 255.f;
        }
        else
        {
          colorsPtr[ 3 * index + 0 ] = float( colorMapBufferPtr[ colormapIndex ] ) / 255.f;
          colorsPtr[ 3 * index + 1 ] = float( colorMapBufferPtr[ colormapIndex ] ) / 255.f;
          colorsPtr[ 3 * index + 2 ] = float( colorMapBufferPtr[ colormapIndex ] ) / 255.f;
        }
      }


    // We compute the normals
      if ( normalsPtr )
      {
        for ( int j = 1; j<height-1; j++ )
          for ( int i = 1; i<width-1; i++ )
          {
            int index = (j*width + i);

            float x = verticesPtr[ 3 * index ];
            float y = verticesPtr[ 3 * index + 1 ];
            float z = verticesPtr[ 3 * index + 2 ];

            SbVec3f vecs[4];

            vecs[0] = SbVec3f( verticesPtr[ 3 * (index+1) ] - x, verticesPtr[ 3 * (index+1) + 1 ] - y, 
              verticesPtr[ 3 * (index+1) + 2 ] - z );

            vecs[1] = SbVec3f( verticesPtr[ 3 * (index+width) ] - x, verticesPtr[ 3 * (index+width) + 1 ] - y, 
              verticesPtr[ 3 * (index+width) + 2 ] - z );

            vecs[2] = SbVec3f( verticesPtr[ 3 * (index-1) ] - x, verticesPtr[ 3 * (index-1) + 1 ] - y, 
              verticesPtr[ 3 * (index-1) + 2 ] - z );

            vecs[3] = SbVec3f( verticesPtr[ 3 * (index-width) ] - x, verticesPtr[ 3 * (index-width) + 1 ] - y, 
              verticesPtr[ 3 * (index-width) + 2 ] - z );

            for ( int v = 0; v<4; v++ )
              vecs[v].normalize();

            SbVec3f normalsVec[4];

            // Cross product for all the directions.
            normalsVec[0] = vecs[0].cross( vecs[1] );
            normalsVec[1] = vecs[1].cross( vecs[2] );
            normalsVec[2] = vecs[2].cross( vecs[3] );
            normalsVec[3] = vecs[3].cross( vecs[0] );

            // We take an average of the normals.
            float xn = (normalsVec[0][0] + normalsVec[1][0] + normalsVec[2][0] + normalsVec[3][0]) / 4.f;
            float yn = (normalsVec[0][1] + normalsVec[1][1] + normalsVec[2][1] + normalsVec[3][1]) / 4.f;
            float zn = (normalsVec[0][2] + normalsVec[1][2] + normalsVec[2][2] + normalsVec[3][2]) / 4.f;

            // We can write the normals buffer.
            normalsPtr[ 3 * index + 0 ] = xn;
            normalsPtr[ 3 * index + 1 ] = yn;
            normalsPtr[ 3 * index + 2 ] = zn;
          }
      }

    indices->unmap();
    vertices->unmap();

    if ( normalsPtr )
    {
      normals->unmap();
    }

    texCoords->unmap();
    colors->unmap();
    }

  /*******************************************/
  /*    4 - We clean everything              */
  /*******************************************/

  g_heightMapBufferObject->unmap( heightMapBuffer );
  g_colorMapBufferObject->unmap( colorMapBuffer );
  colorMapBuffer->unmap();
  heightMapBuffer->unmap();
  colorMapBuffer->unref();
  heightMapBuffer->unref();

  return true;
}


//------------------------------------------------------------------------------
bool readRasterImage( std::string filename, SoBufferObject* bufferObject, int& width, int& height, int& nc, bool convertToGray )
{
  SbRasterImage* rasterImage = new SbRasterImage;

  SoRasterImageFile* rasterFile = new SoRasterImageFile;
  {
    rasterFile->setFileName( filename.c_str() );
  }

  SoRasterImageRW* imageReader = NULL;
  SbString ext;
  
  if ( filename.find( ".png" ) != std::string::npos )
    ext = SbString("PNG");
  if ( filename.find( ".tif" ) != std::string::npos )
    ext = SbString("TIG");
  if ( filename.find( ".jpg" ) != std::string::npos )
    ext = SbString("JPG");

  imageReader = SoRasterReaderSet::getReader( rasterFile, ext );

  if ( !imageReader )
  {
    printf("[D] Error imageReader is NULL\n");
    delete rasterFile;
    return false;
  }


  if ( !imageReader->open( rasterFile, SoRasterImageRW::OPEN_READ ) ) 
  {
    delete rasterFile;
    return false;
  }

  if (!imageReader->read( rasterImage )) 
  {
    imageReader->close();
    delete rasterFile;
    return false;
  }
  
  width = rasterImage->getSize()[0];
  height = rasterImage->getSize()[1];
  int rasterNc = (rasterImage->getComponents() == SbRasterImage::LUMINANCE)? 1 : 3;
  nc = rasterNc;

  if ( convertToGray )
    nc = 1;
  
  SoRef<SoBufferObject> bufferImage = rasterImage->getBufferObject();
  SoRef<SoCpuBufferObject> cpuBufferImage = new SoCpuBufferObject();
  bufferImage->map(cpuBufferImage.ptr(), SoBufferObject::READ_WRITE);
  unsigned char* ptr = (unsigned char*) cpuBufferImage->map(SoBufferObject::READ_ONLY);

  int size = width * height * nc;
  bufferObject->setSize( size );

  SoRef<SoCpuBufferObject> cpuBuffer = new SoCpuBufferObject;
  
  bufferObject->map( cpuBuffer.ptr(), SoBufferObject::SET );
  {
    unsigned char* destPtr = (unsigned char*)cpuBuffer->map( SoBufferObject::SET );

    if ( convertToGray )
    {
      for ( int i = 0; i<width*height; i++ )
        destPtr[ i ] = ptr[ rasterNc*i ];
    }
    else
      memcpy( destPtr, ptr, size );
  }
  cpuBuffer->unmap();
  bufferObject->unmap( cpuBuffer.ptr() );

  cpuBufferImage->unmap();
  bufferImage->unmap( cpuBufferImage.ptr() );

  delete rasterFile;
  delete rasterImage;

  return true;
}

//------------------------------------------------------------------------------
float computeElevation( int i, int j, int width, int height, int heightMapWidth, int heightMapHeight, unsigned char* heightMapBufferPtr )
{
  float computedElevation = 0.f;
  int scaleW = width / heightMapWidth;
  int scaleH = height / heightMapHeight;
  
  if ( (scaleW == 2) && (scaleH == 2 ) )
  {
    if ( !(i&1) && !(j&1) )
      computedElevation =(float( heightMapBufferPtr[ (j>>1) * g_heightMapWidth + (i >> 1) ] ) / 256.f);
    else
      if ( (i&1) && !(j&1) )
      {
        float a =(float( heightMapBufferPtr[ (j>>1) * g_heightMapWidth + (i >> 1) ] ) / 256.f);
        float b =(float( heightMapBufferPtr[ (j>>1) * g_heightMapWidth + (i >> 1) + 1 ] ) / 256.f);
        computedElevation = (a + b) * 0.5f;
      }
      else
        if ( !(i&1) && (j&1) )
        {
          float a =(float( heightMapBufferPtr[ (j>>1) * g_heightMapWidth + (i >> 1) ] ) / 256.f);
          float b =(float( heightMapBufferPtr[ ((j>>1) + 1 ) * g_heightMapWidth + (i >> 1) ] ) / 256.f);
          computedElevation = (a + b) * 0.5f;
        }
        else
          if ( (i&1) && (j&1) )
          {
            float a =(float( heightMapBufferPtr[ (j>>1) * g_heightMapWidth + (i >> 1) ] ) / 256.f);
            float b =(float( heightMapBufferPtr[ ((j>>1) + 1 ) * g_heightMapWidth + (i >> 1) ] ) / 256.f);
            float c =(float( heightMapBufferPtr[ (j>>1) * g_heightMapWidth + (i >> 1) + 1 ] ) / 256.f);
            float d =(float( heightMapBufferPtr[ ((j>>1) + 1 ) * g_heightMapWidth + (i >> 1) + 1 ] ) / 256.f);
            computedElevation = (a + b + c + d) * 0.25f;
          }
  }
  else
  {
    computedElevation =(float( heightMapBufferPtr[ j * g_heightMapWidth + i ] ) / 256.f);
  }

  return computedElevation;
}


