
#include <Inventor/SoDB.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoLightModel.h> 
#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoGradientBackground.h> 
#include <Inventor/nodes/SoShapeHints.h> 
#include <Inventor/nodes/SoEventCallback.h>

#include <Inventor/nodes/SoPointLight.h> 

#include <Inventor/actions/SoWriteAction.h> 

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/helpers/SbFileHelper.h>

#include "geometry.h"

class MyAuditorClass;

//------------------------------------------------------------------------------

// Un-comment this define to dump the geometry to an iv file before displaying it.
//#define DUMP_GEOMETRY_TO_FILE

#define DEFAULT_SCALE_VALUE 0.25

// Scene graph root separator needed for tunning in the event callback
SoSeparator* g_sceneGraph = NULL;

SoBBox* g_bboxNode = NULL;

SoXtExaminerViewer* g_examinerViewer = NULL;

SoBufferedShape* g_bufferedShaped = NULL;

SoSwitch* g_aerialTextureSwitch = NULL;

SoTexture2* g_aerialTexture = NULL;

SoDeviceContext* g_oglContext = NULL;

SoDialogLabel* g_sizeLabel = NULL;

SoShapeHints* g_shapeHints = NULL;

double g_geometryScale = DEFAULT_SCALE_VALUE;

SoTopLevelDialog* g_topLevelDialog = NULL;
MyAuditorClass* g_myAuditor = NULL;
SbString GuiFileName= SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/BufferObjects/SoBufferedShape/data/gui.iv");
const char *g_guiFilename = GuiFileName.toLatin1();

//------------------------------------------------------------------------------


/** This function reads the content of an IV file and returns the scene graph. */
SoSeparator* readIvFile( const char* filename );

/** This function writes the content of a scene graph to an IV file. */
void writeIvFile( const char* filename, SoSeparator* ivSceneGraph );



//------------------------------------------------------------------------------
void updateSizeLabel()
{
  if ( !g_sizeLabel )
    return;

  int primitivesCount = 0;
  int shapesNum = g_bufferedShaped->numVertices.getNum();

  for ( int i = 0; i< shapesNum; i++ )
     primitivesCount += g_bufferedShaped->numVertices[ i ];

  size_t verticesCount = g_bufferedShaped->vertexBuffer.getValue()->getSize() / (3*4);

  char buffer[512];

  sprintf( buffer, "Vertices: %llu, Indices: %d", (unsigned long long)verticesCount, primitivesCount );

  g_sizeLabel->label.setValue( buffer );
}

//------------------------------------------------------------------------------
void 
eventCallback( void* /*userData*/, SoEventCallback *eventCB ) 
{
  // We need the buffered shape node.
  //SoXtExaminerViewer* examiner = (SoXtExaminerViewer*)userData;
  
  const SoEvent *event = eventCB->getEvent();

  // Nothing to do here
  if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW))
    return;

  // We simply recompute the geometry.
  if (SO_KEY_PRESS_EVENT(event, SPACE) )
  {
    // Ok we create our shape.
    g_examinerViewer->viewAll();
  }


  // We simply recompute the geometry.
  if (SO_KEY_PRESS_EVENT(event, RETURN) )
  {
    writeIvFile( "output.iv", g_sceneGraph );
  }

}


////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle inputs

class MyAuditorClass : public SoDialogAuditor
{
  void dialogRealSlider( SoDialogRealSlider* cpt );
  void dialogComboBox( SoDialogComboBox* cpt );
  void dialogPushButton( SoDialogPushButton* cpt );
};

void
MyAuditorClass::dialogRealSlider( SoDialogRealSlider* cpt)
{
  // Change slice number
  if ( cpt->auditorID.getValue() == "Scale" )
  {
    float value = cpt->value.getValue();

    g_bboxNode->boundingBox.setValue( 0.0f, 0.0f, 0.0f, 1.0f, 2.0f, value );

    if ( g_oglContext )
    {
      g_oglContext->bind();
      updateGeometry( g_bufferedShaped, value, false );
      g_examinerViewer->viewAll();
      g_oglContext->unbind();   

      updateSizeLabel();
    }
  }

  if ( cpt->auditorID.getValue() == "CreaseAngle" )
  {
    float value = cpt->value.getValue();

    g_shapeHints->creaseAngle = value;
  }
}

void
MyAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
  // Select attribute to compute
  if (cpt->auditorID.getValue() == "display")
  {
    int value = cpt->selectedItem.getValue();

    if ( value == 0 )
    {
      g_aerialTextureSwitch->whichChild = SO_SWITCH_NONE;
    }
    else
    {
      g_aerialTextureSwitch->whichChild = SO_SWITCH_ALL;
      
      if ( value == 2 )
        g_aerialTexture->model = SoTexture::BLEND;
      else
        g_aerialTexture->model = SoTexture::REPLACE;

    }
  }
}

void
MyAuditorClass::dialogPushButton(SoDialogPushButton*)
{
}


//------------------------------------------------------------------------------
Widget
buildInterface( Widget window )
{
  SoInput myInput;
  if (! myInput.openFile( g_guiFilename ))
    return NULL;

  SoRef<SoGroup> myGroup = SoDB::readAll( &myInput );
  if (! myGroup.ptr() )
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );
  g_topLevelDialog->ref();

  g_myAuditor = new MyAuditorClass;
  g_topLevelDialog->addAuditor( g_myAuditor );

  // setup slice slider range
  SoDialogRealSlider *slider1 =
    (SoDialogRealSlider *)g_topLevelDialog->searchForAuditorId( SbString("Scale") );
  if (slider1)
  {
    slider1->min = 0.01f;
    slider1->max = 5.0f;
    slider1->value = DEFAULT_SCALE_VALUE;
  }

  g_sizeLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("label_size") );

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, customNode != NULL );
  g_topLevelDialog->show();

  return customNode ? customNode->getWidget() : window;

}

void releaseInterface()
{
  g_topLevelDialog->unref();
  delete g_myAuditor;
}

//------------------------------------------------------------------------------
int 
main( int , char** argv)
{
  // We init OIV
  Widget mainWindow = SoXt::init(argv[0]);

  if (mainWindow == NULL)
    exit (1);

  SoDialogViz::init();

  // The gui
  Widget mainWidget = buildInterface( mainWindow );


  // The background
  SoGradientBackground* background = new SoGradientBackground;

  // The lighting node
  SoLightModel* light = new SoLightModel;
  {
    light->model = SoLightModel::PER_VERTEX_PHONG;
  }

  // We need a point light to test the shadows
  SoPointLight* pointLight = new SoPointLight;
  {  
    pointLight->color.setValue( 1.0, 1.0, 1.0 );
    pointLight->location.setValue( -1.0, 1.0, 1.0 );
    pointLight->intensity = 1.8f;
  }

  g_shapeHints = new SoShapeHints;

  // The default material
  SoMaterial* material = new SoMaterial;
  {
    material->diffuseColor.setValue( 1.0, 1.0, 1.0 );
    material->ambientColor.setValue( 1.0, 1.0, 1.0 );
  }
 
  g_bboxNode = new SoBBox;
  g_bboxNode->mode = SoBBox::USER_DEFINED;
  g_bboxNode->boundingBox.setValue( 0.0f, 0.0f, 0.0f, 1.0f, 2.0f, DEFAULT_SCALE_VALUE );

  g_aerialTextureSwitch = new SoSwitch;
  {
    g_aerialTexture = new SoTexture2;

    g_aerialTexture->filename = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/BufferObjects/SoBufferedShape/data/StHelenAerial.jpg");
    g_aerialTexture->model = SoTexture::REPLACE;

    g_aerialTextureSwitch->addChild( g_aerialTexture );

    g_aerialTextureSwitch->whichChild = SO_SWITCH_ALL;
  }

  g_examinerViewer = new SoXtExaminerViewer( mainWidget );

  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(),
			     eventCallback ,g_examinerViewer  );

  // Assembly
  SoSeparator* root = new SoSeparator;
  root->ref();
  {
   root->addChild( eventCB );
   root->addChild( background );
   root->addChild( light );
   root->addChild( pointLight );
  }

  g_sceneGraph = new SoSeparator;
  {
    g_sceneGraph->addChild( material );

    g_sceneGraph->addChild( g_shapeHints );

    g_sceneGraph->addChild( g_aerialTextureSwitch );

    //g_sceneGraph->addChild( g_bboxNode );
  }

  root->addChild(g_sceneGraph);

  // Build and initialize the Inventor render area widget
  {
    //examiner->setSize( SbVec2s( 512, 512 ) );
    g_examinerViewer->setDecoration( TRUE );
    g_examinerViewer->setFeedbackVisibility( FALSE );
    g_examinerViewer->setSceneGraph( root );
    g_examinerViewer->setTitle( "SoBufferedShape" );
    g_examinerViewer->setHeadlight( false );

    g_examinerViewer->show();

    // Ok here we go...
    SoXt::show( mainWindow );
  }

  SbString HelenFile=SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/BufferObjects/SoBufferedShape/data/StHelen.png");
  SbString Colormapfile = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/BufferObjects/SoBufferedShape/data/colormap.png");


  g_examinerViewer->bindNormalContext();
  g_bufferedShaped = 
    buildGeometry( HelenFile.getSString(),
		   Colormapfile.getSString() , 
		   DEFAULT_SCALE_VALUE, true );
  g_examinerViewer->unbindNormalContext();
  

  g_oglContext = g_bufferedShaped->vertexBuffer.getValue()->getContext();

  updateSizeLabel();

  g_sceneGraph->addChild( g_bufferedShaped );

  g_examinerViewer->getCamera()->viewAll( g_sceneGraph, g_examinerViewer->getViewportRegion() );

  SoXt::mainLoop();

  delete g_examinerViewer;

  releaseInterface();

  SoDialogViz::finish();

  SoXt::finish();

  return 0;
}


//------------------------------------------------------------------------------
SoSeparator* readIvFile( const char* filename )
{
  SoSeparator* ivSceneGraph = NULL;

  SoInput input;

  if ( input.openFile( filename ) ) 
  {
    ivSceneGraph = SoDB::readAll( &input ); 
    input.closeFile();
  }
  else
    SoDebugError::post( "", "readIvFile: Error loading model file!");

  return ivSceneGraph;
}


//------------------------------------------------------------------------------
void writeIvFile( const char* filename, SoSeparator* ivSceneGraph )
{
  SoOutput output;
  SoWriteAction writeAction( &output );

  if ( output.openFile( filename ) )
  {
    writeAction.apply( ivSceneGraph );
    output.closeFile();
  }
  else
    SoDebugError::post( "", "readIvFile: Error writing to the IV file!");
}

//------------------------------------------------------------------------------

