
#include <Inventor/SoDB.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoLightModel.h> 
#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoGpuBufferObject.h>
#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoGradientBackground.h> 
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPointLight.h> 
#include <Inventor/nodes/SoShaderObject.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShapeHints.h> 
#include <Inventor/nodes/SoVertexAttribFeedback.h>
#include <Inventor/nodes/SoTranslation.h>

#include <Inventor/actions/SoWriteAction.h> 

#include <DialogViz/SoDialogVizAll.h>

//------------------------------------------------------------------------------
// Default values for the geometry shader

#define DEFAULT_WIDTH 100
#define DEFAULT_HEIGHT 100

#define DEFAULT_A1 1.f
#define DEFAULT_B1 1.f
#define DEFAULT_M1 7.f
#define DEFAULT_N11 20.45f
#define DEFAULT_N12 -0.96f
#define DEFAULT_N13 -0.33f

#define DEFAULT_A2 1.f
#define DEFAULT_B2 1.f
#define DEFAULT_M2 6.f
#define DEFAULT_N21 4.46f
#define DEFAULT_N22 -3.54f
#define DEFAULT_N23 0.52f

//------------------------------------------------------------------------------
// GUI global data
SbString g_guiFilename = 
  "$OIVHOME/examples/source/Inventor/Features/BufferObjects/VertexAttribFeedback/gui.iv";

SoTopLevelDialog* g_topLevelDialog = NULL;

SoDialogLabel* g_sizeLabel = NULL;
SoDialogEditText* g_a1EditText = NULL;
SoDialogEditText* g_b1EditText = NULL;
SoDialogEditText* g_m1EditText = NULL;
SoDialogEditText* g_n11EditText = NULL;
SoDialogEditText* g_n12EditText = NULL;
SoDialogEditText* g_n13EditText = NULL;

SoDialogEditText* g_a2EditText = NULL;
SoDialogEditText* g_b2EditText = NULL;
SoDialogEditText* g_m2EditText = NULL;
SoDialogEditText* g_n21EditText = NULL;
SoDialogEditText* g_n22EditText = NULL;
SoDialogEditText* g_n23EditText = NULL;

SoDialogEditText* g_widthEditText = NULL;
SoDialogEditText* g_heightEditText = NULL;

SoDialogPushButton* g_applyPushButton = NULL;

//------------------------------------------------------------------------------
// Scene Graph global data modified on gui events.

// Scene graph root separator needed for tunning in the event callback
SoSeparator* g_sceneGraph = NULL;

SoBBox* g_bboxNode = NULL;
SoXtExaminerViewer* g_examinerViewer = NULL;

SoBufferedShape* g_bufferedShape = NULL;

SoBufferedShape* g_inputBufferedShape = NULL;
SoSeparator* g_feedbackSceneGraph;
SoVertexAttribFeedback* g_feedback = NULL;

SoGeometryShader* g_geometryShader = NULL;
SoShaderParameter2f* g_ab1Parameter = NULL;
SoShaderParameter2f* g_ab2Parameter = NULL;
SoShaderParameter4f* g_param1Parameter = NULL;
SoShaderParameter4f* g_param2Parameter = NULL;
SoShaderParameter1i* g_widthParameter = NULL;
SoShaderParameter1i* g_heightParameter = NULL;

//------------------------------------------------------------------------------


/** This function reads the content of an IV file and returns the scene graph. */
SoSeparator* readIvFile( const char* filename );

/** This function writes the content of a scene graph to an IV file. */
void writeIvFile( const char* filename, SoSeparator* ivSceneGraph );

SoShaderProgram* loadShader(int width, int height, float a1, float a2, float b1, float b2,
                            float m1, float m2,
                            float n11, float n12, float n13, 
                            float n21, float n22, float n23);

SoShaderProgram* loadShader(int width, int height, float a, float b, float m,
                            float n1, float n2, float n3);

//------------------------------------------------------------------------------
void updateSizeLabel()
{
  char buffer[4096];

  int userDefinedSize = g_widthParameter->value.getValue() * g_heightParameter->value.getValue() * 2;
  // Two triangles per point with two static shapes
  int allSize = DEFAULT_WIDTH * DEFAULT_HEIGHT * 2 * 2 + userDefinedSize; 

  sprintf(buffer, "Triangles generated: All: %d, UserDefine only: %d", allSize, userDefinedSize);

  g_sizeLabel->label.setValue( buffer );
}

//------------------------------------------------------------------------------
void 
eventCallback( void * /*userData*/, SoEventCallback *eventCB ) 
{
  const SoEvent *event = eventCB->getEvent();

  // Nothing to do here
  if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW))
    return;

  // View all.
  if (SO_KEY_PRESS_EVENT(event, SPACE) )
  {
    // Ok we create our shape.
    g_examinerViewer->viewAll();
  }
  // We write the content of the scene graph to a file.
  if (SO_KEY_PRESS_EVENT(event, RETURN) )
  {
    writeIvFile( "output.iv", g_sceneGraph );
  }
}


////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle inputs

class MyAuditorClass : public SoDialogAuditor
{
  void dialogPushButton( SoDialogPushButton* cpt );
};

void
MyAuditorClass::dialogPushButton(SoDialogPushButton* pushButton)
{
  if (pushButton == g_applyPushButton)
  {
    g_ab1Parameter->value = SbVec2f((float)atof(g_a1EditText->editText.getValue().getString()),
      (float)atof(g_b1EditText->editText.getValue().getString()));
    g_ab2Parameter->value = SbVec2f((float)atof(g_a2EditText->editText.getValue().getString()),
      (float)atof(g_b2EditText->editText.getValue().getString()));
    g_param1Parameter->value = SbVec4f((float)atof(g_m1EditText->editText.getValue().getString()),
      (float)atof(g_n11EditText->editText.getValue().getString()), 
      (float)atof(g_n12EditText->editText.getValue().getString()),
      (float)atof(g_n13EditText->editText.getValue().getString()));
    g_param2Parameter->value = SbVec4f((float)atof(g_m2EditText->editText.getValue().getString()),
      (float)atof(g_n21EditText->editText.getValue().getString()), 
      (float)atof(g_n22EditText->editText.getValue().getString()),
      (float)atof(g_n23EditText->editText.getValue().getString()));
    
    int w = atoi(g_widthEditText->editText.getValue().getString());
    int h = atoi(g_heightEditText->editText.getValue().getString());

    g_widthParameter->value = w;
    g_heightParameter->value = h;
    g_inputBufferedShape->numVertices = w * h;

    g_examinerViewer->bindNormalContext();
    g_examinerViewer->getGLRenderAction()->apply(g_feedbackSceneGraph);
    g_examinerViewer->unbindNormalContext();
    g_bufferedShape->numVertices = 3 * g_feedback->getGeneratedPrimitivesCount();

    g_examinerViewer->scheduleRedraw();

    updateSizeLabel();
  }
}


//------------------------------------------------------------------------------
Widget
buildInterface( Widget window )
{
  SoInput myInput;
  if (! myInput.openFile( g_guiFilename ))
    return NULL;

  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup)
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  MyAuditorClass *myAuditor = new MyAuditorClass;
  g_topLevelDialog->addAuditor( myAuditor );

  g_sizeLabel = (SoDialogLabel*)g_topLevelDialog->searchForAuditorId( SbString("label_size") );
  g_a1EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("a1") );
  g_b1EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("b1") );
  g_m1EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("m1") );
  g_n11EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("n11") );
  g_n12EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("n12") );
  g_n13EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("n13") );

  g_a2EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("a2") );
  g_b2EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("b2") );
  g_m2EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("m2") );
  g_n21EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("n21") );
  g_n22EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("n22") );
  g_n23EditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("n23") );

  SbString value;
  g_a1EditText->editText.setValue(value.setNum(DEFAULT_A1));
  g_b1EditText->editText.setValue(value.setNum(DEFAULT_B1));
  g_m1EditText->editText.setValue(value.setNum(DEFAULT_M1));
  g_n11EditText->editText.setValue(value.setNum(DEFAULT_N11));
  g_n12EditText->editText.setValue(value.setNum(DEFAULT_N12));
  g_n13EditText->editText.setValue(value.setNum(DEFAULT_N13));
  g_a2EditText->editText.setValue(value.setNum(DEFAULT_A2));
  g_b2EditText->editText.setValue(value.setNum(DEFAULT_B2));
  g_m2EditText->editText.setValue(value.setNum(DEFAULT_M2));
  g_n21EditText->editText.setValue(value.setNum(DEFAULT_N21));
  g_n22EditText->editText.setValue(value.setNum(DEFAULT_N22));
  g_n23EditText->editText.setValue(value.setNum(DEFAULT_N23));

  g_widthEditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("width") );
  g_heightEditText = (SoDialogEditText*)g_topLevelDialog->searchForAuditorId( SbString("height") );

  g_widthEditText->editText.setValue(value.setNum(DEFAULT_WIDTH));
  g_heightEditText->editText.setValue(value.setNum(DEFAULT_HEIGHT));


  g_applyPushButton = 
    (SoDialogPushButton*)g_topLevelDialog->searchForAuditorId( SbString("UserDefinedApplyButton") );

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, customNode != NULL );
  g_topLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}

void createObject(int width, int height, float a, float b, float m,
                            float n1, float n2, float n3)
{
  SoShaderProgram* shaderProgram = loadShader(width, height, a, b, m, n1, n2, n3);

  SoGpuBufferObject* verticesBufferObject = new SoGpuBufferObject(SoGpuBufferObject::STATIC);
  SoGpuBufferObject* colorsBufferObject = new SoGpuBufferObject(SoGpuBufferObject::STATIC);

  g_feedback->registerFeedback("vPosition", verticesBufferObject);
  // 3 vertices (TRIANGLE) with 4 values (RGBA)
  g_feedback->registerFeedback("vColor", colorsBufferObject, 3 * 4 * sizeof(float));
  
  g_feedbackSceneGraph->insertChild(shaderProgram, 0);

  g_examinerViewer->getGLRenderAction()->apply(g_feedbackSceneGraph);

  g_feedbackSceneGraph->removeChild(shaderProgram);

  g_feedback->unregisterFeedback("vPosition");
  g_feedback->unregisterFeedback("vColor");

  SoBufferedShape* bufferedShape = new SoBufferedShape();
  bufferedShape->vertexBuffer = verticesBufferObject;
  bufferedShape->colorBuffer = colorsBufferObject;
  bufferedShape->colorComponentsCount = 4;
  // - One triangle has three vertices
  bufferedShape->numVertices = 3 * g_feedback->getGeneratedPrimitivesCount();
  bufferedShape->useNormalsGenerator = FALSE;
  g_sceneGraph->addChild(bufferedShape);
}

void createGeometry()
{
  g_examinerViewer->bindNormalContext();
  
  // Fake buffer for the transform feedback input
  SoGpuBufferObject* fakeData = new SoGpuBufferObject(SoGpuBufferObject::STATIC);
  fakeData->setSize(3*DEFAULT_WIDTH * DEFAULT_HEIGHT*sizeof(float));

  // Base geometry for transform feedback
  g_inputBufferedShape = new SoBufferedShape();
  g_inputBufferedShape->vertexBuffer = fakeData;
  g_inputBufferedShape->numVertices = DEFAULT_WIDTH * DEFAULT_HEIGHT;
  g_inputBufferedShape->shapeType = SoBufferedShape::POINTS;
  g_inputBufferedShape->useNormalsGenerator = FALSE;

  // Transform feedback scene graph
  g_feedbackSceneGraph = new SoSeparator();
  g_feedbackSceneGraph->ref();
  g_feedback = new SoVertexAttribFeedback;
  g_feedback->autoResizeBuffers = TRUE;
  g_feedback->queryGeneratedPrimitivesCount = TRUE;
  g_feedback->addChild(g_inputBufferedShape);
  g_feedbackSceneGraph->addChild(g_feedback);

  // Object 1
  createObject(DEFAULT_WIDTH, DEFAULT_HEIGHT, 1.f, 1.f, 7.f, 0.2f, 1.7f, 1.7f);
  SoTranslation* translation = new SoTranslation();
  translation->translation.setValue(3.f, 0.f, 0.f);
  g_sceneGraph->addChild( translation );

  // Object 2
  createObject(DEFAULT_WIDTH, DEFAULT_HEIGHT, 1.f, 1.f, 5.2f, 0.04f, 1.7f, 1.7f);
  g_sceneGraph->addChild( translation );

  // Object 3
  SoShaderProgram* shaderProgram = loadShader(DEFAULT_WIDTH, DEFAULT_HEIGHT, DEFAULT_A1, DEFAULT_A2, 
    DEFAULT_B1, DEFAULT_B2, DEFAULT_M1, DEFAULT_M2, 
    DEFAULT_N11, DEFAULT_N12, DEFAULT_N13, 
    DEFAULT_N21, DEFAULT_N22, DEFAULT_N23);
  
  // We keep a track of this one, it will be used
  // for the user defined values.
  g_geometryShader = (SoGeometryShader*)shaderProgram->shaderObject[1];
  g_ab1Parameter = (SoShaderParameter2f*)g_geometryShader->parameter[0];
  g_ab2Parameter = (SoShaderParameter2f*)g_geometryShader->parameter[1];
  g_param1Parameter = (SoShaderParameter4f*)g_geometryShader->parameter[2];
  g_param2Parameter = (SoShaderParameter4f*)g_geometryShader->parameter[3];
  g_widthParameter = (SoShaderParameter1i*)g_geometryShader->parameter[4];
  g_heightParameter = (SoShaderParameter1i*)g_geometryShader->parameter[5];

  SoGpuBufferObject* verticesBufferObject = new SoGpuBufferObject(SoGpuBufferObject::STATIC);
  SoGpuBufferObject* colorsBufferObject = new SoGpuBufferObject(SoGpuBufferObject::STATIC);
   
  g_feedback->registerFeedback("vPosition", verticesBufferObject);
  g_feedback->registerFeedback("vColor", colorsBufferObject, 3 * 4 * sizeof(float));
  
  g_feedbackSceneGraph->insertChild(shaderProgram, 0);

  g_examinerViewer->getGLRenderAction()->apply(g_feedbackSceneGraph);

  g_bufferedShape = new SoBufferedShape();
  g_bufferedShape->vertexBuffer = verticesBufferObject;
  g_bufferedShape->colorBuffer = colorsBufferObject;
  g_bufferedShape->colorComponentsCount = 4;
  // - One triangle has three vertices
  g_bufferedShape->numVertices = 3 * g_feedback->getGeneratedPrimitivesCount();
  g_bufferedShape->useNormalsGenerator = FALSE;
  g_sceneGraph->addChild(g_bufferedShape);
  
  g_examinerViewer->unbindNormalContext();
}

//------------------------------------------------------------------------------
int 
main( int , char** )
{
  // We init OIV
  Widget mainWindow = SoXt::init("TransformFeedback");
  if (mainWindow == NULL)
    exit (1);

  SoDialogViz::init();

  // The gui
  Widget mainWidget = buildInterface( mainWindow );


  // The background
  SoGradientBackground* background = new SoGradientBackground;

  // The lighting node
  SoLightModel* light = new SoLightModel;
  {
    light->model = SoLightModel::BASE_COLOR;
  }

  // We need a point light to test the shadows
  SoPointLight* pointLight = new SoPointLight;
  {  
    pointLight->color.setValue( 1.0, 1.0, 1.0 );
    pointLight->location.setValue( -1.0, 1.0, 1.0 );
    pointLight->intensity = 1.8f;
  }

  SoShapeHints* shapeHints = new SoShapeHints;

  // The default material
  SoMaterial* material = new SoMaterial;
  {
    material->diffuseColor.setValue( 1.0, 1.0, 1.0 );
    material->ambientColor.setValue( 1.0, 1.0, 1.0 );
  }

  SoPerspectiveCamera* camera = new SoPerspectiveCamera();
  camera->orientation.setValue(SbRotation(SbVec3f(1.0f, 0.0f, 0.0f), 3.141592f/2.0f));
 
  g_bboxNode = new SoBBox;
  g_bboxNode->mode = SoBBox::USER_DEFINED;
  g_bboxNode->boundingBox.setValue( -1.0f, -1.0f, -5.0f, 7.0f, 1.0f, 3.0f );

  g_examinerViewer = new SoXtExaminerViewer( mainWidget );

  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(),
			     eventCallback ,g_examinerViewer  );

  // Assembly
  g_sceneGraph = new SoSeparator;
  {
    g_sceneGraph->addChild(camera);
    g_sceneGraph->addChild(eventCB);

    g_sceneGraph->addChild(background);
    g_sceneGraph->addChild(light);

    g_sceneGraph->addChild(pointLight);

    g_sceneGraph->addChild(material);

    g_sceneGraph->addChild(shapeHints);

    g_sceneGraph->addChild(g_bboxNode);
  }

  // Build and initialize the Inventor render area widget
  {
    //examiner->setSize( SbVec2s( 512, 512 ) );
    g_examinerViewer->setDecoration( TRUE );
    g_examinerViewer->setFeedbackVisibility( FALSE );
    g_examinerViewer->setSceneGraph( g_sceneGraph );
    g_examinerViewer->setTitle( "GPU Geometry" );
    g_examinerViewer->setHeadlight( false );

    g_examinerViewer->show();

    // Ok here we go...
    SoXt::show( mainWindow );
  }

  createGeometry();

  g_examinerViewer->getCamera()->viewAll( g_sceneGraph, g_examinerViewer->getViewportRegion() );

  updateSizeLabel();

  SoXt::mainLoop();

  delete g_examinerViewer;

  SoDialogViz::finish();

  SoXt::finish();

  return 0;
}

//------------------------------------------------------------------------------
SoShaderProgram* loadShader(int width, int height, float a, float b, float m,
                            float n1, float n2, float n3)
{
  return loadShader(width, height, a, a, b, b, m, m, n1, n2, n3, n1, n2, n3);
}

SoShaderProgram* loadShader(int width, int height, 
                            float a1, float a2, float b1, float b2,
                            float m1, float m2,
                            float n11, float n12, float n13, 
                            float n21, float n22, float n23)
{
  SoVertexShader* vertexShader = new SoVertexShader();
  SoGeometryShader* geometryShader = new SoGeometryShader();

  vertexShader->sourceProgram = 
    "$OIVHOME/examples/source/Inventor/Features/BufferObjects/VertexAttribFeedback/shaders/vertexShader.glsl";
  geometryShader->sourceProgram = 
    "$OIVHOME/examples/source/Inventor/Features/BufferObjects/VertexAttribFeedback/shaders/geometryShader.glsl";

  SoShaderProgram* shaderProgram = new SoShaderProgram;
  shaderProgram->shaderObject.set1Value(0, vertexShader);
  shaderProgram->shaderObject.set1Value(1, geometryShader);
  shaderProgram->geometryInputType = SoShaderProgram::POINTS_INPUT;
  shaderProgram->geometryOutputType = SoShaderProgram::TRIANGLE_STRIP_OUTPUT;
  shaderProgram->maxGeometryOutputVertices = 16;

  SoShaderParameter2f* ab1 = new SoShaderParameter2f;
  SoShaderParameter2f* ab2 = new SoShaderParameter2f;  

  SoShaderParameter1i* w = new SoShaderParameter1i;
  SoShaderParameter1i* h = new SoShaderParameter1i;  


  SoShaderParameter4f* param1 = new SoShaderParameter4f;
  SoShaderParameter4f* param2 = new SoShaderParameter4f;

  w->name = "width";
  w->value = width;

  h->name = "height";
  h->value = height;

  ab1->name = "ab1";
  ab1->value.setValue(SbVec2f(a1, b1));
  ab2->name = "ab2";
  ab2->value.setValue(SbVec2f(a2, b2));
  param1->name = "param1";
  param1->value.setValue(SbVec4f(m1, n11, n12, n13));
  param2->name = "param2";
  param2->value.setValue(SbVec4f(m2, n21, n22, n23));

  geometryShader->parameter.set1Value(0, ab1);
  geometryShader->parameter.set1Value(1, ab2);
  geometryShader->parameter.set1Value(2, param1);
  geometryShader->parameter.set1Value(3, param2);
  geometryShader->parameter.set1Value(4, w);
  geometryShader->parameter.set1Value(5, h);

  return shaderProgram;
}

//------------------------------------------------------------------------------
SoSeparator* readIvFile( const char* filename )
{
  SoSeparator* ivSceneGraph = NULL;

  SoInput input;

  if ( input.openFile( filename ) ) 
  {
    ivSceneGraph = SoDB::readAll( &input ); 
    input.closeFile();
  }
  else
    SoDebugError::post( "", "readIvFile: Error loading model file!");

  return ivSceneGraph;
}


//------------------------------------------------------------------------------
void writeIvFile( const char* filename, SoSeparator* ivSceneGraph )
{
  SoOutput output;
  SoWriteAction writeAction( &output );

  if ( output.openFile( filename ) )
  {
    writeAction.apply( ivSceneGraph );
    output.closeFile();
  }
  else
    SoDebugError::post( "", "readIvFile: Error writing to the IV file!");
}

//------------------------------------------------------------------------------

