/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Roberto Calabrese (Feb 2011)
**=======================================================================*/
/*-------------------------------------------------------------- */
#include <Inventor/nodes/SoCircularExtrusion.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <DialogViz/SoDialogVizAll.h>
#include <Inventor/actions/SoBoxHighlightRenderAction.h>
#include <Inventor/actions/SoSearchPathAction.h>
#include <Inventor/engines/SoCompose.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoShapeHints.h>

// forward declarations
SoSeparator* loadSceneGraph( );

// global constants
static const SbString g_absPathToDialogsRes( "$OIVHOME/examples/source/Inventor/Features/CircularExtrusion/" );
static const SbString g_UIDialogFName( "interface.iv" );
static SoCircularExtrusion *g_pCurrCircularExtrusion = 0;             // the current CircularExtrusion

// Auditor
/////////////////////////////////////////////////////////////
class SlidersAuditor : public SoDialogRealSliderAuditor
{
public:
  void dialogRealSlider( SoDialogRealSlider *cpt )
  {
    if ( g_pCurrCircularExtrusion ) 
    {
      if ( cpt->getName() == "PipeRadius" )
      {
        g_pCurrCircularExtrusion->radius = cpt->value.getValue();
      } else 
      {
        SbVec2f acts( g_pCurrCircularExtrusion->activeSection.getValue() );
        if ( cpt->getName() == "SlidStart" )
        {
          acts[0] = cpt->value.getValue();
        } else if ( cpt->getName() == "SlidEnd" ) 
        {
          acts[1] = cpt->value.getValue();
        }
        g_pCurrCircularExtrusion->activeSection = acts;
      }
    }
  }
};

/////////////////////////////////////////////////////////////
class UIManager 
{
public:
  Widget
  setUp( Widget mainWindowWidget, SoComplexity * pCompl  )
  {
    SoDialogCustom *myCusDlg = 0;
    SoDialogViz::init();
    SbString guiDialogFName = g_absPathToDialogsRes + g_UIDialogFName;
    SoTopLevelDialog *myTopDlg = (SoTopLevelDialog *)SoDialogViz::loadFromFile( guiDialogFName );
    if ( myTopDlg )
    {
      myCusDlg = (SoDialogCustom *)(myTopDlg)->searchForAuditorId( "DIALOG_CUSTOM" );
      SoDialogRealSlider *radiusPtr = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "ChangeRadiusAuditor" );
      radiusPtr->addAuditor( &m_auditor );
      SoDialogRealSlider *slStPtr = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "ChangeSliderAuditor1" );
      slStPtr->addAuditor( &m_auditor );
      SoDialogRealSlider *slEnPtr = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "ChangeSliderAuditor2" );
      slEnPtr->addAuditor( &m_auditor );
      SoDialogRealSlider * complexitySlider = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "ChangeSliderAuditor3" );
      pCompl->value.connectFrom( &complexitySlider->value );

      myTopDlg->buildDialog(mainWindowWidget, TRUE);
      myTopDlg->show();
    }

    return myCusDlg ? myCusDlg->getWidget() : mainWindowWidget;
  }

  SlidersAuditor       m_auditor;
};

int
main ( int /*argc*/, char **argv )
{
  Widget mainWindowWidget = SoXt::init( argv[0] );
  if ( mainWindowWidget == NULL )
    exit(1);

  UIManager uiManager;
  // Root node
  SoSeparator * root = new SoSeparator;
  root->ref();
  // Complexity node
  SoComplexity *pCompl = new SoComplexity;
  pCompl->type = SoComplexity::OBJECT_SPACE;
  pCompl->value = .5f;
  root->addChild( pCompl );
  // Hints node
  SoShapeHints *shapeHints = new SoShapeHints;
  shapeHints->creaseAngle.setValue( (float) ( M_PI/4 ) );
  root->addChild(shapeHints);
  // Material node
  SoMaterial *myMaterial = new SoMaterial;
  myMaterial->diffuseColor.setValue( 1.0f , 0.0f , 0.0f );
  root->addChild(myMaterial);

  // Set up UI and auditors
  Widget widg = uiManager.setUp( mainWindowWidget, pCompl );
  // Set up viewer
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer( widg );
  
  // Load scene graph
  SoSeparator *pScGraph = loadSceneGraph( );
  root->addChild( pScGraph );

  // Set up scene graph in viewer and show
  myViewer->setSceneGraph( root );
  myViewer->show();

  SoXt::show( mainWindowWidget );
  SoXt::mainLoop();

  delete myViewer;
  root->unref();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}

//////////////////////////////////////////// helper functions ////////////////////////////////////////////////////////////

// set current pipe-shape from scene graph
void
getPipeFromSceneGraph( SoSeparator * root )
{
  // search for PIPE_SHAPE
  root->ref();
  SoSearchPathAction* spaPipe = new SoSearchPathAction;
  spaPipe->setSearchString( "PIPE_SHAPE" );
  spaPipe->apply( root );
  if ( spaPipe->getPath() )
  {
    SoNode *pNode = spaPipe->getPath()->getTail();
    if ( pNode && pNode->isOfType( SoCircularExtrusion::getClassTypeId() ) )
      g_pCurrCircularExtrusion = (SoCircularExtrusion*)pNode;
    else
      g_pCurrCircularExtrusion = NULL;
  }
  else
    g_pCurrCircularExtrusion = NULL;

  delete spaPipe;
  spaPipe = NULL;
  root->unrefNoDelete();
}

SoSeparator* 
loadSceneGraph( ) 
{
  SbString fname( SbFileHelper::expandString( g_absPathToDialogsRes + "scenegraph.iv" ) );
  SoSeparator *retVal = NULL;
  // is it a .iv file? simple test
  SbString extStr = SbFileHelper::getExtension( fname );
  if ( extStr.lower() == "iv" )
  {
    SoInput in;
    SbString sceneGraphFname = fname;
    SbBool fileOpened = in.openFile( sceneGraphFname.getString(), true );
    if( fileOpened ) 
    {
      retVal = SoDB::readAll(&in);
      if ( retVal )
        getPipeFromSceneGraph( retVal );
      in.closeFile();
    }
  }
  return retVal;
}


