/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Fabien ARNAUD (Jul 1997)
**=======================================================================*/

//------------------------------------------------------------------------------
// Scenes and Objects
#define D_NUM_SCENES 2
const char* scenes_title[D_NUM_SCENES] = {
  "box.iv",
  "Barcelona.iv"};

#define D_NUM_OBJECTS 8
const char* objects_title[D_NUM_OBJECTS] = {
  "cube.iv",
  "canstick.iv",
  "ceilingLamp.iv",
  "chair.iv",
  "martini.iv",
  "plant.iv",
  "table.iv",
  "trackLights.iv"};

//------------------------------------------------------------------------------
// Inventor declarations
#include <Inventor/collision/SoCollisionManager.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/manips/SoCenterballManip.h>
#include <Inventor/manips/SoHandleBoxManip.h>
#include <Inventor/manips/SoJackManip.h>
#include <Inventor/manips/SoTabBoxManip.h>
#include <Inventor/manips/SoTrackballManip.h>
#include <Inventor/manips/SoTransformBoxManip.h>
#include <Inventor/sensors/SoAlarmSensor.h>
#include <Inventor/actions/SoWriteAction.h>

#include <DialogViz/SoDialogVizAll.h>

// Global objects
SoSeparator*        m_root;
SoSeparator*        m_scene;
SoSeparator*        m_object;
SoTransform*        m_object_transform;
SoTransformManip*   m_manip;
SoPath*             m_transform_path;
SoSwitch*           m_switch_material;
SoCollisionManager* m_collision_manager;

//------------------------------------------------------------------------------
// Functions declarations
void flashObject();
void unflashObject(void*, SoSensor*);
SoSeparator* readIvFile(const char* filename);
SoCollisionManager::Resp onCollision(void*, 
				     const SoCollidingPrimitive*, 
				     const SoCollidingPrimitive*);

// Dialog Auditors
class ManipulatorTypeAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    if ( m_manip != NULL )
    {
      m_manip->replaceManip( m_transform_path, NULL );
      m_manip->unref();
    }

    switch ( cpt->selectedItem.getValue() )
    {
    // Centerball manipulator
    case 1:
      m_manip = new SoCenterballManip;
      break;

    // Handlebox manipulator
    case 2:
      m_manip = new SoHandleBoxManip;
      break;

    // Jack manipulator
    case 3:
      m_manip = new SoJackManip;
      break;

    // TabBox manipulator
    case 4:
      m_manip = new SoTabBoxManip;
      break;

    // Trackball manipulator
    case 5:
      m_manip = new SoTrackballManip;
      break;

    // TransformBox manipulator
    case 6:
      m_manip = new SoTransformBoxManip;
      break;

    // No manipulator
    default:
      m_manip = NULL;
      m_collision_manager->setTransform( m_object_transform );
      return;
      break;
    }

    m_manip->setName( "m_manip" );
    m_manip->ref();
    m_manip->replaceNode( m_transform_path );
    m_collision_manager->setTransform( m_manip );
  }
};

class CollsionDetectionAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* cpt )
  {
    m_collision_manager->activate( cpt->state.getValue() );
  }
};

class GluingAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* cpt )
  {
    m_collision_manager->setGluing( cpt->state.getValue() );
  }
};

class GluingLevelAuditor : public SoDialogIntegerSliderAuditor
{
public:
  void dialogIntegerSlider( SoDialogIntegerSlider* cpt )
  {
    m_collision_manager->setGluingLevel( cpt->value.getValue() );
  }
};

class ObjectBBoxAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* cpt )
  {
    m_collision_manager->setObjectBBoxOnly( cpt->state.getValue() );
  }
};

class ObjectFileAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    char filename[1024];
    sprintf( filename, "$OIVHOME/examples/source/Inventor/Features/Collision/ObjectMoving/Objects/%s",
             objects_title[cpt->selectedItem.getValue()] );
    SoSeparator* sep = readIvFile( filename );
    if ( sep != NULL )
    {
      if ( m_manip != NULL )
        m_manip->replaceManip( m_transform_path, NULL );

      m_object->removeAllChildren();
      m_object->addChild( sep );

      if ( m_manip != NULL )
        m_manip->replaceNode( m_transform_path );
      else
        flashObject();
    }
  }
};

class SceneBBoxAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* cpt )
  {
    m_collision_manager->setSceneBBoxOnly( cpt->state.getValue() );
  }
};

class SceneFileAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    char filename[1024];
    sprintf( filename, "$OIVHOME/examples/source/Inventor/Features/Collision/ObjectMoving/Scenes/%s",
             scenes_title[cpt->selectedItem.getValue()] );
    SoSeparator* sep = readIvFile( filename );
    if ( sep != NULL )
    {
      m_scene->removeAllChildren();
      m_scene->addChild( sep );
    }
    else
    {
      flashObject();
    }
  }
};

class SceneCopyAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    SoSeparator* sep = new SoSeparator;
    SoTransform* t = new SoTransform;
    if ( m_manip == NULL )
    {
      t->translation.setValue( m_object_transform->translation.getValue() );
      t->rotation.setValue( m_object_transform->rotation.getValue() );
      t->scaleFactor.setValue( m_object_transform->scaleFactor.getValue() );
      t->scaleOrientation.setValue( m_object_transform->scaleOrientation.getValue() );
      t->center.setValue( m_object_transform->center.getValue() );
    }
    else
    {
      t->translation.setValue( m_manip->translation.getValue() );
      t->rotation.setValue( m_manip->rotation.getValue() );
      t->scaleFactor.setValue( m_manip->scaleFactor.getValue() );
      t->scaleOrientation.setValue( m_manip->scaleOrientation.getValue() );
      t->center.setValue( m_manip->center.getValue() );
    }
    sep->addChild( t );
    sep->addChild( m_object );
    m_scene->addChild( sep );
  }
};

class SceneSaveAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    SoWriteAction myAction;
    myAction.getOutput()->openFile( "output.iv" );
    myAction.getOutput()->setBinary( FALSE );
    myAction.apply( m_scene );
    myAction.getOutput()->closeFile();
  }
};

//******************************************************************************
// Main function
int
main(int, char **argv)
{
  // Windows initialisation ----------------------------------------------------
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL)
    exit(1);

  SoDialogViz::init();

  // Scene graph initialisation ------------------------------------------------
  m_root = new SoSeparator;
  m_root->setName("m_root");
  m_root->ref();
 
  m_scene = new SoSeparator;
  m_scene->setName("m_scene");
  m_root->addChild(m_scene);

  char filename [1024];
  sprintf(filename, "$OIVHOME/examples/source/Inventor/Features/Collision/ObjectMoving/Scenes/%s", scenes_title [0]);
  SoSeparator* sep = readIvFile(filename);
  m_scene->addChild(sep);

  m_object_transform = new SoTransform;
  m_object_transform->setName("m_object_transform");
  m_object_transform->translation.setValue(10, 0, -10);
  m_object_transform->ref();
  m_root->addChild(m_object_transform);

  m_switch_material = new SoSwitch;
  m_switch_material->setName("m_switch_material");
  m_switch_material->whichChild.setValue(SO_SWITCH_NONE);
  m_root->addChild(m_switch_material);

  SoMaterial* material = new SoMaterial;
  material->setName("material");
  material->diffuseColor.setValue(1, 0, 0);
  material->setOverride(TRUE);
  m_switch_material->addChild(material);

  m_object = new SoSeparator;
  m_object->setName("m_object");
  m_root->addChild(m_object);

  sprintf(filename, "$OIVHOME/examples/source/Inventor/Features/Collision/ObjectMoving/Objects/%s", objects_title [0]);
  sep = readIvFile(filename);
  m_object->addChild(sep);

  // Set up the manipulator and the path to the transformation -----------------
  m_manip = NULL;
  m_transform_path = new SoPath(m_root);
  m_transform_path->append(m_object_transform);
  m_transform_path->ref();

  // Set up viewer -------------------------------------------------------------
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(m_root);
  myViewer->setTitle("Object Moving");
  myViewer->show();

  // Set up the collision manager ----------------------------------------------
  SoPath* object_path = new SoPath(m_root);
  object_path->append(m_object);
  object_path->ref();
  m_collision_manager = new SoCollisionManager
    (object_path, m_root, m_object_transform);
  m_collision_manager->addCollisionCallback(onCollision);

  // Setup the control dialog --------------------------------------------------
  SoRef<SoTopLevelDialog> dialog = new SoTopLevelDialog;
  dialog->label.setValue( "Object Moving Control" );

  const char* items[7] = { "None", "Center Ball", "Handle Box", "Jack", "Tab Box", "TrackBall", "Transform Box" };
  SoDialogComboBox* manipulatorType = new SoDialogComboBox;
  manipulatorType->label.setValue( "Manipulator type" );
  manipulatorType->items.setValues( 0, 7, items );
  manipulatorType->addAuditor( new ManipulatorTypeAuditor );
  dialog->addChild( manipulatorType );

  SoDialogCheckBox* collisionDetection = new SoDialogCheckBox;
  collisionDetection->label.setValue( "Collision Detection" );
  collisionDetection->state.setValue( TRUE );
  collisionDetection->onString.setValue( "on" );
  collisionDetection->offString.setValue( "off" );
  collisionDetection->addAuditor( new CollsionDetectionAuditor );
  dialog->addChild( collisionDetection );

  SoDialogCheckBox* gluing = new SoDialogCheckBox;
  gluing->label.setValue( "Gluing" );
  gluing->state.setValue( TRUE );
  gluing->onString.setValue( "on" );
  gluing->offString.setValue( "off" );
  gluing->addAuditor( new GluingAuditor );
  dialog->addChild( gluing );

  SoDialogIntegerSlider* gluingLevel = new SoDialogIntegerSlider;
  gluingLevel->label.setValue( "Gluing level" );
  gluingLevel->min.setValue( 0 );
  gluingLevel->max.setValue( 10 );
  gluingLevel->value.setValue( 1 );
  gluingLevel->addAuditor( new GluingLevelAuditor );
  dialog->addChild( gluingLevel );

  SoDialogCheckBox* objectBBox = new SoDialogCheckBox;
  objectBBox->label.setValue( "Object BBox only" );
  objectBBox->state.setValue( FALSE );
  objectBBox->onString.setValue( "on" );
  objectBBox->offString.setValue( "off" );
  objectBBox->addAuditor( new ObjectBBoxAuditor );
  dialog->addChild( objectBBox );

  SoDialogComboBox* objectFile = new SoDialogComboBox;
  objectFile->label.setValue( "Object file" );
  objectFile->items.setValues( 0, D_NUM_OBJECTS, objects_title );
  objectFile->addAuditor( new ObjectFileAuditor );
  dialog->addChild( objectFile );

  SoDialogCheckBox* sceneBBox = new SoDialogCheckBox;
  sceneBBox->label.setValue( "Scene BBox only" );
  sceneBBox->state.setValue( FALSE );
  sceneBBox->onString.setValue( "on" );
  sceneBBox->offString.setValue( "off" );
  sceneBBox->addAuditor( new SceneBBoxAuditor );
  dialog->addChild( sceneBBox );

  SoDialogComboBox* sceneFile = new SoDialogComboBox;
  sceneFile->label.setValue( "Scene file" );
  sceneFile->items.setValues( 0, D_NUM_SCENES, scenes_title );
  sceneFile->addAuditor( new SceneFileAuditor );
  dialog->addChild( sceneFile );

  SoDialogPushButton* sceneCopy = new SoDialogPushButton;
  sceneCopy->label.setValue( "Copy to scene" );
  sceneCopy->buttonLabel.setValue( "Copy" );
  sceneCopy->addAuditor( new SceneCopyAuditor );
  dialog->addChild( sceneCopy );

  SoDialogPushButton* sceneSave = new SoDialogPushButton;
  sceneSave->label.setValue( "Save scene to file" );
  sceneSave->buttonLabel.setValue( "output.iv" );
  sceneSave->addAuditor( new SceneSaveAuditor );
  dialog->addChild( sceneSave );

  dialog->buildDialog( NULL );
  dialog->show();

  // Main loop -----------------------------------------------------------------
  SoXt::show(myWindow);
  SoXt::mainLoop();

  dialog = NULL;
  SoDialogViz::finish();

  m_root->unref();
  m_object_transform->unref();
  m_transform_path->unref();
  object_path->unref();
  delete myViewer;
  SoXt::finish();

  return 0;
}

//******************************************************************************
// Function called when there is a collision
SoCollisionManager::Resp
onCollision(void*, const SoCollidingPrimitive*, const SoCollidingPrimitive*)
{
  flashObject();
  return SoCollisionManager::ABORT;
}

//******************************************************************************
// Flah one time the object
void
flashObject()
{
  static SoAlarmSensor* sensor = NULL;

  m_switch_material->whichChild.setValue(SO_SWITCH_ALL);
  if (sensor == NULL) {
    sensor = new SoAlarmSensor(unflashObject, NULL);
  }
  sensor->setTimeFromNow(SbTime(0.15));
  sensor->schedule();
}

//******************************************************************************
// Unflash the object
void
unflashObject(void*, SoSensor*)
{
  m_switch_material->whichChild.setValue(SO_SWITCH_NONE);
}

//******************************************************************************
// Reading a file
SoSeparator*
readIvFile(const char* filename)
{
  SoInput input;
  if (!input.openFile(filename))  return NULL;
  SoSeparator* sep = SoDB::readAll(&input);
  input.closeFile();
  return sep;
}

