/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Damien DALLA ROSA (Oct 2008)
**=======================================================================*/

#include <Filtering.h>

#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/devices/SoBufferObject.h>

#include <Inventor/errors/SoDebugError.h>

#include <Inventor/algorithms/SoAlgorithms.h>
#include <Inventor/algorithms/SoArithmetic.h>
#include <Inventor/algorithms/SoConversion.h>
#include <Inventor/algorithms/SoConvolution.h>

//------------------------------------------------------------------------------
Filtering::Filtering( SoAlgorithms* algorithmsInterface )
{
  m_bufferFloats[0] = NULL;
  m_bufferFloats[1] = NULL;
  m_convolutionAlgorithms = NULL;
  m_conversionAlgorithms = NULL;
  m_arithmeticAlgorithms = NULL;
  m_algorithmsInterface = algorithmsInterface;

  init();
}


//------------------------------------------------------------------------------
Filtering::~Filtering()
{
  SO_UNREF_RESET(m_bufferFloats[0]);
  SO_UNREF_RESET(m_bufferFloats[1]);
  delete m_convolutionAlgorithms;
  delete m_conversionAlgorithms;
  delete m_arithmeticAlgorithms;

  m_deviceContext->unref();
}

//------------------------------------------------------------------------------
SbBool
Filtering::init()
{
  m_deviceContext = m_algorithmsInterface->createContext();
  m_deviceContext->ref();

  m_deviceContext->bind();
  {
    m_bufferFloats[0] = m_algorithmsInterface->createBufferObject();
    m_bufferFloats[1] = m_algorithmsInterface->createBufferObject();
    m_bufferFloats[0]->ref();
    m_bufferFloats[1]->ref();

    m_convolutionAlgorithms = m_algorithmsInterface->createConvolutionInterface();
    m_conversionAlgorithms = m_algorithmsInterface->createConversionInterface();
    m_arithmeticAlgorithms = m_algorithmsInterface->createArithmeticInterface();
  }
  m_deviceContext->unbind();

  return TRUE;
}


//------------------------------------------------------------------------------
void 
Filtering::edgeDetect( SoBufferObject* inputBufferObject, 
                   SbDataType::DataType inputBufferType,
                   SoBufferObject* outputBufferObject, 
                   SbDataType::DataType outputBufferType,
                   int width,
                   int height,
                   float scale,
                   float* kernel1,
                   float* kernel2,
                   float* kernel3 )
{
  if ( !m_deviceContext )
    return;

  m_deviceContext->bind();
  {
    updateBuffersSize( width, height );

    m_conversionAlgorithms->convert( inputBufferObject, inputBufferType, m_bufferFloats[0], SbDataType::FLOAT, width*height );

    m_convolutionAlgorithms->doSeparateConvolutionRow( m_bufferFloats[0], m_bufferFloats[1], kernel1, 3, width, height ); 
    m_convolutionAlgorithms->doSeparateConvolutionColumn( m_bufferFloats[1], m_bufferFloats[0], kernel2, 3, width, height ); 
    m_convolutionAlgorithms->doSeparateConvolutionRow( m_bufferFloats[0], m_bufferFloats[1], kernel3, 3, width, height ); 
    m_convolutionAlgorithms->doSeparateConvolutionColumn( m_bufferFloats[1], m_bufferFloats[0], kernel1, 3, width, height ); 

    m_arithmeticAlgorithms->scale( m_bufferFloats[0], SbDataType::FLOAT, m_bufferFloats[1], SbDataType::FLOAT, scale );

    m_conversionAlgorithms->convert( m_bufferFloats[1], SbDataType::FLOAT, outputBufferObject, outputBufferType, width*height );
  }
  m_deviceContext->unbind();
}

//------------------------------------------------------------------------------
void 
Filtering::Sobel( SoBufferObject* inputBufferObject, 
                 SbDataType::DataType inputBufferType,
                 SoBufferObject* outputBufferObject, 
                 SbDataType::DataType outputBufferType,
                 int width,
                 int height,
                 float scale )
{
  float kernel1[3] = { 1.f / 4.f, 2.f / 4.f, 1.f / 4.f };
  float kernel2[3] = { -1.f, 0.f, 1.f };
  float kernel3[3] = { 1.f, 0.f, -1.f };

  edgeDetect( inputBufferObject, inputBufferType, 
              outputBufferObject, outputBufferType, width, height, scale,
              kernel1, kernel2, kernel3 );
}


//------------------------------------------------------------------------------
void 
Filtering::Prewitt( SoBufferObject* inputBufferObject, 
                   SbDataType::DataType inputBufferType,
                   SoBufferObject* outputBufferObject, 
                   SbDataType::DataType outputBufferType,
                   int width,
                   int height,
                   float scale )
{
  float kernel1[3] = { 1.f / 3.f, 1.f / 3.f, 1.f / 3.f };
  float kernel2[3] = { -1.f, 0.f, 1.f };
  float kernel3[3] = { 1.f, 0.f, -1.f };

  edgeDetect( inputBufferObject, inputBufferType, 
              outputBufferObject, outputBufferType, width, height, scale,
              kernel1, kernel2, kernel3 );
}


//------------------------------------------------------------------------------
void 
Filtering::Blur( SoBufferObject* inputBufferObject, 
                SbDataType::DataType inputBufferType,
                SoBufferObject* outputBufferObject, 
                SbDataType::DataType outputBufferType,
                int width,
                int height,
                int blurDepth )
{
  if ( !m_deviceContext )
    return;

  m_deviceContext->bind();
  {
    float k_10[ 11 ] = { 1.f/1024.f, 10.f/1024.f, 45.f/1024.f, 120.f/1024.f, 210.f/1024.f, 
      252.f/1024.f, 210.f/1024.f, 120.f/1024.f, 45.f/1024.f, 10.f/1024.f, 1.f/1024.f };   

    updateBuffersSize( width, height );

    m_conversionAlgorithms->convert( inputBufferObject, inputBufferType, m_bufferFloats[0], SbDataType::FLOAT, width*height );

    m_arithmeticAlgorithms->shift( m_bufferFloats[0], SbDataType::FLOAT, m_bufferFloats[1], SbDataType::FLOAT, -127.f );

    SoBufferObject* inputBuffer = NULL;
    SoBufferObject* outputBuffer = NULL;

    for ( int i = 0; i < blurDepth; i++ )
    {
      inputBuffer = m_bufferFloats[(1-i)&1];
      outputBuffer = m_bufferFloats[i&1];

      m_convolutionAlgorithms->doSeparateConvolution2D( inputBuffer, outputBuffer, k_10, 11, width, height );
    }

    m_arithmeticAlgorithms->shift( outputBuffer, SbDataType::FLOAT, inputBuffer, SbDataType::FLOAT, 127.f );

    m_conversionAlgorithms->convert( inputBuffer, SbDataType::FLOAT, outputBufferObject, outputBufferType, width*height );
 
  }
  m_deviceContext->unbind();
}


//------------------------------------------------------------------------------
SoBufferObject* 
Filtering::createBuffer( int width, int height )
{
  if ( !m_deviceContext )
    return NULL;

  m_deviceContext->bind();

  SoBufferObject* bufferObject = m_algorithmsInterface->createBufferObject();
  bufferObject->ref();

  bufferObject->setSize( width * height );

  m_deviceContext->unbind();

  return bufferObject;
}


//------------------------------------------------------------------------------
void 
Filtering::CustomKernel( SoBufferObject* inputBufferObject, 
                        SbDataType::DataType inputBufferType,
                        SoBufferObject* outputBufferObject, 
                        SbDataType::DataType outputBufferType,
                        int width,
                        int height,
                        float* rowKernel,
                        int rowKernelSize,
                        float* columnKernel,
                        int columnKernelSize,
                        float scale,
                        int iterations,
                        int convolutionCenter )
{
  if ( !m_deviceContext )
    return;

  m_deviceContext->bind();
  {
    updateBuffersSize( width, height );

    m_conversionAlgorithms->convert( inputBufferObject, inputBufferType, m_bufferFloats[0], SbDataType::FLOAT, width*height );

    if ( convolutionCenter )
      m_arithmeticAlgorithms->shift( m_bufferFloats[0], SbDataType::FLOAT, m_bufferFloats[0], SbDataType::FLOAT, -float( convolutionCenter ) );

    SoBufferObject* inputBuffer = m_bufferFloats[0];
    SoBufferObject* outputBuffer = m_bufferFloats[1];

    for ( int i = 0; i < iterations; i++ )
    {
      m_convolutionAlgorithms->doSeparateConvolutionRow( inputBuffer, outputBuffer, rowKernel, rowKernelSize, width, height );
      m_convolutionAlgorithms->doSeparateConvolutionColumn( outputBuffer, inputBuffer, columnKernel, columnKernelSize, width, height );
    }

    if ( convolutionCenter )
      m_arithmeticAlgorithms->shift( inputBuffer, SbDataType::FLOAT, inputBuffer, SbDataType::FLOAT, float( convolutionCenter ) );

    m_arithmeticAlgorithms->scale( inputBuffer, SbDataType::FLOAT, inputBuffer, SbDataType::FLOAT, scale );
    m_conversionAlgorithms->convert( inputBuffer, SbDataType::FLOAT, outputBufferObject, outputBufferType, width*height );
  }
  m_deviceContext->unbind();
}


//------------------------------------------------------------------------------
void 
Filtering::updateBuffersSize( int width, int height )
{
  size_t sizeInFloats = width * height * sizeof( float );

  if ( m_bufferFloats[1]->getSize() != sizeInFloats )
    m_bufferFloats[1]->setSize( sizeInFloats );

  if ( m_bufferFloats[0]->getSize() != sizeInFloats )
    m_bufferFloats[0]->setSize( sizeInFloats );
}


//------------------------------------------------------------------------------

