
#include <Inventor/SoDB.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoLightModel.h> 
#include <Inventor/nodes/SoTexture2.h>

#include <Inventor/image/SbRasterImage.h>
#include <Inventor/image/SoRasterReaderSet.h>
#include <Inventor/image/SoRasterImageFile.h>

#include <Inventor/algorithms/SoAlgorithms.h>

#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/devices/SoCpuBufferObject.h>

#include <Inventor/actions/SoWriteAction.h> 

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/helpers/SbFileHelper.h>
#include <Filtering.h>

//------------------------------------------------------------------------------

SoXtExaminerViewer* g_examinerViewer = NULL;

SoBufferObject* g_imageBufferObject = NULL;
SoBufferObject* g_inputBufferObject = NULL;
SoBufferObject* g_textureBufferObject = NULL;

int g_inputWidth = -1;
int g_inputHeight = -1;
int g_inputComponentsCount = -1;

float g_scale = 1.0f;
int g_blurIterations = 10;

SoTexture2* g_computeTexture = NULL;

int g_filterId = 0;
SoTopLevelDialog* g_topLevelDialog = NULL;

const char *g_guiFilename = "$OIVHOME/examples/source/Inventor/Features/Compute/TextureConvolution/data/gui.iv";

Filtering* g_filteringInterface = NULL;

//------------------------------------------------------------------------------

void setupFilteringInterface();

bool readRasterImage( std::string filename, SoBufferObject* bufferObject, int& width, int& height, int& nc, bool convertToGray );

void updateInterface();

void enableInterfaceComponent( SbString name, SbBool enable );

float getTextEditValue( SbString name );

//------------------------------------------------------------------------------
void enableInterfaceComponent( SbString name, SbBool enable )
{
  SoDialogComponent* component = (SoDialogComponent*)g_topLevelDialog->searchForAuditorId( name );

  if ( component )
    component->enable = enable;
}

//------------------------------------------------------------------------------
float getTextEditValue( SbString name )
{
  SoDialogEditText* dialogEditText = (SoDialogEditText *)g_topLevelDialog->searchForAuditorId( name );
  if ( dialogEditText ) 
    return float( atof( dialogEditText->editText.getValue().getString() ) );

  return 0.f;
}

//------------------------------------------------------------------------------
void updateInterface()
{
  if ( !g_topLevelDialog )
    return;

  enableInterfaceComponent( SbString("CustomKernelApplyButton"), FALSE );

  if ( g_filterId == 0 )
  {
    enableInterfaceComponent( SbString("Scale"), FALSE );
    enableInterfaceComponent( SbString("BlurIterations"), FALSE );
  }

  if ( g_filterId == 1 )
  {
    enableInterfaceComponent( SbString("Scale"), FALSE );
    enableInterfaceComponent( SbString("BlurIterations"), TRUE );
  }

  if ( (g_filterId == 2) || (g_filterId == 3) )
  {
    enableInterfaceComponent( SbString("Scale"), TRUE );
    enableInterfaceComponent( SbString("BlurIterations"), FALSE );
  }

 if ( g_filterId == 4 )
  {
   enableInterfaceComponent( SbString("Scale"), TRUE );
   enableInterfaceComponent( SbString("BlurIterations"), TRUE );
   enableInterfaceComponent( SbString("CustomKernelApplyButton"), TRUE );
  }
}


//------------------------------------------------------------------------------
void setupFilteringInterface()
{
  SoDeviceContext* context = NULL;
  static SoAlgorithms* algorithmsInterface = NULL;
  
  if ( g_inputBufferObject )
  {
    context = g_inputBufferObject->getContext();

    context->bind();
    SO_UNREF_RESET(g_inputBufferObject);
    context->unbind();
  }

  if ( g_textureBufferObject )
  {
    context = g_textureBufferObject->getContext();

    context->bind();
    SO_UNREF_RESET(g_textureBufferObject);
    context->unbind();
  }

  delete g_filteringInterface;
  g_filteringInterface = NULL;

  delete algorithmsInterface;
  algorithmsInterface = NULL;



  algorithmsInterface = new SoAlgorithms();

  if ( algorithmsInterface )
  {
    g_filteringInterface = new Filtering( algorithmsInterface );

    g_inputBufferObject = g_filteringInterface->createBuffer( g_inputWidth, g_inputHeight );
    g_textureBufferObject = g_filteringInterface->createBuffer( g_inputWidth, g_inputHeight );

    g_inputBufferObject->getContext()->bind();
    g_inputBufferObject->memcpy( g_imageBufferObject );
    g_inputBufferObject->getContext()->unbind();
  }
}

//------------------------------------------------------------------------------
void updateTexture()
{
  if ( !g_filteringInterface )
    return;

  if ( g_filterId == 0 )
  {
    g_textureBufferObject->getContext()->bind();
    g_textureBufferObject->memcpy( g_inputBufferObject );
    g_textureBufferObject->getContext()->unbind();
  }

  if ( g_filterId == 1 )
  {
    g_filteringInterface->Blur( g_inputBufferObject, SbDataType::UNSIGNED_BYTE,
      g_textureBufferObject, SbDataType::UNSIGNED_BYTE,
      g_inputWidth,
      g_inputHeight,
      g_blurIterations );
  }

  if ( g_filterId == 2 )
  {
    g_filteringInterface->Sobel( g_inputBufferObject, SbDataType::UNSIGNED_BYTE,
      g_textureBufferObject, SbDataType::UNSIGNED_BYTE,
      g_inputWidth,
      g_inputHeight,
      g_scale );
  }

  if ( g_filterId == 3 )
  {
    g_filteringInterface->Prewitt( g_inputBufferObject, SbDataType::UNSIGNED_BYTE,
      g_textureBufferObject, SbDataType::UNSIGNED_BYTE,
      g_inputWidth,
      g_inputHeight,
      g_scale );
  }

  if ( g_filterId == 4 )
  {
    float rowKernel[5];
    float columnKernel[5];

    rowKernel[ 0 ] = getTextEditValue( SbString("kernel_1_1") );
    rowKernel[ 1 ] = getTextEditValue( SbString("kernel_1_2") );
    rowKernel[ 2 ] = getTextEditValue( SbString("kernel_1_3") );
    rowKernel[ 3 ] = getTextEditValue( SbString("kernel_1_4") );
    rowKernel[ 4 ] = getTextEditValue( SbString("kernel_1_5") );
    
    columnKernel[ 0 ] = getTextEditValue( SbString("kernel_2_1") );
    columnKernel[ 1 ] = getTextEditValue( SbString("kernel_2_2") );
    columnKernel[ 2 ] = getTextEditValue( SbString("kernel_2_3") );
    columnKernel[ 3 ] = getTextEditValue( SbString("kernel_2_4") );
    columnKernel[ 4 ] = getTextEditValue( SbString("kernel_2_5") );

     SoDialogIntegerSlider* convolutionCenterSlider =
      (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId( SbString("ConvolutionCenter") );

    int convolutionCenter = ((convolutionCenterSlider != NULL)?convolutionCenterSlider->value.getValue():0);

    float sum = 0;
    for ( int i = 0; i<5; i++ )
      sum += rowKernel[ i ];

    if ( fabs( sum ) > 0.001 )
      for ( int i = 0; i<5; i++ )
        rowKernel[ i ] /= sum;

    sum = 0;
    for ( int i = 0; i<5; i++ )
      sum += columnKernel[ i ];

    if ( fabs( sum ) > 0.001 )
      for ( int i = 0; i<5; i++ )
        columnKernel[ i ] /= sum;

    g_filteringInterface->CustomKernel( g_inputBufferObject, SbDataType::UNSIGNED_BYTE,
            g_textureBufferObject, SbDataType::UNSIGNED_BYTE,
            g_inputWidth,
            g_inputHeight,
            rowKernel,
            5,
            columnKernel,
            5,
            g_scale,
            g_blurIterations,
            convolutionCenter
            );
  }

  g_computeTexture->image.setValue( SbVec2s( g_inputWidth, g_inputHeight ), 
    g_inputComponentsCount, SoSFImage::UNSIGNED_BYTE, 
    g_textureBufferObject, SoSFImage::NO_COPY );
}

////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle inputs

class MyAuditorClass : public SoDialogAuditor
{
  void dialogRealSlider( SoDialogRealSlider* cpt );
  void dialogIntegerSlider( SoDialogIntegerSlider* cpt );
  void dialogComboBox( SoDialogComboBox* cpt );
  void dialogPushButton( SoDialogPushButton* cpt );
};


void
MyAuditorClass::dialogIntegerSlider( SoDialogIntegerSlider* cpt )
{
  if ( cpt->auditorID.getValue() == "BlurIterations" )
  {
    g_blurIterations = cpt->value.getValue();

    updateTexture();
  }
}

void
MyAuditorClass::dialogRealSlider( SoDialogRealSlider* cpt )
{
  // Change slice number
  if ( cpt->auditorID.getValue() == "Scale" )
  {
    g_scale = cpt->value.getValue();

    updateTexture();
  }
}

void
MyAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
  if ( cpt->auditorID.getValue() == "Filter" )
  {
    g_filterId = cpt->selectedItem.getValue();

    updateInterface();

    updateTexture();
  }

  if ( cpt->auditorID.getValue() == "Mode" )
  {
    setupFilteringInterface();

    updateTexture();
  }
}

void
MyAuditorClass::dialogPushButton(SoDialogPushButton* cpt)
{
  if ( cpt->auditorID.getValue() == "CustomKernelApplyButton" )
  {
    updateTexture();
  }
}


//------------------------------------------------------------------------------
Widget
buildInterface( Widget window )
{
  SoInput myInput;
  if (! myInput.openFile( g_guiFilename ))
    return NULL;

  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup)
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  MyAuditorClass *myAuditor = new MyAuditorClass;
  g_topLevelDialog->addAuditor( myAuditor );

  // setup slice slider range

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, customNode != NULL );
  g_topLevelDialog->show();

  updateInterface();

  return customNode ? customNode->getWidget() : window;

}

//------------------------------------------------------------------------------
int 
main( int /*argc*/, char** /*argv*/ )
{
  // We init OIV
  Widget mainWindow = SoXt::init( "Unit test" );
  if (mainWindow == NULL)
    exit (1);

  SoDialogViz::init();
  
  SoAlgorithms::init();

  // Setup input buffers
  g_imageBufferObject = new SoCpuBufferObject;

  SbString RasterFile = SbFileHelper::expandString("$OIVHOME/data/textures/misc/lena.png");
  if ( readRasterImage( RasterFile.getSString(), g_imageBufferObject, g_inputWidth, g_inputHeight, 
    g_inputComponentsCount, true ) == false )
  {
    printf( "[F] Fatal cannot read input file lena.png\n" );
    return -1;
  } 

  // The gui
  Widget mainWidget = buildInterface( mainWindow );

  // The light
  SoLightModel* light = new SoLightModel;
  {
    light->model = SoLightModel::BASE_COLOR;
  }

  // The default material
  SoMaterial* material = new SoMaterial;
  {
    material->diffuseColor.setValue( 1.0, 1.0, 1.0 );
  }

  // The result texture
  g_computeTexture = new SoTexture2;

  g_computeTexture->image.setValue( SbVec2s( g_inputWidth, g_inputHeight ), 
    g_inputComponentsCount, SoSFImage::UNSIGNED_BYTE, 
    g_textureBufferObject, SoSFImage::NO_COPY );


  SoFaceSet* faceSet = new SoFaceSet;
  {
    SoVertexProperty* vp = new SoVertexProperty;
    {     
      vp->vertex.set1Value( 0, -1.0, -1.0, 0.0 );
      vp->vertex.set1Value( 1, -1.0, 1.0, 0.0 );
      vp->vertex.set1Value( 2, 1.0, 1.0, 0.0 );
      vp->vertex.set1Value( 3, 1.0, -1.0, 0.0 ); 

      vp->texCoord.set1Value( 0, 0.0, 0.0 );
      vp->texCoord.set1Value( 1, 0.0, 1.0 );
      vp->texCoord.set1Value( 2, 1.0, 1.0 );
      vp->texCoord.set1Value( 3, 1.0, 0.0 );
    }
    faceSet->vertexProperty = vp;
  }

  // Assembly
  SoSeparator* root = new SoSeparator;
  root->ref();
  {
    root->addChild( light );
    root->addChild( material );
    root->addChild( g_computeTexture );
    root->addChild( faceSet );
  }

  // Build and initialize the Inventor render area widget
  g_examinerViewer = new SoXtExaminerViewer( mainWidget );
  {
    //examiner->setSize( SbVec2s( 512, 512 ) );
    g_examinerViewer->setDecoration( TRUE );
    g_examinerViewer->setFeedbackVisibility( FALSE );
    g_examinerViewer->setSceneGraph( root );
    g_examinerViewer->setTitle( "Unit Test" );
    g_examinerViewer->setHeadlight( false );

    g_examinerViewer->show();
    g_examinerViewer->viewAll();
  }

  setupFilteringInterface();
  updateTexture();

  // Ok here we go...
  SoXt::show( mainWindow );
  SoXt::mainLoop();
  
  delete g_examinerViewer;
  
  root->unref();
  
  SoAlgorithms::finish();
  
  SoDialogViz::finish();
  
  SoXt::finish();

  return 0;
}


//------------------------------------------------------------------------------
bool readRasterImage( std::string filename, SoBufferObject* bufferObject, int& width, int& height, int& nc, bool convertToGray )
{
  SbRasterImage* rasterImage = new SbRasterImage;

  SoRasterImageFile* rasterFile = new SoRasterImageFile;
  {
    rasterFile->setFileName( filename.c_str() );
  }

  SoRasterImageRW* imageReader = NULL;
  SbString ext;

  if ( filename.find( ".png" ) != std::string::npos )
    ext = SbString("PNG");
  if ( filename.find( ".tif" ) != std::string::npos )
    ext = SbString("TIF");
  if ( filename.find( ".jpg" ) != std::string::npos )
    ext = SbString("JPG");

  imageReader = SoRasterReaderSet::getReader( rasterFile, ext );

  if ( !imageReader )
  {
    printf("[D] Error imageReader is NULL\n");
    delete rasterFile;
    return false;
  }


  if ( !imageReader->open( rasterFile, SoRasterImageRW::OPEN_READ ) ) 
  {
    delete rasterFile;
    return false;
  }

  if (!imageReader->read( rasterImage )) 
  {
    imageReader->close();
    delete rasterFile;
    return false;
  }

  width = rasterImage->getSize()[0];
  height = rasterImage->getSize()[1];
  int rasterNc = (rasterImage->getComponents() == SbRasterImage::LUMINANCE)? 1 : 3;
  nc = rasterNc;

  if ( convertToGray )
    nc = 1;

  SoRef<SoBufferObject> bufferImage = rasterImage->getBufferObject();
  SoRef<SoCpuBufferObject> cpuBufferImage = new SoCpuBufferObject();
  bufferImage->map(cpuBufferImage.ptr(), SoBufferObject::READ_WRITE);
  unsigned char* ptr = (unsigned char*) cpuBufferImage->map(SoBufferObject::READ_ONLY);

  int size = width * height * nc;
  bufferObject->setSize( size );

  SoRef<SoCpuBufferObject> cpuBuffer = new SoCpuBufferObject;

  bufferObject->map( cpuBuffer.ptr(), SoBufferObject::SET );
  {
    unsigned char* destPtr = (unsigned char*)cpuBuffer->map( SoBufferObject::SET );

    if ( convertToGray )
    {
      for ( int i = 0; i<width*height; i++ )
      {
        if ( rasterNc != 3 )
          destPtr[ i ] = ptr[ rasterNc*i ];
        else
        {
          destPtr[ i ] = (unsigned char)( 
            0.299f * float( ptr[ 3*i ] ) + 
            0.587f * float( ptr[ 3*i + 1 ] ) + 
            0.114f * float( ptr[ 3*i + 2 ] ) 
            );
        }
      }
    }
    else
    {
      memcpy( destPtr, ptr, size );
    }

    cpuBuffer->unmap();
  }
  bufferObject->unmap( cpuBuffer.ptr() );

  cpuBufferImage->unmap();
  bufferImage->unmap( cpuBufferImage.ptr() );

  delete rasterFile;
  delete rasterImage;

  return true;
}

//------------------------------------------------------------------------------


