#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtFileSelectionDialog.h>

#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoWWWInline.h>

#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/helpers/SbFileHelper.h>


SoSeparator         *ReadScene ;
SoXtExaminerViewer  *MyViewer ;
SoSwitch            *InfoSwitch ;
SoText2             *InfoText ;

#define TITLE "Correct Transparency"
#define CUR_TRANSP_TITLE  "Cur. Transp. : "
#define MAX_TRANSP_PASSES 32

#define UNSUPPORTED_MSG "Sorted pixels blend transparency is not supported by your graphic board !"

int CurrentTransp = 0 ;
static const char *TransparencyNames[] = 
{"NO_SORT",
 "OPAQUE_FIRST",
 "SORTED_OBJECT",
 "SORTED_PIXEL"} ;

#define NUM_TRANPARENCIES (sizeof(TransparencyNames)/sizeof(TransparencyNames[0]))

int NumTranspPasses = 4 ;

/*----------------------------------------------------------------------------*/
void
fpsCB(float fps, void *, SoXtViewer *viewer) {
  SbString titleStr(TITLE) ;
  SbString fpsStr((int)fps) ;
  titleStr += " : " ;
  titleStr += fpsStr ;
  titleStr += " fps" ;
  
  viewer->setTitle(titleStr.getString()) ;
}/*----------------------------------------------------------------------------*/

void
updateCurTranspInfo()
{
  SbString str(CUR_TRANSP_TITLE) ;
  str+=TransparencyNames[CurrentTransp] ;
  if(CurrentTransp==1||CurrentTransp==3) { // OPAQUE_FIRST | SORTED_PIXEL
     str+= " (" ;
     str+= NumTranspPasses ;
     str+= " layers)";
  }
  InfoText->string.set1Value(3, str) ;  
}/*----------------------------------------------------------------------------*/

void 
myKeyPressCB (void *, SoEventCallback *eventCB) 
{
  
  const SoEvent *event = eventCB->getEvent();
  
  // check for the keys being pressed
  if (SO_KEY_PRESS_EVENT(event, T)) {
    CurrentTransp =  (CurrentTransp+1)%NUM_TRANPARENCIES ;
    switch(CurrentTransp) {
    case 0:
      MyViewer->setTransparencyType(SoGLRenderAction::NO_SORT);
      break ;

    case 1:
      MyViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
      break ;

    case 2:
      MyViewer->setTransparencyType(SoGLRenderAction::SORTED_OBJECT);
      break ;

    case 3:
      MyViewer->setTransparencyType(SoGLRenderAction::SORTED_PIXEL);
      break ;
    }
    updateCurTranspInfo() ;
  } 
  else if (SO_KEY_PRESS_EVENT(event, PAD_ADD)) {
    NumTranspPasses++;
    if(NumTranspPasses > MAX_TRANSP_PASSES) NumTranspPasses = MAX_TRANSP_PASSES ;
    updateCurTranspInfo() ;
    MyViewer->getSceneManager()->getGLRenderAction()->setSortedLayersNumPasses(NumTranspPasses) ;
  }
  else if (SO_KEY_PRESS_EVENT(event, PAD_SUBTRACT)) {
    NumTranspPasses-- ;
    if(NumTranspPasses < 1) NumTranspPasses = 1 ;
    updateCurTranspInfo() ;
    MyViewer->getSceneManager()->getGLRenderAction()->setSortedLayersNumPasses(NumTranspPasses) ;
  }
  else if (SO_KEY_PRESS_EVENT(event, H)) {
	   if(InfoSwitch->whichChild.getValue() == SO_SWITCH_ALL)
       InfoSwitch->whichChild = SO_SWITCH_NONE ;
     else
       InfoSwitch->whichChild = SO_SWITCH_ALL ;   
  }
}/*---------------------------------------------------------------------------*/

SoSeparator *
readFile(const SbString& filename)
{
  // Open the input file
  SoInput mySceneInput;
  if (!mySceneInput.openFile(filename)) {
    fprintf(stderr, "Cannot open file %s\n", filename.toLatin1());
    return NULL;
  }
  SoWWWInline::setReadAsSoFile(TRUE);
  // Read the whole file into the database
  SoSeparator *myGraph = SoDB::readAll(&mySceneInput);
  if (myGraph == NULL) {
    fprintf(stderr, "Problem reading file\n");
    return NULL;
  } 
  
  mySceneInput.closeFile();
  return myGraph;
}/*----------------------------------------------------------------------------*/

void readScene(const SbString& filename)
{
  SoSeparator *scene = readFile(filename) ;
  
  if(scene) {
    SbViewportRegion vpRegion ;
    
    ReadScene->removeAllChildren() ;
    ReadScene->addChild(scene) ;
    MyViewer->viewAll() ;
  }
}/*----------------------------------------------------------------------------*/

void
openFileCB(void* /*data*/, SoXtFileSelectionDialog *dialog)
{
  readScene(dialog->getFilePath());
}/*----------------------------------------------------------------------------*/

void
abortCB(void* /*data*/, SoXtFileSelectionDialog* /*dialog*/)
{
  exit(0) ;
}/*---------------------------------------------------------------------------*/

SoSwitch*
displayInfo() 
{
  // Informations
  SoSwitch *infoSwitch = new SoSwitch ;
  infoSwitch->ref() ;
  infoSwitch->whichChild = SO_SWITCH_ALL ;
  
  SoSeparator *infoSep = new SoSeparator ;

  SoOrthographicCamera *cam = new SoOrthographicCamera ;
  infoSep->addChild(cam) ;
  cam->viewportMapping = SoOrthographicCamera::LEAVE_ALONE ;
  cam->position.setValue(SbVec3f(0.005f,0.005f,1)) ;
  cam->height = 0.01f ;

  
  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;
  infoSep->addChild(pickStyle) ;
  
  infoSwitch->addChild(infoSep) ;
  
  SoLightModel *lModel = new SoLightModel ;
  lModel->model = SoLightModel::BASE_COLOR ;
  infoSep->addChild(lModel) ;
  
  SoFont *fontInfo = new SoFont ;
  fontInfo->name = "Courier New" ;
  fontInfo->size = 12 ;
  infoSep->addChild(fontInfo) ;
  
  SoBaseColor *infoColor = new SoBaseColor ;
  infoColor->rgb.setValue(SbColor(1, 1, 0.f)) ;
  infoSep->addChild(infoColor) ;
  
  SoTranslation *transInfo = new SoTranslation ;
  transInfo->translation.setValue(0.00025f, 0.00975f, 0.) ;
  infoSep->addChild(transInfo) ;
  
  InfoText = new SoText2 ;
  InfoText->string.set1Value(0, "H   : Toggle this display") ;
  InfoText->string.set1Value(1, "+/- : Inc./Dec. the number of passes") ;
  InfoText->string.set1Value(2, "T   : Change transparency type") ;
  updateCurTranspInfo() ;
  infoSep->addChild(InfoText) ;
  
  infoSwitch->unrefNoDelete() ;
  return infoSwitch ;
}/*---------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]);

  if(!SoGLRenderAction::isSortedLayersSupported()) {
#ifdef _WIN32
    MessageBox(NULL, UNSUPPORTED_MSG, "Correct Transparency",MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
    printf("%s\n", UNSUPPORTED_MSG) ;
#endif
  }

  SoSeparator *root = new SoSeparator ;
  root->ref() ;
  
  // Display Info
  InfoSwitch = displayInfo() ;
  root->addChild(InfoSwitch) ;
  
  // Track the keyboard events
  SoEventCallback *myEventCB = new SoEventCallback;
  myEventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(),
                              myKeyPressCB, NULL);
  root->addChild(myEventCB);
  
  SoPerspectiveCamera *camera = new SoPerspectiveCamera ;
  root->addChild(camera) ;

  // Force the 2 side lighting.
  SoShapeHints *sh = new SoShapeHints ;
  sh->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE ;

  root->addChild(sh) ;

  // Create the shadow group node
  ReadScene = new SoSeparator ;
  root->addChild(ReadScene) ;
  
  // Create a viewer
  MyViewer = new SoXtExaminerViewer(myWindow);
  
  // Attach and show viewer
  MyViewer->setSceneGraph(root);
  MyViewer->setCamera(camera) ;
  MyViewer->setTitle(TITLE);
  //MyViewer->setFramesPerSecondCallback(fpsCB) ;
  MyViewer->setTransparencyType(SoGLRenderAction::SORTED_PIXEL);

  MyViewer->show();

  SoXtFileSelectionDialog *fsd = 0;
  if (argc == 2)
  {
    readScene(argv[1]);
  }
  else
  {
    fsd = new SoXtFileSelectionDialog;
    // File selection dialog to choose an Inventor file
    SbString fileDir = SbFileHelper::expandString( "$OIVHOME" ) +  "/examples/data/Inventor/CorrectTransp";
    fsd->setFileDirectory((char*)fileDir.toLatin1 ()) ;
    fsd->setTitle((char*)"Select an Inventor file") ;
    fsd->setMode(SoXtFileSelectionDialog::OPEN_FILE) ;
    fsd->setSize(SbVec2s(400, 400));
    fsd->setFileName((char*)"teapot.iv") ;
#ifdef _WIN32
    fsd->setFilter("iv", "Inventor files (*.iv)");
#else
    fsd->setFilter((char*)"iv", NULL);
#endif
    fsd->setAcceptCallback(openFileCB,NULL) ;
    fsd->setAbortCallback(abortCB,NULL) ;
    fsd->show() ;
  }
  
  // Loop forever
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete fsd;
  
  root->unref();
  delete MyViewer;
  
  SoXt::finish();

  return 0;
}/*----------------------------------------------------------------------------*/


