#include "CustomNodeOpenGLCore.h"

#include <Inventor/elements/SoLightModelElement.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/sys/SoGL.h>

// external stuff for bunny data access
extern uint32_t numVertices;
extern float vertex[];
extern float normal[];
extern float color[];

SO_NODE_SOURCE(CustomNodeOpenGLCore);

SoShaderProgram*
CustomNodeOpenGLCore::getShaderProgram()
{
  SoShaderProgram* program = new SoShaderProgram;
  program->setVertexShader(0, "shaders/CustomNodeCore.vert");
  program->setFragmentShader(1, "shaders/CustomNodeCore.frag");
  return program;
}

CustomNodeOpenGLCore::CustomNodeOpenGLCore()
 : m_initialized(false)
{
  SO_NODE_CONSTRUCTOR(CustomNodeOpenGLCore);
}

CustomNodeOpenGLCore::~CustomNodeOpenGLCore()
{
}

void
CustomNodeOpenGLCore::initClass()
{
  getClassRenderEngineMode().setRenderMode(SbRenderEngineMode::OIV_OPENGL_CORE_RENDERING);
  SO__NODE_INIT_CLASS(CustomNodeOpenGLCore, "CustomNodeOpenGLCore", SoShape);
}

void
CustomNodeOpenGLCore::exitClass()
{
  SO__NODE_EXIT_CLASS(CustomNodeOpenGLCore);
}

void
CustomNodeOpenGLCore::initialize()
{
  if (m_initialized)
    return;

  glGenBuffers(1, &m_vertexBuffer);
  glBindBuffer(GL_ARRAY_BUFFER, m_vertexBuffer);
  glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(float) * 3, &vertex[0], GL_STATIC_DRAW);

  glGenBuffers(1, &m_colorBuffer);
  glBindBuffer(GL_ARRAY_BUFFER, m_colorBuffer);
  glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(float) * 3, &color[0], GL_STATIC_DRAW);

  glGenBuffers(1, &m_normalBuffer);
  glBindBuffer(GL_ARRAY_BUFFER, m_normalBuffer);
  glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(float) * 3, &normal[0], GL_STATIC_DRAW);

  m_initialized = true;
}

void
CustomNodeOpenGLCore::GLRender(SoGLRenderAction *action)
{
  // create resources
  initialize();

  // vertex positions
  glEnableVertexAttribArray(0);
  glBindBuffer(GL_ARRAY_BUFFER, m_vertexBuffer );
  glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

  SoState* state = action->getState();
  SoLightModelElement::Model model = SoLightModelElement::get(state);

  if (model == SoLightModelElement::PER_PIXEL_PHONG || model == SoLightModelElement::PER_VERTEX_PHONG)
  {
    // vertex colors
    glEnableVertexAttribArray(1);
    glBindBuffer(GL_ARRAY_BUFFER, m_colorBuffer);
    glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, 0);

    // vertex normals
    glEnableVertexAttribArray(2);
    glBindBuffer(GL_ARRAY_BUFFER, m_normalBuffer);
    glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, 0, 0);
  }
  glDrawArrays(GL_TRIANGLES, 0, numVertices);

  if (model == SoLightModelElement::PER_PIXEL_PHONG || model == SoLightModelElement::PER_VERTEX_PHONG)
  {
    glDisableVertexAttribArray(2);
    glDisableVertexAttribArray(1);
  }
  glDisableVertexAttribArray(0);
}

void
CustomNodeOpenGLCore::computeBBox(SoAction* /*action*/, SbBox3f &box, SbVec3f &center)
{
  SbBox3f bbox;

  float* vert = vertex;
  for (uint32_t i = 0; i < numVertices; ++i)
  {
    bbox.extendBy(SbVec3f(vert[0], vert[1], vert[2]));
    vert += 3;
  }

  box = bbox;
  center = box.getCenter();
}
