#pragma once

#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/sys/SoGL.h>
#include "CustomNodeOpenGLCore.h"

/**
 * CustomNodeBase
 *
 * @DESCRIPTION
 * This custom node handles transparency, if any, before forward traversing an internal node.
 *
 * The GLRender() method shows how custom nodes should handle transparency.
 * In OIV10, we can't mix Inventor and OpenGL API calls in the same custom node.
 * So, in order to render a transparent shape using a custom node, we need to
 * encapsulate it inside a pure Open Inventor custom node "wrapper" that applies a forward
 * traversal on an internal custom OpenGL node that will handle the actual draw calls.
 *
 * In this example we know that the custom OpenGL geometry is transparent, so the
 * "wrapper" node's GLRender() method calls:
 * \code
 * if (action->handleTransparency(TRUE))
 *   return;
 * \endcode
 * This line of code does two things.  It tells Open Inventor that the geometry is
 * partially transparent (passes true to the method) and it returns immediately
 * if rendering of transparent geometry is deferred (method returns true).
 */
class CustomNodeBase : public SoNode
{
  SO_NODE_HEADER(CustomNodeBase);

public:

  /**
  * Default constructor.
  */
  CustomNodeBase();

  // Method used to set the internal custom node containing the actual draw calls
  void setCustomNodeType(SoNode* node);

  static void initClass();
  static void exitClass();

protected:

  // destructor
  virtual ~CustomNodeBase();

  void generatePrimitives(SoAction* action);

  // create OpenGL resources
  void initialize();

  // destroy OpenGL resources
  void destroyResources();

SoEXTENDER public:

  // Rendering stuff
  virtual void GLRender(SoGLRenderAction *action);

  // BBox computation
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

private:
  // Pointer to an internal OpenGL custom node
  SoRef<SoNode> m_internalGLNode;
};
