#include "CustomNodeGLCallback.h"

#include <Inventor/elements/SoLightModelElement.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/sys/SoGL.h>

// external stuff for bunny data access
extern uint32_t numVertices;
extern float vertex[];
extern float normal[];
extern float color[];


CustomNodeGLCallback::CustomNodeGLCallback()
  : m_initialized(false)
{

}

CustomNodeGLCallback::~CustomNodeGLCallback()
{
  destroy();
}

SoShaderProgram*
CustomNodeGLCallback::getShaderProgram()
{
  SoShaderProgram* program = new SoShaderProgram;
  program->setVertexShader(0, "shaders/CustomNodeLegacy.vert");
  program->setFragmentShader(1, "shaders/CustomNodeLegacy.frag");
  program->generateTransparency = true;
  return program;
}

void
CustomNodeGLCallback::initialize()
{
  if (m_initialized)
  {
    return;
  }

  glGenBuffers(1, &m_vertexBuffer);
  glBindBuffer(GL_ARRAY_BUFFER, m_vertexBuffer);
  glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(float) * 3, &vertex[0], GL_STATIC_DRAW);

  glGenBuffers(1, &m_colorBuffer);
  glBindBuffer(GL_ARRAY_BUFFER, m_colorBuffer);
  glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(float) * 4, &color[0], GL_STATIC_DRAW);

  glGenBuffers(1, &m_normalBuffer);
  glBindBuffer(GL_ARRAY_BUFFER, m_normalBuffer);
  glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(float) * 3, &normal[0], GL_STATIC_DRAW);

  m_initialized = true;
}

void
CustomNodeGLCallback::destroy()
{
  if (!m_initialized)
    return;

  glDeleteBuffers(1, &m_vertexBuffer);
  glDeleteBuffers(1, &m_normalBuffer);
  glDeleteBuffers(1, &m_colorBuffer);

  m_initialized = false;
}

void
CustomNodeGLCallback::callback(void* data, SoAction *action)
{
  CustomNodeGLCallback* cb = static_cast<CustomNodeGLCallback*>(data);

  cb->initialize();

  glBindBuffer(GL_ARRAY_BUFFER, cb->m_vertexBuffer);
  glEnableClientState(GL_VERTEX_ARRAY);
  glVertexPointer(3, GL_FLOAT, 0, 0);

  SoState* state = action->getState();
  SoLightModelElement::Model model = SoLightModelElement::get(state);

  if (model == SoLightModelElement::PHONG || model == SoLightModelElement::PER_VERTEX_PHONG)
  {
    // vertex colors + alpha
    glBindBuffer(GL_ARRAY_BUFFER, cb->m_colorBuffer);
    glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    glEnableClientState(GL_COLOR_ARRAY);
    glColorPointer(4, GL_FLOAT, 0, 0);

    // vertex normals
    glBindBuffer(GL_ARRAY_BUFFER, cb->m_normalBuffer);
    glEnableClientState(GL_NORMAL_ARRAY);
    glNormalPointer(GL_FLOAT, 0, 0);
  }

  glDrawArrays(GL_TRIANGLES, 0, numVertices);

  if (model == SoLightModelElement::PHONG || model == SoLightModelElement::PER_VERTEX_PHONG)
  {
    glDisable(GL_COLOR_MATERIAL);
    glDisableClientState(GL_NORMAL_ARRAY);
    glDisableClientState(GL_COLOR_ARRAY);
  }

  glDisableClientState(GL_VERTEX_ARRAY);

  glDisable(GL_BLEND);
}
