#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/elements/SoTextureCoordinateElement.h>
#include <Inventor/elements/SoTextureEnabledElement.h>
#include <Inventor/elements/SoModelMatrixElement.h>
#include <Inventor/misc/SoState.h>

#include "CustomNodeOIV.h"

SO_NODE_SOURCE(Pyramid);

// Vertex positions
std::vector<SbVec3f> Pyramid::s_vertices;

// Vertex normals
std::vector<SbVec3f> Pyramid::s_normals;

// Texture coordinates
std::vector<SbVec2f> Pyramid::s_texCoords;

// Vertex indices
std::vector<int32_t> Pyramid::s_vertexIndices;

// Normal indices
std::vector<int32_t> Pyramid::s_normalIndices;

// Texture Coordinates indices
std::vector<int32_t> Pyramid::s_texCoordIndices;

// Material indices
std::vector<int32_t> Pyramid::s_materialIndices;

// static vertex property node containing the geometry of the pyramid
SoRef<SoVertexProperty> Pyramid::s_vertexProperty;

// This initializes the Pyramid class.
void
Pyramid::initClass()
{
  // Initialize type id variables
  getClassRenderEngineMode().setRenderMode(SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING);
  SO__NODE_INIT_CLASS(Pyramid, "Pyramid", SoShape);
}

void
Pyramid::exitClass()
{
  // Clear static data
  s_vertexProperty = NULL;
  s_vertices.clear();
  s_normals.clear();
  s_texCoords.clear();
  s_vertexIndices.clear();
  s_normalIndices.clear();
  s_texCoordIndices.clear();
  s_materialIndices.clear();

  // Deinit type id variables
  SO__NODE_EXIT_CLASS(Pyramid);
}

// Constructor
Pyramid::Pyramid()
  : m_internalShapeCache(new SoNodeDependencies)
{
  SO_NODE_CONSTRUCTOR(Pyramid);
  SO_NODE_ADD_FIELD(parts, (ALL));
  SO_NODE_ADD_FIELD(baseWidth, (2.0f));
  SO_NODE_ADD_FIELD(baseDepth, (2.0f));
  SO_NODE_ADD_FIELD(height, (2.0f));

  // Set up static values and strings for the "parts"
  // enumerated type field. This allows the SoSFEnum class to
  // read values for this field. For example, the first line
  // below says that the first value (index 0) has the value
  // SIDES (defined in the header file) and is represented in
  // the file format by the string "SIDES".
  SO_NODE_DEFINE_ENUM_VALUE(Part, SIDES);
  SO_NODE_DEFINE_ENUM_VALUE(Part, BASE);
  SO_NODE_DEFINE_ENUM_VALUE(Part, ALL);

  // Copy static information for "parts" enumerated type field
  // into this instance. 
  SO_NODE_SET_SF_ENUM_TYPE(parts, Part);

  // If this is the first time the constructor is called, set
  // Set up the static buffers representing the geometry of the pyramid
  if (SO_NODE_IS_FIRST_INSTANCE())
  {
    // Vertex positions
    s_vertices.resize(5);
    s_vertices[0].setValue(0.0f, 1.0f, 0.0f);
    s_vertices[1].setValue(-1.0f, -1.0f, 1.0f);
    s_vertices[2].setValue(1.0f, -1.0f, 1.0f);
    s_vertices[3].setValue(1.0f, -1.0f, -1.0f);
    s_vertices[4].setValue(-1.0f, -1.0f, -1.0f);

    // Vertex normals
    s_normals.resize(5);
    float invRoot5 = static_cast<float>(1.0 / sqrt(5.0));
    float invRoot5Twice = 2.0f * invRoot5;
    s_normals[0].setValue(0.0f, invRoot5, invRoot5Twice); // front normal
    s_normals[1].setValue(invRoot5Twice, invRoot5, 0.0f);  // right normal
    s_normals[2].setValue(0.0F, invRoot5, -invRoot5Twice); // rear normal
    s_normals[3].setValue(-invRoot5Twice, invRoot5, 0.0f); // left normal
    s_normals[4].setValue(0.0f, -1.0f, 0.0f);              // base normal

                                                           // Vertex texture coordinates
    s_texCoords.resize(11);
    s_texCoords[0].setValue(0.0f, 0.0f);
    s_texCoords[1].setValue(0.25f, 0.0f);
    s_texCoords[2].setValue(0.5f, 0.0f);
    s_texCoords[3].setValue(0.75f, 0.0f);
    s_texCoords[4].setValue(1.0f, 0.0f);
    s_texCoords[5].setValue(0.0f, 1.0f);
    s_texCoords[6].setValue(0.125f, 1.0f);
    s_texCoords[7].setValue(0.325f, 1.0f);
    s_texCoords[8].setValue(0.625f, 1.0f);
    s_texCoords[9].setValue(0.825f, 1.0f);
    s_texCoords[10].setValue(1.0f, 1.0f);

    // Vertex indices (the shape is indexed)
    // 3 indices for each of the 6 triangles => 18 indices
    s_vertexIndices.resize(18);
    s_vertexIndices[0] = 1;
    s_vertexIndices[1] = 2;
    s_vertexIndices[2] = 0;
    s_vertexIndices[3] = 2;
    s_vertexIndices[4] = 3;
    s_vertexIndices[5] = 0;
    s_vertexIndices[6] = 3;
    s_vertexIndices[7] = 4;
    s_vertexIndices[8] = 0;
    s_vertexIndices[9] = 4;
    s_vertexIndices[10] = 1;
    s_vertexIndices[11] = 0;
    s_vertexIndices[12] = 4;
    s_vertexIndices[13] = 3;
    s_vertexIndices[14] = 1;
    s_vertexIndices[15] = 1;
    s_vertexIndices[16] = 3;
    s_vertexIndices[17] = 2;

    // Normals indices (1 for each triangle)
    s_normalIndices.resize(6);
    s_normalIndices[0] = 0;
    s_normalIndices[1] = 1;
    s_normalIndices[2] = 2;
    s_normalIndices[3] = 3;
    s_normalIndices[4] = 4;
    s_normalIndices[5] = 4;

    // Texture coordinates indices
    s_texCoordIndices.resize(18);
    s_texCoordIndices[0] = 1;
    s_texCoordIndices[1] = 2;
    s_texCoordIndices[2] = 7;
    s_texCoordIndices[3] = 2;
    s_texCoordIndices[4] = 3;
    s_texCoordIndices[5] = 8;
    s_texCoordIndices[6] = 3;
    s_texCoordIndices[7] = 4;
    s_texCoordIndices[8] = 9;
    s_texCoordIndices[9] = 0;
    s_texCoordIndices[10] = 1;
    s_texCoordIndices[11] = 6;
    s_texCoordIndices[12] = 0;
    s_texCoordIndices[13] = 4;
    s_texCoordIndices[14] = 5;
    s_texCoordIndices[15] = 5;
    s_texCoordIndices[16] = 4;
    s_texCoordIndices[17] = 10;

    // Material indices (1 for each triangle)
    // Set a material index for the first 4 triangles (the sides)
    // and another one for the last 2 triangles (the base)
    s_materialIndices.resize(6);
    for (size_t i = 0; i < 4; i++)
      s_materialIndices[i] = 0;
    for (size_t i = 4; i < s_materialIndices.size(); i++)
      s_materialIndices[i] = 1;

    // Initialize static vertex property with the static buffers.
    s_vertexProperty = new SoVertexProperty;
    s_vertexProperty->normalBinding.setValue(SoVertexProperty::PER_FACE_INDEXED);
    s_vertexProperty->materialBinding.setValue(SoVertexProperty::PER_PART_INDEXED);

    s_vertexProperty->vertex.setValuesPointer(static_cast<int>(s_vertices.size()), &s_vertices[0]);
    s_vertexProperty->normal.setValuesPointer(static_cast<int>(s_normals.size()), &s_normals[0]);
    s_vertexProperty->texCoord.setValuesPointer(static_cast<int>(s_texCoords.size()), &s_texCoords[0]);
  }

  // Add dependency to the parts field. This allows the cache handler
  // to automatically invalidate itself when the field is changed.
  m_internalShapeCache->addDependency(parts);
}

// Destructor
Pyramid::~Pyramid()
{
}

// Turns on a part of the pyramid. (Convenience function.)
void
Pyramid::addPart(Part part)
{
  parts.setValue(parts.getValue() | part);
}

// Turns off a part of the pyramid. (Convenience function.)
void
Pyramid::removePart(Part part)
{
  parts.setValue(parts.getValue() & ~part);
}

// Returns whether a given part is on or off. (Convenience
// function.)
SbBool
Pyramid::hasPart(Part part) const
{
  int curParts = (parts.isIgnored() ? ALL : parts.getValue());
  return ((curParts & part) != 0);
}

// Setup internal shape index buffers depending on the value of the #parts field.
void
Pyramid::updateInternalShape()
{
  // There are 6 triangles in the pyramid's geometry.
  // The first 4 triangles represent the side triangles,
  // while the last 2 triangles represent the base.
  // minTriangle and maxTriangle represent the indices of
  // the first and the last triangle that need to be drawn.
  // Their values are used to get an offset and a size in order
  // to set the index buffers correctly.
  int minTriangle = 0;
  int maxTriangle = static_cast<int>(s_normalIndices.size());

  // If parts doesn't contain SIDES, skip the first 4 triangles
  if (!hasPart(SIDES))
    minTriangle = 4;

  // If parts doesn't contain BASE, skip the last triangles
  if (!hasPart(BASE))
    maxTriangle = 4;

  int numTriangles = maxTriangle - minTriangle;

  if (numTriangles == 0)
  {
    // Set the buffers empty if no triangle need to be drawn.
    m_triangleSet->coordIndex.setNum(0);
    m_triangleSet->normalIndex.setNum(0);
    m_triangleSet->textureCoordIndex.setNum(0);
    m_triangleSet->materialIndex.setNum(0);
  }
  else
  {
    // Set the index buffers according to the offset and size that have been computed.
    // In our case, normal and material index are related to triangles.
    m_triangleSet->normalIndex.setValuesPointer(numTriangles, &s_normalIndices[minTriangle]);
    m_triangleSet->materialIndex.setValuesPointer(numTriangles, &s_materialIndices[minTriangle]);

    // However, vertex and texture coordinate indices are related to vertices.
    // We need to compute indices offset and number first, which are 3 times the triangle values.
    int numIndices = 3 * numTriangles;
    int minIndex = 3 * minTriangle;
    m_triangleSet->coordIndex.setValuesPointer(numIndices, &s_vertexIndices[minIndex]);
    m_triangleSet->textureCoordIndex.setValuesPointer(numIndices, &s_texCoordIndices[minIndex]);
  }
}

// Retrieve internal shape representing the pyramid
// This method performs the shape's initialization during the first call
SoNode*
Pyramid::getInternalShape(SoState* state)
{
  // Initialize internal shape
  if (m_triangleSet.ptr() == NULL)
  {
    m_triangleSet = new SoIndexedTriangleSet;
    m_triangleSet->vertexProperty.setValue(s_vertexProperty.ptr());
  }

  // If the cache handler is invalid, we need an update
  if (!m_internalShapeCache->isValid(state))
  {
    updateInternalShape();
    m_internalShapeCache->updateCache(state); // make the cache valid again after the update
  }

  return m_triangleSet.ptr();
}

// This method performs the "typical" operation of a node for any action.
void
Pyramid::doAction(SoAction* action)
{
  // Access the state from the action
  SoState* state = action->getState();

  // Do a push before modifying elements as we don't want the shape to leak.
  state->push();

  // Retrieve our implementation scenegraph
  SoNode* internalShape = getInternalShape(state);

  // Here comes the central part of the implementation. We delegate the rendering part to our internal sceneGraph.
  // The forwardTraversal() method enables the action to traverse a sub-scenegraph which is not
  // actually part of the whole inventor sceneGraph. Outside of this node, this sub-scenegraph won't
  // be visible. It won't appear in current path and won't be visible to a search action for instance.
  action->forwardTraversal(internalShape);

  // Do a pop after modifying elements as we don't want the shape to leak.
  state->pop();
}

// Implements the SoGLRenderAction for the Pyramid node.
void
Pyramid::GLRender(SoGLRenderAction* action)
{
  // First see if the object is visible and should be rendered
  // now. This is a method on SoShape that checks for INVISIBLE
  // draw style, BOUNDING_BOX complexity, and delayed
  // transparency.
  if (!shouldGLRender(action))
    return;

  doAction(action);
}

// Generates triangles representing a pyramid.
void
Pyramid::generatePrimitives(SoAction* action)
{
  // Redirect action to the doAction() method because
  // all the work is done by the internal shape.
  doAction(action);
}

// Computes the bounding box and center of a pyramid.
void
Pyramid::computeBBox(SoAction* /*action*/, SbBox3f& box, SbVec3f& center)
{
  // Here, we could have called the computeBBox() method of the internal shape
  // so that all the work is done by the internal shape, but by
  // knowing the height and base size of the pyramid, we can compute
  // its bounding box much more efficiently.

  SbBox3f bbox;

  bbox.extendBy(SbVec3f(-1., -1., -1.));
  bbox.extendBy(SbVec3f(1., 1., 1.));

  box = bbox;
  center = box.getCenter();
}

// Computes and returns half-width, half-height, and half-depth
// based on current field values.
void
Pyramid::getSize(float& halfWidth, float &halfHeight, float& halfDepth) const
{
  halfWidth = (baseWidth.isIgnored() ? 1.0f : baseWidth.getValue() / 2.0f);
  halfHeight = (height.isIgnored() ? 1.0f : height.getValue() / 2.0f);
  halfDepth = (baseDepth.isIgnored() ? 1.0f : baseDepth.getValue() / 2.0f);
}
