#pragma once

#include <Inventor/SbLinear.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoIndexedTriangleSet.h>
#include <Inventor/caches/SoNodeDependencies.h>

/**
 * Pyramid
 *
 * @DESCRIPTION
 * This class implements a custom rendered object using Open Inventor nodes.
 *
 * This is a sort of "composite" custom node where new behavior is created
 * by combining standard Open Inventor nodes.  It demonstrates that custom
 * rendering can be implemented without making direct calls to OpenGL.  This
 * is the recommended method for creating new nodes in Open Inventor.
 *
 * The actual rendering is done by an SoIndexedTriangleSet node which is a
 * hidden child of the custom node.  The custom node calls forwardTraversal()
 * to traverse the hidden node (or a hidden sub-scenegraph if desired).
 */

class Pyramid : public SoShape
{
  SO_NODE_HEADER(Pyramid);

public:

  enum Part
  {                  // Pyramid parts:
    SIDES = 0x01,    // The 4 side faces
    BASE = 0x02,     // The bottom square face
    ALL = 0x03       // All parts
  };

  // Fields
  SoSFBitMask parts;         // Visible parts
  SoSFFloat   baseWidth;     // Width of base
  SoSFFloat   baseDepth;     // Depth of base
  SoSFFloat   height;        // Height, base to apex

  // Initializes this class
  static void initClass();
  static void exitClass();

  // Constructor
  Pyramid();

  // Turns on/off a part of the pyramid. (Convenience)
  void addPart(Part part);
  void removePart(Part part);

  // Returns whether a given part is on or off. (Convenience)
  SbBool hasPart(Part part) const;

protected:
  // This implements the GL rendering action. We will inherit
  // all other action behavior, including rayPick(), which is
  // defined by SoShape to pick against all of the triangles
  // created by generatePrimitives.
  virtual void GLRender(SoGLRenderAction* action);

  // Generates triangles representing a pyramid
  virtual void generatePrimitives(SoAction* action);

  // This computes the bounding box and center of a pyramid. It
  // is used by SoShape for the SoGetBoundingBoxAction and also
  // to compute the correct box to render or pick when
  // complexity is BOUNDING_BOX. Note that we do not have to
  // define a getBoundingBox() method, since SoShape already
  // takes care of that (using this method).
  virtual void computeBBox(SoAction* action, SbBox3f& box, SbVec3f& center);

  virtual void doAction(SoAction* action);

  // Destructor
  virtual ~Pyramid();

private:
  // Static data representing the geometry of the pyramid.
  // Static members are used in order to minimize memory consumption.

  // Vertex positions
  static std::vector<SbVec3f> s_vertices;

  // Vertex normals
  static std::vector<SbVec3f> s_normals;

  // Texture coordinates
  static std::vector<SbVec2f> s_texCoords;

  // Vertex indices
  static std::vector<int32_t> s_vertexIndices;

  // Normal indices
  static std::vector<int32_t> s_normalIndices;

  // Texture Coordinates indices
  static std::vector<int32_t> s_texCoordIndices;

  // Material indices
  static std::vector<int32_t> s_materialIndices;

  // static vertex property node containing the geometry of the pyramid
  static SoRef<SoVertexProperty> s_vertexProperty;

  // Internal shape representing the pyramid
  SoRef<SoIndexedTriangleSet> m_triangleSet;

  // Retrieve internal shape representing the pyramid
  SoNode* getInternalShape(SoState* state);

  // Update internal shape geometry depending on the value of the #parts field.
  void updateInternalShape();

  // Computes and returns half-width, half-height, and
  // half-depth based on current field values
  void getSize(float& halfWidth, float& halfHeight, float& halfDepth) const;

  // This cache handler is used as a flag to check whether or not the internal shape needs to be rebuilt.
  // It is needed to add a dependency on the shape's fields, so that the cache automatically invalidates
  // itself when one of the fields is changed.
  // In our case, we need a dependancy on the #parts field.
  SoRef<SoNodeDependencies> m_internalShapeCache;
};
