#include <Inventor/Xt/SoXt.h>
#include <Inventor/SbEventHandler.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoSeparator.h>
/*
 *    This example shows how to use the SbEventHandler class to define a custom
 *    event, send an event and handle an event. The SceneBuilder class is simple
 *    but represents a realistic application task.  It builds a scene graph by
 *    adding SoCone nodes one at a time. It defines 2 custom events: onBuildScene
 *    and onCreateCone. The onBuildScene event is sent exactly once, when the
 *    scene build begins.  The onCreateCone event is sent every time a cone object
 *    is added to the scene graph.  So the application could use this event to,
 *    for example, implement a progress bar.  The application sets some callbacks
 *    to handle these events, showing the different types of callback that
 *    SbEventHandler supports.
 */

/** Small class used to illustrate usage of SbEventHandler.
 * This class contains 2 events: onBuildScene, which is raised
 * when buildScene() method is called, and onCreateCone which is raised
 * each time a cone is created. It can be used to display progress of scene creation. */
class SceneBuilder
{
public:

  /** Event raised when scene building start.
   * The given argument is the progress of creation in [0, 100].
   * For this event, it is always 0.0. */
  SbEventHandler<float> onBuildScene;

  /** Event raised each time a cone is created.
   * The given argument is the progress of creation in [0, 100]. */
  SbEventHandler<float> onCreateCone;

  /** Launch scene creation */
  SoSeparator* buildScene()
  {
    // Call the onBuildScene event with 0.0 as argument.
    onBuildScene(0.0f);

    SoSeparator* sep = new SoSeparator;
    const int numCones = 10;
    for ( int i = 1; i <= numCones; i++ )
    {
      SoTranslation* translation = new SoTranslation;
      translation->translation.setValue(2, 0, 0);
      sep->addChild(translation);
      sep->addChild(new SoCone);

      // Call the onCreateCone event with "progress" as argument.
      float progress = 100.0f * i / numCones;
      onCreateCone(progress);
    }

    return sep;
  }
};

/** Dummy class used to illustrate how to connect an event to a member function */
class CustomClass
{
public:
  // a member function
  void
  buildSceneCallback( float /*progress*/ )
  {
    std::cout << "Build scene" << std::endl;
  }
  // a static function
  static void
  buildSceneCallbackStatic( float /*progress*/ )
  {
    std::cout << "Build scene static" << std::endl;
  }
};

// a free function
void createConeCallback(float progress)
{
  std::cout << "progress: " << progress << "%" << std::endl;
}

int
main(int /*argc*/, char **argv)
{
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL)
    exit(1);

  SceneBuilder sceneBuilder;
  CustomClass customClass;

  // connect onBuildScene event to myClass:buildSceneCallback member function
  sceneBuilder.onBuildScene.add(customClass, &CustomClass::buildSceneCallback);
  // You can also connect the event to a static method. This doesn't erase the previous connection.
  sceneBuilder.onBuildScene.add(CustomClass::buildSceneCallbackStatic);

  // connect onCreateCone to createConeCallback free function
  sceneBuilder.onCreateCone.add(createConeCallback);

  // Launch processing
  SoRef<SoSeparator> root = sceneBuilder.buildScene();

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root.ptr());
  myViewer->setTitle("Volume rendering");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root = NULL;
  SoXt::finish();

  return 0;
}


