// 
// Here is an example of using SoExtSelection for fast selection of large number 
// of primitives (shape details).
// This example displays a set of NUM_POINTS random points laying along a sphere. 
// Only points in front of the sphere are selectable (facing the camera).
// For this we *do not* use any more the VISIBLE_SHAPES lasso mode because :
//  - it does not allow to retrieve selection details with the primitive callbacks.
//  - it does not allow to select overlapping shapes (e.g. points overlapping due to 
//  display resolution)
// Highlighting is done by changing explicitely individual point color 
// (we don't use here SoXXXHighlightRenderAction).
//------------------------------------------------------------------------------

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoMarkerSet.h>

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/SoPrimitiveVertex.h>
#include <Inventor/details/SoPointDetail.h>

#include <Inventor/nodes/SoExtSelection.h>



#define NUM_POINTS 5000
// ... number of random points

#define MARKER_PRIMITIVE SoPointSet
// ...Possible alternative : #define MARKER_PRIMITIVE SoMarkerSet

#define RAND_01() (rand()/(float)RAND_MAX)

// Some predefined colors
SbColor redColor_(1.0f, 0.0f, 0.0f);
SbColor greenColor_(0.0f, 1.0f, 0.0f);
SbColor blueColor_(0.0f, 0.0f, 1.0f);
SbColor greyColor_(0.6f, 0.6f, 0.6f);

// Selection status for each point
SbBool primitiveSelected[NUM_POINTS];

// Point coordinate storage 
SoVertexProperty *myVertexProp=NULL;

SoXtExaminerViewer *viewer;
SoExtSelection* root;


// Pick filter callback : just to prevent shape selection
SoPath *pickFilterCB(void *, const SoPickedPoint *) // userData pick
{
  return NULL;
  // ... prevents objects from being added to the selection list
  // Here we just want to handle primitive-level selection
}

// Primitive filter callback
// This callback function is called whenever a point primitive lays inside the selection area.
// The returned flags indicates whether this primitive is considered for selection, allowing 
// filtering.
SbBool 
soExtSelectionPointCB(void *, SoCallbackAction *, const SoPrimitiveVertex *v)
{
  const SoDetail * detail = v->getDetail();
  int	coordIndex = ((SoPointDetail*)detail)->getCoordinateIndex();
  primitiveSelected[coordIndex] = TRUE;

  return FALSE;  // continue examining next primitives
}

// Start callback : reset selection state for points
void 
startCB(void *, SoSelection *)
{
	for (int i=0; i < NUM_POINTS; i++)
		primitiveSelected[i]= FALSE;
}


// Finish callback : highlight primitives surrounded by the lasso
// We consider only frontfacing points over the sphere.
void
finishCB(void *, SoSelection *)
{
	int i;
	// retrieve the camera dirrection vector
	SbVec3f cameraVector;
	SoCamera * camera = viewer->getCamera();
	SbRotation orient = camera->orientation.getValue(); // retrieve camera orientation
	orient.multVec(SbVec3f(0., 0., -1.), cameraVector); // apply orient to the default camera direction
	
	uint32_t *packedColor = myVertexProp->orderedRGBA.startEditing();
	
	for (i=0; i < NUM_POINTS; i++) {
		SbVec3f pointVector = myVertexProp->vertex[i]; // retrieve point's coordinates
		// We assume here that the center of the sphere is 0 0 0
		// otherwise we should add : pointVector = pointVector - sphereCenter;
		// The dot product between cameraVector and pointVector tells us whether the point is facing 
		// the camera or is behind the sphere :
		
		if (primitiveSelected[i] && cameraVector.dot(pointVector) <= 0.0f)
			packedColor[i] = redColor_.getPackedValue();
		else
			packedColor[i] = greyColor_.getPackedValue();
	}
	
	myVertexProp->orderedRGBA.finishEditing();
}

SoSeparator*
readIvFile (const char* filename)
{
  SoInput input;
  if (!input.openFile (filename))  return NULL;
  SoSeparator* sep = SoDB::readAll (&input);
  input.closeFile ();
  return sep;
}

int 
main(int, char **argv)
{
  // Initialize Inventor and Xt ------------------------------------------------
  Widget myWindow = SoXt::init(argv[0]);
  if(myWindow == NULL) 
    exit(1);
  
  // Selection -----------------------------------------------------------------
  viewer = new SoXtExaminerViewer(myWindow);
  root = new SoExtSelection;
  root->ref();
  root->useFastEditing(TRUE);
  root->animateLasso = TRUE;
  
  // lasso selection parameters :
  root->lassoColor = redColor_;
  root->policy.setValue(SoSelection::SINGLE);
  root->lassoPolicy.setValue(SoExtSelection::PART);
  // ... PART is required to be able to get all detailed selected primitives 
  // in shapes.
  
  root->lassoType.setValue(SoExtSelection::LASSO);
  
  // Define callbacks 
  root->addStartCallback(startCB);
  root->addFinishCallback(finishCB);
  root->setPickFilterCallback(pickFilterCB);
  
  // Some inner sphere 
  SoSeparator *sceneSep = readIvFile("$OIVHOME/data/models/geography/earth.iv");

  if (sceneSep == NULL) {
    return 0;
  }
  root->addChild(sceneSep);
  
  SoSeparator *pointSep = new SoSeparator;
  root->addChild(pointSep);

  // Outer spherical point set
  SoBaseColor *pointsColor = new SoBaseColor;
  pointsColor ->rgb.setValue(greyColor_);
  pointSep->addChild(pointsColor);
  
  //
  // -- Fill coordinates and colors for the spherical point set --
  //
  myVertexProp= new SoVertexProperty;
  MARKER_PRIMITIVE *myPointSet= new MARKER_PRIMITIVE;
  myPointSet->vertexProperty.setValue(myVertexProp);
  myPointSet->numPoints.setValue(NUM_POINTS);
  myVertexProp->vertex.setNum(NUM_POINTS);
  myVertexProp->orderedRGBA.setNum(NUM_POINTS);
  myVertexProp->materialBinding.setValue(SoVertexProperty::PER_VERTEX);
  
  root->setPointFilterCallback(soExtSelectionPointCB);
  
  pointSep->addChild(myPointSet);
  SbVec3f *coords = myVertexProp->vertex.startEditing();
  uint32_t *colors = myVertexProp->orderedRGBA.startEditing();
  
  // Create points laying on a sphere
  for(int j=0; j < NUM_POINTS; j++) {
    // random spherical coordinates
    double theta, psi;
    theta = RAND_01() * M_PI * 2;
    psi = (RAND_01()-0.5) * M_PI;
    // convert to cartesian
    float x, y, z;
    x = (float)(cos(theta) * cos(psi));
    y = (float)(sin(theta) * cos(psi));
    z = (float)sin(psi);
    
    // fill point coordinates and color
    coords[j].setValue(x,y,z);
    colors[j]  = greyColor_.getPackedValue();
  }
  myVertexProp->orderedRGBA.finishEditing();
  myVertexProp->vertex.finishEditing();
  
  
  viewer->setSceneGraph(root);
  
  // Exemple of using the SoExtSelection::select(...contour...) method .
  // This simulates the selection of points laying in the upper right corner of 
  // the visible spherical point set.
  
  // Contour for lasso
  const int numLassoPoints = 5;
  static SbVec2f vec[numLassoPoints] = {
    SbVec2f(0.f, 0.f),
    SbVec2f(0.f, 1.f),
    SbVec2f(1.f, 1.f),
    SbVec2f(1.f, 0.f),
    SbVec2f(0.f, 0.f)
  };
  
  root->select(viewer->getSceneManager()->getSceneGraph(), 
    numLassoPoints, vec, viewer->getViewportRegion(), FALSE);
  
  viewer->setTitle("Detail selection");
  viewer->setViewing(FALSE);
  viewer->show();
  
  
  // Run the application -------------------------------------------------------
  SoXt::show(myWindow);
  SoXt::mainLoop();
 
  root->unref();
  delete viewer;
  SoXt::finish();

  return 0;
}


