//
// This sample program shows how to use the new SoExtSelection node. This node performs
// lasso or rectangle selection.
//
//------------------------------------------------------------------------------
// Include files

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/actions/SoHaloHighlightRenderAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/nodes/SoWWWInline.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoExtSelection.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoTransform.h>

#include <DialogViz/SoDialogVizAll.h>

//------------------------------------------------------------------------------
// Globals
SoXtExaminerViewer *viewer;
SoExtSelection *root;

// Dialog Auditors
class SelectionPolicyAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    switch ( cpt->selectedItem.getValue() )
    {
    // Single policy
    default:
    case 0:
      root->policy = SoSelection::SINGLE;
      break;
    // Toggle policy
    case 1:
      root->policy = SoSelection::TOGGLE;
      break;
    // Shift policy
    case 2:
      root->policy = SoSelection::SHIFT;
      break;
    // Disable policy
    case 3:
      root->policy = SoSelection::DISABLE;
      break;
    }
  }
};

class LassoTypeAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    switch ( cpt->selectedItem.getValue() )
    {
    // No Lasso
    default:
    case 0:
      root->lassoType = SoExtSelection::NOLASSO;
      break;
    // Lasso
    case 1:
      root->lassoType = SoExtSelection::LASSO;
      break;
    // Rectangle
    case 2:
      root->lassoType = SoExtSelection::RECTANGLE;
      break;
    case 3:
      root->lassoType = SoExtSelection::ELLIPSE;
      break;
    }
  }
};

class LassoPolicyAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    switch ( cpt->selectedItem.getValue() )
    {
    // Full Bounding Box
    default:
    case 0:
      root->lassoPolicy = SoExtSelection::FULL_BBOX;
      break;
    // Part bounding box
    case 1:
      root->lassoPolicy = SoExtSelection::PART_BBOX;
      break;
    // Full
    case 2:
      root->lassoPolicy = SoExtSelection::FULL;
      break;
    // Part
    case 3:
      root->lassoPolicy = SoExtSelection::PART;
      break;
    }
  }
};

class LassoModeAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    switch ( cpt->selectedItem.getValue() )
    {
    // Visible shapes
    case 0:
      root->lassoMode = SoExtSelection::VISIBLE_SHAPES;
      break;

    // All shapes
    default:
    case 1:
      root->lassoMode = SoExtSelection::ALL_SHAPES;
      break;
    }
  }
};

//******************************************************************************
// Entry point of the application
int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt ------------------------------------------------
  Widget myWindow = SoXt::init(argv[0]);
  if(myWindow == NULL) 
    exit(1);

  SoDialogViz::init();

  // Selection -----------------------------------------------------------------
  root = new SoExtSelection;
  root->ref();
  root->useFastEditing(TRUE);
  root->lassoColor = SbColor(0, 1, 0);
  root->lassoPattern = 0xE0E0;
  
  root->animateLasso = TRUE;
  root->policy = SoSelection::SINGLE;
  root->lassoType = SoExtSelection::RECTANGLE;
  root->lassoPolicy = SoExtSelection::FULL_BBOX;
  root->lassoMode = SoExtSelection::ALL_SHAPES;

  // Scene graph ---------------------------------------------------------------
  if (argc != 2) {
    // build test scene
    SoTransform *t = new SoTransform;
    SoMaterial *m = new SoMaterial;

    SoSeparator *pSep = new SoSeparator;
    pSep->setName("Assy");
    pSep->ref();

    t->translation.setValue(1.5, 1.5, 0);
    m->diffuseColor.setValue(1, 0, 0);
    pSep->addChild(t);
    pSep->addChild(m);
    SoSphere *sphere = new SoSphere;
    sphere->setName("Sphere");
    pSep->addChild(sphere);

    t = new SoTransform;
    t->translation.setValue(-3, 0, 0);
    m = new SoMaterial;
    m->diffuseColor.setValue(0, 1, 0);
    pSep->addChild(t);
    pSep->addChild(m);
    SoCone *cone = new SoCone;
    cone->setName("Cone");
    pSep->addChild(cone);

    t = new SoTransform;
    t->translation.setValue(0, -3, 0);
    m = new SoMaterial;
    m->diffuseColor.setValue(0, 0, 1);
    pSep->addChild(t);
    pSep->addChild(m);
    SoCylinder *cyl= new SoCylinder;
    cyl->setName("Cylinder");
    pSep->addChild(cyl);

    t = new SoTransform;
    t->translation.setValue(3, 0, 0);
    m = new SoMaterial;
    m->diffuseColor.setValue(1, 1, 0);
    pSep->addChild(t);
    pSep->addChild(m);
    SoCube *cube = new SoCube;
    cube->setName("Cube");
    pSep->addChild(cube);

    // 2 instances of pSep
    root->addChild(pSep);

    t = new SoTransform;
    t->translation.setValue(10, 10, 0);
    root->addChild(t);
    root->addChild(pSep);

  }
  else {
    SoInput myInput;
    if (!myInput.openFile(argv[1]))
      exit(0);
    SoWWWInline::setReadAsSoFile(TRUE);

    SoSeparator *fileContent = SoDB::readAll(&myInput);
    if (fileContent == NULL)
      exit(0);
    root->addChild(fileContent);
  }

  SoHaloHighlightRenderAction* highlightRenderAction = new SoHaloHighlightRenderAction();
  highlightRenderAction->setHaloWidth( 7 );
  highlightRenderAction->setHaloSmoothFactor( 0.75f );

  // Viewer initialisation -----------------------------------------------------
  viewer = new SoXtExaminerViewer(myWindow);
  viewer->setSceneGraph(root);
  viewer->setGLRenderAction( highlightRenderAction );
  viewer->setFastEditSavePolicy( SoGLRenderAction::WHEN_NEEDED );
  viewer->redrawOnSelectionChange(root);
  viewer->setTitle("Selection Test");
  viewer->setViewing(FALSE);
  viewer->show();

  // Setup the control dialog --------------------------------------------------

  SoRef<SoTopLevelDialog> dialog = new SoTopLevelDialog;
  dialog->label.setValue( "Extended Selection Control" );

  const char* selection_policy_items[4] = { "Single", "Toggle", "Shift", "Disable" };
  SoDialogComboBox* selectionPolicy = new SoDialogComboBox;
  selectionPolicy->label.setValue( "Selection policy" );
  selectionPolicy->items.setValues( 0, 4, selection_policy_items );
  selectionPolicy->addAuditor( new SelectionPolicyAuditor );
  dialog->addChild( selectionPolicy );

  const char* lasso_type_items[4] = { "No Lasso", "Lasso", "Rectangle", "Ellipse" };
  SoDialogComboBox* lassoType = new SoDialogComboBox;
  lassoType->label.setValue( "Lasso type" );
  lassoType->items.setValues( 0, 4, lasso_type_items );
  lassoType->addAuditor( new LassoTypeAuditor );
  dialog->addChild( lassoType );

  const char* lasso_policy_items[4] = { "Full Bounding Box", "Part Bounding Box", "Full", "Part" };
  SoDialogComboBox* lassoPolicy = new SoDialogComboBox;
  lassoPolicy->label.setValue( "Lasso policy" );
  lassoPolicy->items.setValues( 0, 4, lasso_policy_items );
  lassoPolicy->addAuditor( new LassoPolicyAuditor );
  dialog->addChild( lassoPolicy );

  const char* lasso_mode_items[2] = { "Visible Shapes", "All Shapes" };
  SoDialogComboBox* lassoMode = new SoDialogComboBox;
  lassoMode->label.setValue( "Lasso mode" );
  lassoMode->items.setValues( 0, 2, lasso_mode_items );
  lassoMode->addAuditor( new LassoModeAuditor );
  dialog->addChild( lassoMode );

  dialog->buildDialog( NULL );
  dialog->show();

  // Run the application -------------------------------------------------------
  SoXt::show(myWindow);
  SoXt::mainLoop();

  dialog = NULL;
  SoDialogViz::finish();

  root->unref();
  delete viewer;
  SoXt::finish();

  return 0;
}
