/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Damien C.OGNIER ROSA (Jul 2001)
**=======================================================================*/

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoExtTexture2.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/nodes/SoRotation.h>

#include <DialogViz/SoDialogVizAll.h>

SoExtTexture2* texture1;
SoExtTexture2* texture2;
SoExtTexture2* texture3;
SoExtTexture2* texture4;

////////////////////////////////////////////////////////
// Dialog Callbacks (tunes the texture1 loading) //
////////////////////////////////////////////////////////

class TextureManualLoadAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* /*cpt*/ )
  {
    if (texture1->loadingMode.getValue() == SoExtTexture2::AUTO)
      texture1->loadingMode.setValue(SoExtTexture2::MANUAL);
    else
      texture1->loadingMode.setValue(SoExtTexture2::AUTO);
  }
};

class TextureLoadAuditor : public SoDialogPushButtonAuditor
{
public:
  void dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    if (texture1->loadingMode.getValue() == SoExtTexture2::MANUAL)
      texture1->loadTexture();
  }
};

class TextureSyncLoadAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox( SoDialogCheckBox* /*cpt*/ )
  {
    if (texture1->syncMode.getValue() == SoExtTexture2::SYNCHRONOUS)
      texture1->syncMode.setValue(SoExtTexture2::ASYNCHRONOUS);
    else
      texture1->syncMode.setValue(SoExtTexture2::SYNCHRONOUS);
  }
};

// This function is called 10 times/second
static void
rotationCallback(void* data, SoSensor*)
{
  // Rotates the scene
  SoRotation* rot = (SoRotation*) data;
  SbRotation currentRotation = rot->rotation.getValue();
  currentRotation = SbRotation(SbVec3f(0.0f, 0.0f, 1.0f), 
                               (float)(M_PI/90.0f)) * currentRotation;
  rot->rotation.setValue(currentRotation);
}

// Called when starting the loading of texture1
void
startTex1Load(void* /*userData*/, SoExtTexture2* /*tex*/)
{
  printf("Started loading the texture 1...\n");
}

// Called when finishing the loading of texture1
void
finishTex1Load(void* /*userData*/, SoExtTexture2* /*tex*/)
{
  printf("... Finished loading the texture 1.\n");
}

int
main(int argc, char **argv)
{
  // Init of thread classes for asynchronous loading
  SoDB::init();

  // Initialize Inventor and Xt
  Widget appWindow = SoXt::init(argv[0]);
  if (appWindow == NULL)
    exit(1);

  SoDialogViz::init();

  bool fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);

#ifdef _WIN32
  // Create 3 Windows for the 3 viewers.
  HWND newWindow2, newWindow3;

  RECT rect;
  GetWindowRect(appWindow, &rect);
  newWindow2 = CreateWindow("button",
                            "",
                            WS_OVERLAPPEDWINDOW | BS_OWNERDRAW,
                            rect.left + 300,          // X position
                            rect.top,                 // Y position
                            rect.right - rect.left,   // Width
                            rect.bottom - rect.top,   // Height
                            NULL,                     // Parent
                            NULL,                     // Menu
                            SoXt::getInstance(),      // App instance handle
                            NULL);                    // Window creation data

  newWindow3 = CreateWindow("button",
                            "",
                            WS_OVERLAPPEDWINDOW | BS_OWNERDRAW,
                            rect.left + 600,          // X position
                            rect.top,                 // Y position
                            rect.right - rect.left,   // Width
                            rect.bottom - rect.top,   // Height
                            NULL,                     // Parent
                            NULL,                     // Menu
                            SoXt::getInstance(),      // App instance handle
                            NULL);                    // Window creation data
#endif   

  // Each textured cube is placed under a separator to test the 
  // textureElement pop/push
  // Cube 1
  SoSeparator*   sep1 = new SoSeparator;
  SoTranslation* tr1  = new SoTranslation();
  sep1->addChild(tr1);
  tr1->translation.setValue(-1.25, -1.25, 0.);
  texture1 = NULL;
  texture1 = new SoExtTexture2();
  texture1->loadingThreadPriority.setValue(10);
  if (texture1 == NULL) printf("null\n");
  texture1->loadingMode.setValue(SoExtTexture2::MANUAL);
  texture1->filename = "$OIVHOME/examples/source/Inventor/Features/ExtTexture2/test1.jpg";
  texture1->addOnLoadStartCB((SoExtTexture2::SoExtTexture2CB*) startTex1Load , texture1);
  texture1->addOnLoadFinishCB((SoExtTexture2::SoExtTexture2CB*) finishTex1Load, texture1);
  sep1->addChild(texture1);
  sep1->addChild(new SoCube);

  // Cube 2
  SoSeparator*   sep2 = new SoSeparator;
  SoTranslation* tr2  = new SoTranslation();
  sep2->addChild(tr2);
  tr2->translation.setValue(-1.25, 1.25, 0.);
  texture2 = new SoExtTexture2;
  texture2->filename = "$OIVHOME/examples/source/Inventor/Features/ExtTexture2/test2.jpg";
  sep2->addChild(texture2);
  sep2->addChild(new SoCube);
   
  // Cube 3
  SoSeparator*   sep3 = new SoSeparator;   
  SoTranslation* tr3  = new SoTranslation();
  sep3->addChild(tr3);
  tr3->translation.setValue(1.25, -1.25, 0.);
  texture3 = new SoExtTexture2;
  texture3->filename = "$OIVHOME/examples/source/Inventor/Features/ExtTexture2/test3.jpg";
  sep3->addChild(texture3);
  sep3->addChild(new SoCube);
  
  // Cube 4
  SoSeparator*   sep4 = new SoSeparator;
  SoTranslation* tr4  = new SoTranslation();
  sep4->addChild(tr4);
  tr4->translation.setValue(1.25, 1.25, 0.);
  texture4 = new SoExtTexture2;
  texture4->filename = "$OIVHOME/examples/source/Inventor/Features/ExtTexture2/test4.jpg";
  sep4->addChild(texture4);
  sep4->addChild(new SoCube);

  SoRotation* rot = new SoRotation;

  SoTimerSensor* rotatingSensor = new SoTimerSensor(rotationCallback, rot);
  rotatingSensor->setInterval(0.1f); // scheduled once per 1/10th of second
  if (fullDemo) rotatingSensor->schedule();

  SoSeparator *root = new SoSeparator;
  root->ref();

  root->addChild(rot);
  root->addChild(sep1);
  root->addChild(sep2);
  root->addChild(sep3);
  root->addChild(sep4);

  // Initialize the viewers
  // Each viewer displays the same scene (checks OpenGL texture object sharing in1 3 contexts)
  // Viewer 1
  SoXtExaminerViewer *viewer = new SoXtExaminerViewer(appWindow);
  viewer->setSceneGraph(root);
  viewer->setTitle("ExtTexture2 Viewer 1");
  viewer->setSize(SbVec2s(300, 300));

  // Viewer 2
#ifdef _WIN32
  SoXtExaminerViewer *viewer2 = new SoXtExaminerViewer(newWindow2);
  ShowWindow(newWindow2, SW_SHOW);
#else
  SoXtExaminerViewer *viewer2 = new SoXtExaminerViewer();
#endif
  viewer2->setSceneGraph(root);
  viewer2->setTitle("ExtTexture2 Viewer 2");
  viewer2->setSize(SbVec2s(300, 300));

  // Viewer 3
#ifdef _WIN32
  SoXtExaminerViewer *viewer3 = new SoXtExaminerViewer(newWindow3);
  ShowWindow(newWindow3, SW_SHOW);
#else
  SoXtExaminerViewer *viewer3 = new SoXtExaminerViewer();
#endif
  viewer3->setSceneGraph(root);
  viewer3->setTitle("ExtTexture2 Viewer 3");
  viewer3->setSize(SbVec2s(300, 300));

  /******************************************************************
   *                                Dialog box
   ******************************************************************/
  SoRef<SoTopLevelDialog> dialog = new SoTopLevelDialog;
  dialog->label.setValue( "SoExtTexture2 Demo" );

  /* Concerns the first texture only */
  SoDialogCheckBox* tex1ManualLoad = new SoDialogCheckBox;
  tex1ManualLoad->label.setValue( "Texture 1 Manual Load" );
  tex1ManualLoad->state.setValue( TRUE );
  tex1ManualLoad->addAuditor( new TextureManualLoadAuditor );
  dialog->addChild( tex1ManualLoad );

  SoDialogPushButton* tex1Load = new SoDialogPushButton;
  tex1Load->label.setValue( "Manual loading Only" );
  tex1Load->buttonLabel.setValue( "Load texture 1" );
  tex1Load->addAuditor( new TextureLoadAuditor );
  dialog->addChild( tex1Load );

  SoDialogCheckBox* tex1SyncLoad = new SoDialogCheckBox;
  tex1SyncLoad->label.setValue( "Texture 1 Synchronous Load" );
  tex1SyncLoad->state.setValue( TRUE );
  tex1SyncLoad->addAuditor( new TextureSyncLoadAuditor );
  dialog->addChild( tex1SyncLoad );

  /* Dialog */
  dialog->width.setValue( 250 );
  dialog->height.setValue( 50 );
  dialog->windowResizedByChildren.setValue( TRUE );
  dialog->buildDialog( NULL );
  dialog->show();

  // In Inventor 2.1, if the machine does not have hardware texture
  // mapping, we must override the default drawStyle to display textures.
  viewer->setDrawStyle(SoXtViewer::STILL, SoXtViewer::VIEW_AS_IS);
  viewer->show();
   
  viewer2->setDrawStyle(SoXtViewer::STILL, SoXtViewer::VIEW_AS_IS);
  viewer2->show();

  viewer3->setDrawStyle(SoXtViewer::STILL, SoXtViewer::VIEW_AS_IS);
  viewer3->show();

  SoXt::show(appWindow);
  SoXt::mainLoop();

  delete rotatingSensor;

  dialog = NULL;
  SoDialogViz::finish();

  delete viewer;
  delete viewer2;
  delete viewer3;

  root->unref();
  SoXt::finish();

  SoDB::finish();

  return 0;
}

