/*================================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),       ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                ***
***                                                                           ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                      ***
***                                                                           ***
***                        RESTRICTED RIGHTS LEGEND                           ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        ***
***                                                                           ***
***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                   ***
***                        BORDEAUX, FRANCE                                   ***
***                      ALL RIGHTS RESERVED                                  ***
================================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoColorMask.h>
#include <Inventor/nodes/SoDepthOffset.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFile.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoSeparator.h>

#include <DialogViz/auditors/SoDialogAuditor.h>
#include <DialogViz/dialog/SoDialogCustom.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>
#include <DialogViz/dialog/SoMenuFileSelection.h>


SoXtExaminerViewer* g_viewer = nullptr;
SoFile* g_modelFile = nullptr;

// Read a scene graph file and return the corresponding separator
SoSeparator* readFile( const char* filename )
{
  SoInput mySceneInput;
  if ( !mySceneInput.openFile( filename ) )
  {
    std::cerr << "Cannot open file " << filename << std::endl;
    return NULL;
  }

  SoSeparator* myGraph = SoDB::readAll( &mySceneInput );
  if ( myGraph == NULL )
  {
    std::cerr << "Problem reading file" << std::endl;
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

// Class used to interpret events from the GUI.
class AuditorClass : public SoDialogAuditor
{
  void menuFileSelection( SoMenuFileSelection* cpt )
  {
    if ( cpt->auditorID.getValue() == "open_model" )
    {
      g_modelFile->name.setValue( cpt->fileDirectory.getValue() + "/" + cpt->filename.getValue() );
      g_viewer->viewAll();
    }
  }
};

SoNode*
buildSceneGraph()
{
  SoSeparator* scene = new SoSeparator;
  scene->addChild( new SoGradientBackground );

  SoSeparator* firstPassSeparator = new SoSeparator;
  scene->addChild( firstPassSeparator );

  // For the first pass, use a ColorMask with all fields to FALSE in order to
  // draw only into the depth buffer
  SoColorMask* colorMask = new SoColorMask;
  colorMask->red = FALSE;
  colorMask->green = FALSE;
  colorMask->blue = FALSE;
  colorMask->alpha = FALSE;
  firstPassSeparator->addChild( colorMask );

  g_modelFile = new SoFile;
  g_modelFile->name = "$OIVHOME/examples/source/Inventor/Features/HiddenLine/HiddenLine_model.iv";
  firstPassSeparator->addChild( g_modelFile );

  SoSeparator* secondPassSeparator = new SoSeparator;
  scene->addChild( secondPassSeparator );

  // For the second pass, draw the model in BASE_COLOR mode and DrawStyle LINES
  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::BASE_COLOR;
  secondPassSeparator->addChild( lightModel );

  SoDrawStyle* drawStyle = new SoDrawStyle;
  drawStyle->style = SoDrawStyle::LINES;
  secondPassSeparator->addChild( drawStyle );

  // Add a DepthOffset to make sure the lines will always be drawn in front
  // of the model
  SoDepthOffset* depthOffset = new SoDepthOffset;
  secondPassSeparator->addChild( depthOffset );

  secondPassSeparator->addChild( g_modelFile );

  return scene;
}

int
main( int argc, char** argv )
{
  Widget window = SoXt::init( argv[0] );
  if ( window == nullptr )
    return EXIT_FAILURE;

  SoDialogViz::init();
  SoRef<SoTopLevelDialog> topLevelDialog =
    (SoTopLevelDialog*) SoDialogViz::loadFromFile( "$OIVHOME/examples/source/Inventor/Features/HiddenLine/HiddenLine_interface.iv" );
  topLevelDialog->buildDialog( window, TRUE );
  AuditorClass* auditor = new AuditorClass;
  topLevelDialog->label.setValue( "Hidden Line Rendering" );
  topLevelDialog->addAuditor( auditor );
  topLevelDialog->show();

  SoRef<SoNode> root = buildSceneGraph();

  // Load model if specified
  if ( argc > 1 )
    g_modelFile->name.setValue( argv[1] );

  SoWidget viewerWidget = static_cast<SoDialogCustom*>(topLevelDialog->searchForAuditorId( "VIEWER" ))->getWidget();
  g_viewer = new SoXtExaminerViewer( viewerWidget );
  g_viewer->setBorder( FALSE );
  g_viewer->setDecoration( FALSE );
  g_viewer->setTransparencyType( SoGLRenderAction::NO_SORT );
  g_viewer->setAntialiasing( 1.0f, SoSceneManager::FSAA );
  g_viewer->setSceneGraph( root.ptr() );
  g_viewer->viewAll();
  g_viewer->show();
  SoXt::show( window );

  SoXt::mainLoop();

  delete auditor;

  topLevelDialog = nullptr;
  root = nullptr;

  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}
