/*=============================================================================
** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),       **
**              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                **
**                                                                           **
**  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  **
**  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  **
**  WRITTEN AUTHORIZATION OF FEI S.A.S.                                      **
**                                                                           **
**                        RESTRICTED RIGHTS LEGEND                           **
**  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS **
**  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN **
**  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT **
**  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN **
**  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        **
**                                                                           **
**                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                   **
**                             MERIGNAC, FRANCE                              **
**                           ALL RIGHTS RESERVED                             **
=============================================================================*/


#include "HighLightWidget.h"
#include <Inventor/actions/SoBoxHighlightRenderAction.h>
#include <Inventor/actions/SoColorHighlightRenderAction.h>
#include <Inventor/actions/SoHaloHighlightRenderAction.h>
#include <Inventor/actions/SoLineHighlightRenderAction.h>
#include <QColorDialog>
#include "ui_HighLightWidget.h"

HighLightWidget::HighLightWidget( QWidget* parent )
  : QWidget( parent )
  , ui( new Ui::HighLightWidget )
{
  // ui setting
  ui->setupUi( this );
  ui->typeCombo->addItem( "Line", LINE );
  ui->typeCombo->addItem( "Box", BOX );
  ui->typeCombo->addItem( "Color", COLOR );
  ui->typeCombo->addItem( "Halo", HALO );

  ui->widthSlider->setRange( 1, 10 );
  ui->widthSpinBox->setRange( 1, 10 );

  ui->smoothFactorSlider->setRange( 0, 10 );
  ui->smoothFactorSpinBox->setRange( 0, 1 );
  ui->smoothFactorSpinBox->setSingleStep( 0.1 );

  // Sync Sliders <=> SpinBoxes
  connect( ui->widthSlider, &QSlider::valueChanged, ui->widthSpinBox, &QSpinBox::setValue );
  connect( ui->widthSpinBox, QOverload<int>::of( &QSpinBox::valueChanged ), ui->widthSlider, &QSlider::setValue );

  connect( ui->smoothFactorSlider, &QSlider::valueChanged, ui->smoothFactorSpinBox, [this]( int value ) {
    ui->smoothFactorSpinBox->setValue( value / 10. );
  } );

  connect( ui->smoothFactorSpinBox, QOverload<double>::of( &QDoubleSpinBox::valueChanged ), ui->smoothFactorSlider, [this]( double value ) {
    ui->smoothFactorSlider->setValue( value * 10 );
  } );

  // ColorDialog
  connect( ui->colorButton, &QPushButton::clicked, [this]() {
    float hsv[3];
    m_currentColor.getHSVValue( hsv );
    QColor currentColor = QColor::fromHsvF( hsv[0], hsv[1], hsv[2] );

    QColor newColor = QColorDialog::getColor( currentColor, this );
    if ( newColor != currentColor )
    {
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
      float tab[3];
#else
      qreal tab[3];
#endif
      newColor.getHsvF( tab, tab + 1, tab + 2 );
      m_currentColor.setHSVValue( tab[0], tab[1], tab[2] );
      emit modified();
    }
  } );

  // modified() signal emissions
  connect( ui->typeCombo, QOverload<int>::of( &QComboBox::currentIndexChanged ), this, [this]( int value ) {
    switch ( value )
    {
    default:
    case HALO:
      ui->widgetWidth->setEnabled( true );
      ui->widgetSmooth->setEnabled( true );
      ui->alwaysVisibleCB->setEnabled( true );
      break;
    case COLOR:
      ui->widgetWidth->setEnabled( false );
      ui->widgetSmooth->setEnabled( false );
      ui->alwaysVisibleCB->setEnabled( false );
      break;
    case BOX:
      ui->widgetWidth->setEnabled( true );
      ui->widgetSmooth->setEnabled( false );
      ui->alwaysVisibleCB->setEnabled( false );
      break;
    case LINE:
      ui->widgetWidth->setEnabled( true );
      ui->widgetSmooth->setEnabled( false );
      ui->alwaysVisibleCB->setEnabled( false );
      break;
    }
    emit modified();
    setCursor(QCursor(Qt::ArrowCursor)); //Fix QCombobox bug
  } );

  connect( ui->alwaysVisibleCB, &QCheckBox::toggled, this, &HighLightWidget::modified );
  connect( ui->widthSpinBox, QOverload<int>::of( &QSpinBox::valueChanged ), this, &HighLightWidget::modified );
  connect( ui->smoothFactorSpinBox, QOverload<double>::of( &QDoubleSpinBox::valueChanged ), this, &HighLightWidget::modified );

  // Default setting
  ui->typeCombo->setCurrentIndex( ui->typeCombo->findData( HALO ) );
  SoHaloHighlightRenderAction dummyAction;
  ui->widthSpinBox->setValue( dummyAction.getHaloWidth() );
  ui->smoothFactorSpinBox->setValue( dummyAction.getHaloSmoothFactor() );
  ui->alwaysVisibleCB->setChecked( dummyAction.isHaloAlwaysVisible() );
  m_currentColor = dummyAction.getHaloColor();
}

SoGLRenderAction*
HighLightWidget::createAction(SoGLRenderAction* currentAction) const
{
  SoGLRenderAction* action = currentAction;

  switch ( ui->typeCombo->currentData().toInt() )
  {
  case HALO: {
    SoHaloHighlightRenderAction* tmpAct = nullptr;
    if (action && action->isOfType<SoHaloHighlightRenderAction>())
      tmpAct = static_cast<SoHaloHighlightRenderAction*>(action);
    else
      tmpAct = new SoHaloHighlightRenderAction;
    tmpAct->setHaloAlwaysVisible( ui->alwaysVisibleCB->isChecked() );
    tmpAct->setHaloWidth( ui->widthSpinBox->value() );
    tmpAct->setHaloSmoothFactor( ui->smoothFactorSpinBox->value() );
    tmpAct->setHaloColor( m_currentColor );
    action = tmpAct;
    break;
  }
  case COLOR: {
    SoColorHighlightRenderAction* tmpAct = nullptr;
    if (action && action->isOfType<SoColorHighlightRenderAction>())
      tmpAct = static_cast<SoColorHighlightRenderAction*>(action);
    else
      tmpAct = new SoColorHighlightRenderAction;
    tmpAct->setColor( m_currentColor );
    action = tmpAct;
    break;
  }
  case BOX: {
    SoBoxHighlightRenderAction* tmpAct = nullptr;
    if (action && action->isOfType<SoBoxHighlightRenderAction>())
      tmpAct = static_cast<SoBoxHighlightRenderAction*>(action);
    else
      tmpAct = new SoBoxHighlightRenderAction;
    tmpAct->setLineWidth( ui->widthSpinBox->value() );
    tmpAct->setColor( m_currentColor );
    action = tmpAct;
    break;
  }
  case LINE: {
    SoLineHighlightRenderAction* tmpAct = nullptr;
    if (action && action->isOfType<SoLineHighlightRenderAction>())
      tmpAct = static_cast<SoLineHighlightRenderAction*>(action);
    else
      tmpAct = new SoLineHighlightRenderAction;
    tmpAct->setLineWidth( ui->widthSpinBox->value() );
    tmpAct->setColor( m_currentColor );
    action = tmpAct;
    break;
  }
  }

  // Copy parameters
  if (currentAction && currentAction != action)
  {
    action->setTransparencyType(currentAction->getTransparencyType());
  }

  return action;
}

HighLightWidget::~HighLightWidget()
{
  delete ui;
}
