#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoDB.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/actions/SoToHTMLAction.h>
#include <Inventor/helpers/SbFileHelper.h>

typedef struct cbdata {
  SoXtExaminerViewer *vwr;
  char *filename;
  SoNode *scene;
} callbackData;

static SbBool
generateHTMLFile(SoNode *root, FILE *file, const SbString dirName,
                 SoXtExaminerViewer *, //viewer,
                 int) //printerDPI)
{
  SoToHTMLAction *toHTMLAction = new SoToHTMLAction;
  SbViewportRegion vpRegion(800,400);
  toHTMLAction->setViewportRegion(vpRegion);

  toHTMLAction->setImageURLName( SbString(dirName+"vsg.jpg").toLatin1() );
  //toHTMLAction->setRegionShapeType(SoToHTMLAction::RECTANGLE);
  toHTMLAction->setHTMLFilePointer(file);
  toHTMLAction->apply(root);
  return TRUE;
}

static void
processKeyEvents(void *data, SoEventCallback *cb)
{
  if (SO_KEY_PRESS_EVENT(cb->getEvent(), P)) { 
    callbackData *cbData = (callbackData *)data;
    FILE *myFile = fopen(cbData->filename, "wb");

    if (myFile == NULL) {
      fprintf(stderr, "Cannot open output file\n");
      exit(1);
    }
    printf("Generate HTML file... ");
    fflush(stdout);

    if (!generateHTMLFile(cbData->scene, myFile,
                          SbFileHelper::getDirName(cbData->filename),
                          cbData->vwr, 75)) {
      fprintf(stderr, "Cannot generate HTML file\n");
      fclose(myFile);
      exit(1);
    }
    fclose(myFile);
    printf("  ...done.\n");
    fflush(stdout);
    cb->setHandled(); 
  } 
}

int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt
  Widget appWindow = SoXt::init(argv[0]);
  if (appWindow == NULL)
    exit(1);

  // Verify the command line arguments
  if (argc != 3) {
    fprintf(stderr, "Usage: %s infile.iv outfile.html\n", argv[0]);
    exit(1); 
  }

  printf("To generate the HTML file of the scene: press the 'p' key while in picking mode\n");

  // Make a scene containing an event callback node
  SoSeparator *root = new SoSeparator;
  SoEventCallback *eventCB = new SoEventCallback;
  root->ref();
  root->addChild(eventCB);

  // Read the geometry from a file and add to the scene
  SoInput myInput;
  if (!myInput.openFile(argv[1]))
    exit (1);

  SoSeparator *geomObject = SoDB::readAll(&myInput);
  if (geomObject == NULL)
    exit (1);
  root->addChild(geomObject);

  SoXtExaminerViewer *viewer =
    new SoXtExaminerViewer(appWindow, NULL, TRUE, 
                           SoXtExaminerViewer::BUILD_ALL, 
                           SoXtExaminerViewer::EDITOR);
  viewer->setSceneGraph(root);

  viewer->setTitle("Print to HTML");

  // Setup the event callback data and 
  // routine for performing the HTML Output
  callbackData *data = new callbackData;
  data->vwr = viewer;
  data->filename = argv[2];
  data->scene = viewer->getSceneGraph();
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(),
                            processKeyEvents, data);
  viewer->show();

  SoXt::show(appWindow);
  SoXt::mainLoop();

  root->unref();
  delete viewer;
  SoXt::finish();

  return 0;
}


