/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
/*--------------------------------------------------------------
 *  This example builds several lineSet to expose 
 *  SoDrawStyle line pattern scale factor enhancement.
 *------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTranslation.h>

#include <DialogViz/SoDialogVizAll.h>

#define SHAPE_NAME      "A_Shape"
#define DRAWSTYLE_NAME  "A_DrawStyle"

#define DEFAULT_PATTERN 2
#define DEFAULT_FACTOR  1

// Material Identifiers
enum patternID {
  FULL_PATTERN,
  DOTL_PATTERN
};

const unsigned short
patterns[5] = {
  0xFFFF, // solid
  0x000F, // dash 
  0x1111, // dot
  0x041F, // dash-dot
  0x111F, // dash-dot-dot
};		

const char *
strings[5] = {
  "solid", 
  "dash", 
  "dot", 
  "dash-dot", 
  "dash-dot-dot" 
};		

static SoSeparator *makeRectangle();
static SoSeparator *makeDiamond();
static SoSeparator *makeCircle();

class ScaleFactorAuditor : public SoDialogIntegerSliderAuditor
{
public:
  static SoDrawStyle* s_drawStyle;
  void dialogIntegerSlider( SoDialogIntegerSlider* cpt )
  {
    if ( s_drawStyle != NULL )
      s_drawStyle->linePatternScaleFactor = cpt->value.getValue();
  }
};
SoDrawStyle* ScaleFactorAuditor::s_drawStyle = NULL;

class PatternChoiceAuditor : public SoDialogChoiceAuditor
{
public:
  static SoDrawStyle* s_drawStyle;
  void dialogChoice( SoDialogChoice* cpt )
  {
    if ( s_drawStyle != NULL )
      s_drawStyle->linePattern = patterns[cpt->selectedItem.getValue()];
  }
};
SoDrawStyle* PatternChoiceAuditor::s_drawStyle = NULL;

class ShapeTypeAuditor : public SoDialogChoiceAuditor
{
public:
  static SoSeparator* s_root;
  void dialogChoice( SoDialogChoice* cpt )
  {
    if ( s_root == NULL )
      return;

    SoNode* oldShape = SoNode::getByName( SHAPE_NAME );
    SoNode* newNode;

    switch ( cpt->selectedItem.getValue() )
    {
    case 0: // Rectangle
      newNode = makeRectangle();
      break;
    case 1: // Diamond
      newNode = makeDiamond();
      break;
    case 2: // Circle
      newNode = makeCircle();
      break;
    default:
      newNode = makeRectangle();
      break;
    }

    newNode->setName( SHAPE_NAME );
    s_root->replaceChild( oldShape, newNode );
  }
};
SoSeparator* ShapeTypeAuditor::s_root = NULL;

class CustomPatternAuditor : public SoDialogEditTextAuditor
{
public:
  static SoDrawStyle* s_drawStyle;
  void  dialogEditText (SoDialogEditText *cpt)
  {
    if ( s_drawStyle == NULL )
      return;

    char *patternString = strdup(cpt->editText.getValue().toLatin1());

#define _IS_HEXA_CHAR(myChar) \
  ((myChar >= '0' && myChar <= '9') || \
   (myChar >= 'A' && myChar <= 'F') || \
   (myChar >= 'a' && myChar <= 'f'))
#define _GET_HEXA_CHAR(myChar) (unsigned short)\
  (myChar >= '0' && myChar <= '9' ? \
   myChar - '0' : \
   (myChar >= 'A' && myChar <= 'F' ? myChar - 'A' + 10 : \
    (myChar >= 'a' && myChar <= 'f' ? myChar - 'a' + 10 : 0)))
    if (_IS_HEXA_CHAR(patternString[0]) &&
        _IS_HEXA_CHAR(patternString[1]) &&
        _IS_HEXA_CHAR(patternString[2]) &&
        _IS_HEXA_CHAR(patternString[3]))
    {

      unsigned short newPattern =
        (_GET_HEXA_CHAR(patternString[0]) << 0) +
        (_GET_HEXA_CHAR(patternString[1]) << 4) +
        (_GET_HEXA_CHAR(patternString[2]) << 8) +
        (_GET_HEXA_CHAR(patternString[3]) << 12);

      s_drawStyle->linePattern = newPattern;
    }
  }
};
SoDrawStyle* CustomPatternAuditor::s_drawStyle = NULL;

static void
buildDialog( SoTopLevelDialog* dialog )
{
  SoTabDialog* tabDialog = new SoTabDialog;
  dialog->addChild( tabDialog );

  SoRowDialog* normalMode = new SoRowDialog;
  normalMode->label.setValue( "Normal Mode" );
  tabDialog->addChild( normalMode );

  SoRowDialog* expertMode = new SoRowDialog;
  expertMode->label.setValue( "Expert Mode" );
  tabDialog->addChild( expertMode );

  SoDialogComboBox* patternChoice = new SoDialogComboBox;
  patternChoice->label.setValue( "Predefined Pattern :" );
  patternChoice->items.setValues( 0, 5, strings );
  patternChoice->selectedItem.setValue( DEFAULT_PATTERN );
  patternChoice->addAuditor( new PatternChoiceAuditor );
  normalMode->addChild( patternChoice );

  SoDialogIntegerSlider* scaleFactor = new SoDialogIntegerSlider;
  scaleFactor->label.setValue( "Scale Factor :" );
  scaleFactor->min.setValue( 1 );
  scaleFactor->max.setValue( 20 );
  scaleFactor->value.setValue( 1 );
  scaleFactor->addAuditor( new ScaleFactorAuditor );
  normalMode->addChild( scaleFactor );

  SoDialogEditText* customPattern = new SoDialogEditText;
  customPattern->label.setValue( "Custom Pattern HEX" );
  customPattern->addAuditor( new CustomPatternAuditor );
  expertMode->addChild( customPattern );

  const SbString strings[3] = {
    "Rectangle", "Diamond", "Circle",
  };
  SoDialogComboBox* shapeChoice = new SoDialogComboBox;
  shapeChoice->label.setValue( "Shape Type :" );
  shapeChoice->items.setValues( 0, 3, strings );
  shapeChoice->addAuditor( new ShapeTypeAuditor );
  expertMode->addChild( shapeChoice );

  dialog->buildDialog( NULL );
}

static void
buildSceneGraph( SoSeparator* root )
{
  SoMaterial *faceSetMaterial = new SoMaterial;	
  faceSetMaterial->diffuseColor.setValue(1.0F, 0.3F, 0.3F);
  root->addChild(faceSetMaterial);

  const float pts[4][3] = {
    {  3.0f, -3.0f, -1.1f },
    {  3.0f,  3.0f, -1.1f },
    { -3.0f,  3.0f, -1.1f },
    { -3.0f, -3.0f, -1.1f },
  };
  SoSeparator *newNode = new SoSeparator;
  SoCoordinate3 *controlPts =(SoCoordinate3 *)new SoCoordinate3;
  controlPts->point.setValues(0, 4, pts);
  newNode->addChild(controlPts);
  newNode->addChild(new SoFaceSet);
  root->addChild(newNode);

  SoMaterial *objMaterial = new SoMaterial;	
  objMaterial->ambientColor.setValue(1.0F, 1.0F, 1.0F);
  root->addChild(objMaterial);

  SoMaterial **materialDataCB = new SoMaterial *[2];
  materialDataCB[FULL_PATTERN] = objMaterial;
  materialDataCB[DOTL_PATTERN] = faceSetMaterial;
    
  // Set the draw style for object.
  SoDrawStyle *myDrawStyle = new SoDrawStyle;
  myDrawStyle->setName(DRAWSTYLE_NAME);
  myDrawStyle->style = SoDrawStyleElement::LINES;
  myDrawStyle->linePattern = patterns[DEFAULT_PATTERN];
  myDrawStyle->linePatternScaleFactor = DEFAULT_FACTOR;
  root->addChild(myDrawStyle);

  SoNode *object = makeRectangle();
  object->setName(SHAPE_NAME);
  root->addChild(object);

  ScaleFactorAuditor::s_drawStyle = myDrawStyle;
  PatternChoiceAuditor::s_drawStyle = myDrawStyle;
  CustomPatternAuditor::s_drawStyle = myDrawStyle;
  ShapeTypeAuditor::s_root = root;
}

int
main( int, char** argv )
{
  Widget myWindow = SoXt::init( argv[0] );
  if ( myWindow == NULL )
    exit( 1 );

  SoDialogViz::init();

  // Set up viewer:
  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer( myWindow );

  SoRef<SoTopLevelDialog> dialog = new SoTopLevelDialog;
  dialog->label.setValue( "Editing Tools" );
  buildDialog( dialog.ptr() );

  // Build Scene Graph
  SoRef<SoSeparator> root = new SoSeparator;
  buildSceneGraph( root.ptr() );
  dialog->width.setValue( 300 );
  dialog->height.setValue( 150 );
  dialog->windowResizedByChildren.setValue( TRUE );

  myViewer->setSceneGraph( root.ptr() );
  myViewer->setTitle( "SoDrawStyle Line Pattern Scale Factor" );

  myViewer->show();
  dialog->show();

  SoXt::show( myWindow );
  SoXt::mainLoop();

  dialog = NULL;
  SoDialogViz::finish();

  root = NULL;
  delete myViewer;
  SoXt::finish();

  return 0;
}

static SoSeparator *
makeForm(float **pts, int nb)
{
  SoSeparator *myForm = new SoSeparator;
  myForm->ref();
  SoCoordinate3 *controlPts = (SoCoordinate3 *)new SoCoordinate3;
  controlPts->point.setValues(0, nb, (SbVec3f *)pts);
  myForm->addChild(controlPts);
  myForm->addChild(new SoLineSet);

  myForm->unrefNoDelete();
  return myForm;
}

const float 
rectanglePoints[5][3] = {
  {-2.0f,-1.0f, 0.0f},
  { 2.0f,-1.0f, 0.0f},
  { 2.0f, 1.0f, 0.0f},
  {-2.0f, 1.0f, 0.0f},
  {-2.0f,-1.0f, 0.0f}
};

static SoSeparator *
makeRectangle()
{
  return makeForm((float **)rectanglePoints, 5);
}

const float 
diamondPoints[5][3] = {
  {-2.0f, 0.0f, 0.0f},
  { 0.0f,-1.0f, 0.0f},
  { 2.0f, 0.0f, 0.0f},
  { 0.0f, 1.0f, 0.0f},
  {-2.0f, 0.0f, 0.0f}
};

static SoSeparator *
makeDiamond()
{
  return makeForm((float **)diamondPoints, 5);
}

const float 
circlePoints[13][3] = {
  { 1.0f,   0.0f,   0.0f},
  { 0.866f, 0.5f,   0.0f},
  { 0.5f,   0.866f, 0.0f},
  { 0.0f,   1.0f,   0.0f},
  {-0.5f,   0.866f, 0.0f},
  {-0.866f, 0.5f,   0.0f},
  {-1.0f,   0.0f,   0.0f},
  {-0.866f,-0.5f,   0.0f},
  {-0.5f,  -0.866f, 0.0f},
  { 0.0f,  -1.0f,   0.0f},
  { 0.5f,  -0.866f, 0.0f},
  { 0.866f,-0.5f,   0.0f},
  { 1.0f,   0.0f,   0.0f}
};

static SoSeparator *
makeCircle()
{
  return makeForm((float **)circlePoints, 13);
}

