#if !defined(BUFFER_OBJECT_MAPPER_H)
#define BUFFER_OBJECT_MAPPER_H

#include <Inventor/devices/SoLocalCpuBufferObject.h>

/**
*
* This class provides an safe and simplified way to access data through typed a cpu array.
* It allows to use the following syntax while app wants to read or write some data from an SoBufferObject.
* Example:
*    void updateABufferObject(SoBufferObject* bufferObject);
*    {
*      BufferObjectMapper<SbVec3f> vec3array(bufferObject,SoBufferObject::SET);
*      for (int i=0;i<vec3array.getSize();++i)
*        vec3array[i] = SbVec3f(0.f,0.f,0.f);
*    }
*
*    // which replace and add safety to the following code
*    void updateABufferObject(SoBufferObject* bufferObject);
*    {
*      SoRef<SoCpuBufferObject> cpuBufferObject = new SoCpuBufferObject;
*      bufferObject->map(cpuBufferObject,SoBufferObject::SET);
*      SbVec3f* vec3array = static_cast<SbVec3f*>(cpuBufferObject->map(SoBufferObject::SET));
*      size_t arraySize=cpuBufferObject.getSize()/sizeof(SbVec3f);
*      for (int i=0;i<arraySize;++i)
*        vec3array[i] = SbVec3f(0.f,0.f,0.f);
*      cpuBufferObject->unmap();
*      bufferObject->unmap(cpuBufferObject);
*    }
*
*/
template<typename T>
class BufferObjectMapper
{
public:

  /** Constructor. In charge of setting mapping */
  BufferObjectMapper(SoBufferObject* bufferObject, SoBufferObject::AccessMode accessMode)
    :m_bufferObject(bufferObject)
  {
    m_bufferObject->map(&m_cpuBufferObject,accessMode);
    m_pointer = static_cast<T*>(m_cpuBufferObject.map(accessMode));
    m_arraySize = bufferObject->getSize()/sizeof(T);
  }

  /** Destructor. In charge of releasing mapping */
  ~BufferObjectMapper()
  {
    m_cpuBufferObject.unmap();
    m_bufferObject->unmap(&m_cpuBufferObject);
  }

  /** access to T element at index i */
  T& operator[](const size_t i)
  {
    assert(i<m_arraySize);
    return (m_pointer[i]);
  }

  /** const acces to T element at index i */
  const T& operator[](const size_t i) const
  {
    assert(i<m_arraySize);
    return (m_pointer[i]);
  }

  /** returns the number of T element in the mapped array */
  size_t getSize() const
  { return m_arraySize; }

private:
  // private default constructor to avoid copy by reference
  BufferObjectMapper(){}

  // cpu bufferobject to keep track of the current mapping
  SoLocalCpuBufferObject m_cpuBufferObject;

  // keep track of the source mapped buffer object
  SoBufferObject* m_bufferObject;

  // keep track of the pointer target of the mapping
  T* m_pointer;
  
  // keep track of the buffer size in number of element
  size_t m_arraySize;
};

#endif // BUFFER_OBJECT_MAPPER_H
