#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/engines/SoCalculator.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMultipleInstance.h>
#include <Inventor/nodes/SoInstanceParameter.h>
#include <Inventor/nodes/SoExtSelection.h>
#include <Inventor/nodes/SoShaderProgram.h>


#include <Inventor/devices/SoGLContext.h>
#include <Inventor/devices/SoCpuBufferObject.h>

#include <Inventor/actions/SoBoxHighlightRenderAction.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoDialogViz.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include "BufferObjectMapper.h"

static std::string s_resourceRootDir;

// create SoBufferObject containing per instance translations
SoBufferObject*
createTranslationInstanceParameterBuffer( int numInstanceX, int numInstanceY, int numInstanceZ )
{
  SoCpuBufferObject* bufferValues = new SoCpuBufferObject;
  bufferValues->setSize(numInstanceX * numInstanceY * numInstanceZ * sizeof(SbVec3f));

  BufferObjectMapper<SbVec3f> translationMapping(bufferValues, SoBufferObject::SET);
  for (int k = 0; k < numInstanceZ; k++)
    for (int j = 0; j < numInstanceY; j++)
      for (int i = 0; i < numInstanceX; i++)
        translationMapping[i + j * numInstanceX + k * numInstanceX * numInstanceY].setValue(i * 100.f, j * 100.f, k * 100.f);

  return bufferValues;
}

// This method is provided as an explicit example of setting a generic SoInstanceParameter.
// As a matter of fact the parameter set is one of the predefined ones.
// The same setting could be done in a less verbose way calling
// the SoInstanceParameter::createPredefinedParameter method directly passing
// the SoBufferObject returned by createTranslationInstanceParameterBuffer()
SoInstanceParameter*
createTranslationInstanceParameter( int numInstanceX, int numInstanceY, int numInstanceZ )
{
  SoInstanceParameter* translation = new SoInstanceParameter;
  translation->name = SoInstanceParameter::getPredefinedParameterName( SoInstanceParameter::TRANSLATION );
  translation->components = 3;
  translation->type = SbDataType::FLOAT;

  SoRef<SoBufferObject> bufferValues( createTranslationInstanceParameterBuffer( numInstanceX, numInstanceY, numInstanceZ ) );
  translation->value.setValue(bufferValues.ptr());

  return translation;
}

// create SoBufferObject containing per instance scaling factors
void
createScaleInstanceParameterBuffer( int numInstanceX, int numInstanceY, int numInstanceZ, std::vector<SbVec3f>& buffer )
{
  buffer.resize(numInstanceX * numInstanceY * numInstanceZ * sizeof(SbVec3f));

  for (int k = 0; k < numInstanceZ; k++)
  {
    for (int j = 0; j < numInstanceY; j++)
    {
      for (int i = 0; i < numInstanceX; i++)
      {
        float s = static_cast<float>(1. + 0.5*cos(10.*M_PI*(i/(float)numInstanceZ))*cos(10.*M_PI*(j/(float)numInstanceY))*cos(10.*M_PI*(k/(float)numInstanceX)));
        buffer[i+j*numInstanceX+k*numInstanceX*numInstanceY] = SbVec3f(s, s, s);
      }
    }
  }
}

// create SoBufferObject containing per instance colors
void
createColorInstanceParameterBuffer( int numInstanceX, int numInstanceY, int numInstanceZ, std::vector<SbColor>& buffer )
{
  buffer.resize(numInstanceX * numInstanceY * numInstanceZ * sizeof(SbColor));

  for (int k = 0; k < numInstanceZ; k++)
    for (int j = 0; j < numInstanceY; j++)
      for (int i = 0; i < numInstanceX; i++)
        buffer[i+j*numInstanceX+k*numInstanceX*numInstanceY] = SbColor( 1.0f * i / (numInstanceX - 1.0f),
                                                                        1.0f * j / (numInstanceY - 1.0f),
                                                                        1.0f * k / (numInstanceZ - 1.0f) );
}

// create SoBufferObject containing per instance rotations factors
void
createRotationInstanceParameterBuffer(std::vector<SbVec4f>& buffer)
{
  const size_t numRotations = 4;

  buffer.resize(numRotations * sizeof(SbVec4f));

  std::vector<SbRotation> rotations;
  rotations.push_back(SbRotation(SbVec3f(1.f, 0.f, 0.f), (float)M_PI / 2.f)); //Pi/2 rotation on x
  rotations.push_back(SbRotation(SbVec3f(0.f, 1.f, 0.f), (float)M_PI / 2.f)); //Pi/2 rotation on y
  rotations.push_back(SbRotation(SbVec3f(0.f, 0.f, 1.f), (float)M_PI / 2.f)); //Pi/2 rotation on z
  rotations.push_back(SbRotation(SbVec3f(1.f, 1.f, 0.f), (float)M_PI / 2.f)); //Pi/2 rotation on x and y

  for (size_t i = 0; i < rotations.size(); ++i)
  {
    SbVec4f rot;
    rotations[i].getValue(rot[0], rot[1], rot[2], rot[3]);
    buffer[i] = rot;
  }
}

/////////////////////////////////////////////////////////////
//
// Auditor class to handle events in user interface
class GUIManager : public SoDialogIntegerSliderAuditor
{
public:
  GUIManager( SoMultipleInstance* mi, SoWidget window )
    : m_nodePtr( mi )
    , m_numInstX( NULL )
    , m_numInstY( NULL )
    , m_numInstZ( NULL )
    , m_viewer( NULL )
    , m_isValid(false)
  {  
    // get UI controls handles
    SoInput input;
    std::string temp = s_resourceRootDir + "Instancing.iv";
    input.openFile(temp.c_str());
    m_iv = SoDB::readAll(&input);

    SoTopLevelDialog* gui = static_cast<SoTopLevelDialog*>(m_iv->getByName("GUI"));
    if ( gui != NULL)
    {
      gui->buildDialog(window, true);

      m_numInstX = static_cast<SoDialogIntegerSlider*>(m_iv->getByName("Number_Instances_X"));
      m_numInstX->addAuditor( this );
      m_numInstY = static_cast<SoDialogIntegerSlider*>(m_iv->getByName("Number_Instances_Y"));
      m_numInstY->addAuditor( this );
      m_numInstZ = static_cast<SoDialogIntegerSlider*>(m_iv->getByName("Number_Instances_Z"));
      m_numInstZ->addAuditor( this );
      m_viewer = static_cast<SoDialogCustom*>(m_iv->getByName("Viewer"));
      m_isValid = true;
    }
  }

  void dialogIntegerSlider(SoDialogIntegerSlider* cpt)
  {
    if ( (cpt->auditorID.getValue() == "Number_Instances_X") || 
         (cpt->auditorID.getValue() == "Number_Instances_Y") ||
         (cpt->auditorID.getValue() == "Number_Instances_Z")  )
    {
      setVariableInstanceData();
    } 
  }

  SoDialogCustom* getViewer() { return m_viewer; }

  // set multi instance group per instance data
  void setVariableInstanceData()
  {
    if ( m_isValid == false )
      return;

    int numInstances = m_numInstX->value.getValue() * m_numInstY->value.getValue() * m_numInstZ->value.getValue();

    // set number of instances
    m_nodePtr->numInstances.setValue( numInstances );

    // Set per instance translations
    // The following code provided just as an explicit example of setting 
    // a generic SoInstanceParameter see createTranslationInstanceParameter.
    // The same settings could be done in a less verbose way by calling
    // SoInstanceParameter::createPredefinedParameter as it is done 
    // afterwards for scale factors and rotations parameters
    SoInstanceParameter* translationParameter = m_nodePtr->parameters.findParameterByName( SoInstanceParameter::getPredefinedParameterName(SoInstanceParameter::TRANSLATION) );
    if ( translationParameter == NULL )
    {
      translationParameter = createTranslationInstanceParameter( m_numInstX->value.getValue(),
                                                                 m_numInstY->value.getValue(),
                                                                 m_numInstZ->value.getValue() );
      m_nodePtr->parameters.set1Value( m_nodePtr->parameters.getNum(), translationParameter );
    }
    else
    {
      SoRef<SoBufferObject> translationsBuffer( createTranslationInstanceParameterBuffer( m_numInstX->value.getValue(),
                                                                                          m_numInstY->value.getValue(),
                                                                                          m_numInstZ->value.getValue() ) );
      translationParameter->value.setValue( translationsBuffer.ptr() );
    }

    // set per instance scale factors
    createScaleInstanceParameterBuffer( m_numInstX->value.getValue(),  m_numInstY->value.getValue(), m_numInstZ->value.getValue(), m_scaleBuffer );
    m_nodePtr->parameters.setScales( &m_scaleBuffer[0], m_scaleBuffer.size()  );


    // set per instance rotations
    const uint32_t numRotations = 4;
    // This value specifies how many instances will share the same rotation.
    // So a fourth of the instances will use the same rotation, meaning we only need 4 rotations
    uint32_t rotationDivisor = 1 + numInstances / numRotations;
    createRotationInstanceParameterBuffer( m_rotationBuffer );
    m_nodePtr->parameters.setRotations( &m_rotationBuffer[0], m_rotationBuffer.size(), rotationDivisor );

    // set per instance colors
    createColorInstanceParameterBuffer( m_numInstX->value.getValue(),  m_numInstY->value.getValue(), m_numInstZ->value.getValue(), m_colorBuffer );
    m_nodePtr->parameters.setColors( &m_colorBuffer[0], m_colorBuffer.size() );
  }

private:

  std::vector<SbVec3f> m_scaleBuffer;
  std::vector<SbColor> m_colorBuffer;
  std::vector<SbVec4f> m_rotationBuffer;

  SoMultipleInstance* m_nodePtr;
  SoDialogIntegerSlider* m_numInstX;
  SoDialogIntegerSlider* m_numInstY;
  SoDialogIntegerSlider* m_numInstZ;
  SoDialogCustom* m_viewer;
  SoRef<SoSeparator> m_iv;
  bool m_isValid;
};


//
// Read an Inventor file and return the top separator
//
SoSeparator*
readFile(const char* filename)
{
  // Open the input file
  SoInput mySceneInput;
  if (!mySceneInput.openFile(filename))
  {
    fprintf(stderr, "Cannot open file %s\n", filename);
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator *myGraph = SoDB::readAll(&mySceneInput);
  if (myGraph == NULL)
  {
    fprintf(stderr, "Problem reading file\n");
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

int
main(int, char **argv)
{
  SoWidget window = SoXt::init(argv[0]);
  SoDialogViz::init();

  s_resourceRootDir = "$OIVHOME/examples/source/Inventor/Features/MultiInstancing/BufferedShapes/";

  // create main scene-graph
  SoSeparator* root = new SoSeparator;
  root->addChild( new SoGradientBackground );

  // create selection
  SoExtSelection* sel = new SoExtSelection;
  sel->useFastEditing(TRUE);
  sel->lassoColor = SbColor(0, 1, 0);
  sel->lassoPattern = 0xE0E0;

  sel->animateLasso = TRUE;
  sel->policy = SoSelection::SINGLE;
  sel->lassoType = SoExtSelection::RECTANGLE;
  sel->lassoPolicy = SoExtSelection::FULL_BBOX;
  sel->lassoMode = SoExtSelection::ALL_SHAPES;

  // add selection to scene graph
  root->addChild( sel );

  // create multiple instance group
  SoMultipleInstance* multipleInstanceGroup = new SoMultipleInstance;

  // the following code is scoped to force destruction 
  // of the GUIManager object before calling finish methods
  {
    // set up dialog auditor and UI data
    GUIManager dialogManager( multipleInstanceGroup, window );

    // this method sets the UI related multi instance data
    dialogManager.setVariableInstanceData(); 

    // add multiple instance group to selection group
    sel->addChild( multipleInstanceGroup );

    // set examiner viewer in the custom dialog
    SoWidget viewerWidget = dialogManager.getViewer()->getWidget();
    SoXtExaminerViewer* viewer = new SoXtExaminerViewer(viewerWidget);

    // viewer boilerplate methods to show scene-graph
    viewer->show();
    SoXt::show( window );

    // Bind context to make SoDB::readAll() use SoGpuBufferObject when reading the BufferedShape
    viewer->getNormalSoContext()->bind();

    // add BufferShape representing a screw as single node of the SoMultipleInstance group
    std::string tempStr = s_resourceRootDir + "screw_buffered_shape.iv";
    SoSeparator* sg = readFile(tempStr.c_str());
    if (sg == NULL)
      exit(EXIT_FAILURE);
    multipleInstanceGroup->addChild( sg );

    viewer->getNormalSoContext()->unbind();

    // set scene graph into the viewer
    viewer->setSceneGraph( root );

    // set auto selection feedback
    SoBoxHighlightRenderAction* boxHighlightRenderAction = new SoBoxHighlightRenderAction;
    boxHighlightRenderAction->setColor(SbColor(1.f, 0.f, 0.f));
    viewer->setGLRenderAction(boxHighlightRenderAction);
    viewer->redrawOnSelectionChange(sel);

    viewer->viewAll();
    SoXt::mainLoop();

    delete viewer;
  }

  SoDialogViz::finish();
  SoXt::finish();

  return EXIT_SUCCESS;
}
