/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoGetMatrixAction.h>

#include <Inventor/sensors/SoNodeSensor.h>

#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTexture2Transform.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoRotor.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoBlinker.h>
#include <Inventor/nodes/SoTextureMatrix.h>
#include <Inventor/nodes/SoTextureUnit.h>
#include <Inventor/nodes/SoTextureCoordinateEnvironment.h>
#include <Inventor/nodes/SoTextureCoordinateObject.h>
#include <Inventor/nodes/SoComplexity.h>

#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/SoWinApp.h>


#define SO_DEFAULT_LIGHT_FOV            1.04720F
#define SO_DEFAULT_SPOT_DIR             SbVec3f(0, -1, 0)
#define SO_DEFAULT_SPOT_ORIENTATION     0.0f, 0.0f, 0.382683f, 0.923880f
#define SO_TITLE                        "Multitexturing, Light Mapping and Projective texture"
#define SO_TEXTURE_PATH                 "$OIVHOME/examples/data/Inventor/Textures/"


// Scale and bias [-1,1]^3 clip space into [0,1]^3 texture space. 
static SbMatrix Smatrix(0.5f, 0.0f, 0.0f, 0.0f,
                        0.0f, 0.5f, 0.0f, 0.0f,
                        0.0f, 0.0f, 0.5f, 0.0f,
                        0.5f, 0.5f, 0.5f, 1.0f);

SoRotor *SpotRotor;
SoTextureMatrix *ProjectiveTexMat;
SoSeparator *SpotRotorSep;
SoSwitch    *InfoSwitch;
SoBlinker   *BlinkerProjTex;

SoXtExaminerViewer *MyViewer;
char Filename[256];

/*---------------------------------------------------------------------------*/

SoTexture2*
buildProjectiveTex(const char *texName)
{
  SoTexture2 *projectiveTex = new SoTexture2;
  projectiveTex->wrapS = SoTexture2::CLAMP;
  projectiveTex->wrapT = SoTexture2::CLAMP;
  projectiveTex->model = SoTexture2::BLEND;
  projectiveTex->filename.setValue(strcat(strcpy(Filename,SO_TEXTURE_PATH),texName));

  return projectiveTex;
}/*---------------------------------------------------------------------------*/

SoSwitch*
displayInfo() 
{
  // Informations
  SoSwitch *infoSwitch = new SoSwitch;
  infoSwitch->ref();
  infoSwitch->whichChild = SO_SWITCH_ALL;
  
  SoSeparator *infoSep = new SoSeparator;
  
  SoPickStyle *pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;
  infoSep->addChild(pickStyle);
  
  infoSwitch->addChild(infoSep);
  
  SoLightModel *lModel = new SoLightModel;
  lModel->model = SoLightModel::BASE_COLOR;
  infoSep->addChild(lModel);
  
  SoFont *fontInfo = new SoFont;
  fontInfo->name = "Courier New";
  fontInfo->size = 12;
  infoSep->addChild(fontInfo);
  
  SoBaseColor *infoColor = new SoBaseColor;
  infoColor->rgb.setValue(SbColor(1, 1, 0.f));
  infoSep->addChild(infoColor);
  
  SoTranslation *transInfo = new SoTranslation;
  transInfo->translation.setValue(-0.95f, 0.95f, 0.);
  infoSep->addChild(transInfo);
  
  SoText2 *infoText = new SoText2;
  infoText->string.set1Value(0, "H     : Toggle this display");
  infoText->string.set1Value(1, "P     : Start/Stop projection animation");
  infoText->string.set1Value(2, "SPACE : Start/Stop lights animation");
  infoText->string.set1Value(3, "T     : Change projection");
  infoSep->addChild(infoText);
  
  infoSwitch->unrefNoDelete();
  return infoSwitch;
}/*---------------------------------------------------------------------------*/

void
fpsCB(float fps, void *, SoXtViewer *viewer) {
  SbString titleStr(SO_TITLE);
  SbString fpsStr((int)fps);
  titleStr += " : ";
  titleStr += fpsStr;
  titleStr += " fps";
  
  viewer->setTitle(titleStr.getString());
}/*----------------------------------------------------------------------------*/

void 
myKeyPressCB (void *, SoEventCallback *eventCB) 
{
  
  const SoEvent *event = eventCB->getEvent();
  
  // check for the keys being pressed
  if (SO_KEY_PRESS_EVENT(event, H)) { // Toggle help display
	   if(InfoSwitch->whichChild.getValue() == SO_SWITCH_ALL)
       InfoSwitch->whichChild = SO_SWITCH_NONE;
     else
       InfoSwitch->whichChild = SO_SWITCH_ALL;   
  }
  else if (SO_KEY_PRESS_EVENT(event, SPACE)) { // Animation
    if(SpotRotor->on.getValue())
      SpotRotor->on = FALSE;
    else
      SpotRotor->on = TRUE;
  }
  else if (SO_KEY_PRESS_EVENT(event, P)) { // Change projection
    if(BlinkerProjTex->on.getValue())
      BlinkerProjTex->on = FALSE;
    else
      BlinkerProjTex->on = TRUE;
  }
  else if (SO_KEY_PRESS_EVENT(event, T)) { // Change projection
    BlinkerProjTex->whichChild.setValue( (BlinkerProjTex->whichChild.getValue()+1) % BlinkerProjTex->getNumChildren() );
  }

}/*---------------------------------------------------------------------------*/

void
updateProjectiveTexture()
{
  // Compute a view volume of the spot
  SbViewVolume viewVol;
  SbMatrix viewingMatrix, projectionMatrix;

  viewVol.perspective(SO_DEFAULT_LIGHT_FOV, 1, 1, 40);

  SbViewportRegion vp;
  SoGetBoundingBoxAction bboxAction(vp);
  bboxAction.apply(SpotRotorSep);

  SbRotation spotRot(SO_DEFAULT_SPOT_ORIENTATION);
  SbVec3f spotDir(SO_DEFAULT_SPOT_DIR);
  spotRot.multVec(spotDir, spotDir);

  SbRotation orientation;
  orientation.setValue(SbVec3f(0,0,-1), spotDir);
  orientation *= SpotRotor->rotation.getValue();

  viewVol.rotateCamera(orientation);
  viewVol.translateCamera(bboxAction.getCenter());

  viewVol.getMatrices(viewingMatrix, projectionMatrix);

  // Update the texture matrix
  ProjectiveTexMat->matrix.setValue(viewingMatrix*projectionMatrix*Smatrix);
}/*---------------------------------------------------------------------------*/

void  
rotorSensorCB(void *, SoSensor *)
{
  updateProjectiveTexture();
}/*---------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if(myWindow == NULL) exit(1);
  
  bool fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);

  SoSeparator *root = new SoSeparator;
  root->ref();

    // Display Info
  InfoSwitch = displayInfo();
  root->addChild(InfoSwitch);

  SoSeparator *scene3D = new SoSeparator;
  root->addChild(scene3D);

  SoPerspectiveCamera *camera = new SoPerspectiveCamera;
  scene3D->addChild(camera);


 // Track the keyboard events
  SoEventCallback *myEventCB = new SoEventCallback;
  myEventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), myKeyPressCB, NULL);
  scene3D->addChild(myEventCB);

  SoComplexity *complexity = new SoComplexity;
  complexity->value = 0.8f;
  complexity->textureQuality = 1.0f;

  scene3D->addChild(complexity);

  
  //************** Build texture unit 0 with a light map texture ***********

  // The default texture unit is 0, so it is not
  // necessary to specify it.
  SoTextureUnit *texUnit0 = new SoTextureUnit;
  texUnit0->unit = (unsigned long)0;

  SoTexture2 *lightMapTex = new SoTexture2;
  lightMapTex->model = SoTexture2::REPLACE;
  lightMapTex->filename.setValue(strcat(strcpy(Filename,SO_TEXTURE_PATH),"Lightmap.png"));

  scene3D->addChild(texUnit0);
  // Texture coordinates are defined by a function
  scene3D->addChild(new SoTextureCoordinateEnvironment);
  scene3D->addChild(lightMapTex);

  //************** Build texture unit 2 with projective textures ***********
  SoGroup *projectiveTexGroup = new SoGroup;


  SoTextureUnit *texUnit2 = new SoTextureUnit;
  texUnit2->unit = (unsigned long)2;

  BlinkerProjTex = new SoBlinker;
  BlinkerProjTex->speed = 0.02f;
  BlinkerProjTex->addChild(buildProjectiveTex("Spot1.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot2.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot3.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot4.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot5.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot6.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot7.jpg"));
  BlinkerProjTex->addChild(buildProjectiveTex("Spot8.jpg"));
  if ( !fullDemo ) BlinkerProjTex->on = FALSE;

  ProjectiveTexMat = new SoTextureMatrix;
  projectiveTexGroup->addChild(texUnit2);
  projectiveTexGroup->addChild(new SoTextureCoordinateObject);
  projectiveTexGroup->addChild(ProjectiveTexMat);
  projectiveTexGroup->addChild(BlinkerProjTex);

  //************** Build texture unit 1 with an oxidated texture ***********

  SoTextureUnit *texUnit1 = new SoTextureUnit;
  texUnit1->unit = (unsigned long)1;
  scene3D->addChild(texUnit1);

  // We let Open Inventor compute texture coordinates for texture unit 1
  SoTexture2 *oxidatedTex = new SoTexture2;
  oxidatedTex->filename.setValue(strcat(strcpy(Filename,SO_TEXTURE_PATH),"Oxidated.jpg"));
  oxidatedTex->model = SoTexture2::BLEND;
  oxidatedTex->blendColor.setValue(0.5f, 0.2f, 0.f);

    
  // Make a sphere, cone and cube with :
  //    - light map texture in texture unit 0
  //    - oxidated texture in texture unit 1
  //    - projective texture in texture unit 2
  SoSeparator *objectsSep = new SoSeparator;

  // Sphere
  SoSeparator *sphereSep = new SoSeparator;

  SoSeparator *sphereShapeSep = new SoSeparator;
  SoSphere *sphere = new SoSphere;
  sphere->radius = 2;
  sphereShapeSep->addChild(sphere);

  sphereSep->addChild(sphereShapeSep);

  // Cone
  SoSeparator *coneSep = new SoSeparator;

  SoSeparator *coneShapeSep = new SoSeparator;
  SoTranslation *transCone = new SoTranslation;
  transCone->translation.setValue(4,0,4);
  SbMatrix transMatrix;
  transMatrix.setTranslate(SbVec3f(4,0,4));
  SoTextureMatrix *projConeTexMat = new SoTextureMatrix;
  projConeTexMat->matrix = transMatrix;

  SoCone *cone = new SoCone;
  cone->bottomRadius = 2;
  cone->height = 4;
  coneShapeSep->addChild(transCone);
  coneShapeSep->addChild(cone);

  coneSep->addChild(projConeTexMat);
  coneSep->addChild(coneShapeSep);

  // Cube
  SoSeparator *cubeSep = new SoSeparator;

  SoSeparator *cubeShapeSep = new SoSeparator;
  SoTranslation *transCube = new SoTranslation;
  transCube->translation.setValue(-5,0,-5);

  transMatrix.identity();
  transMatrix.setTranslate(SbVec3f(-5,0,-5));
  SoTextureMatrix *projCubeTexMat = new SoTextureMatrix;
  projCubeTexMat->matrix = transMatrix;

  SoCube *cube = new SoCube;
  cube->width = 4;
  cube->depth = 4;
  cube->height = 4;
  cubeShapeSep->addChild(transCube);
  cubeShapeSep->addChild(cube);

  objectsSep->addChild(oxidatedTex);
  objectsSep->addChild(projectiveTexGroup);
  objectsSep->addChild(coneSep);
  objectsSep->addChild(cubeSep);
  objectsSep->addChild(sphereSep);

  cubeSep->addChild(projCubeTexMat);
  cubeSep->addChild(cubeShapeSep);


  scene3D->addChild(objectsSep);
 
  // Make Floor ground with
  //    - light map texture in texture unit 0
  //    - oxidated texture in texture unit 1
  //    - projective texture in texture unit 2
  SoSeparator *floorGroundSep = new SoSeparator;
  SoTranslation *transFloorGround = new SoTranslation;
  transFloorGround->translation.setValue(0,-2,0);
  SoTexture2Transform *projGroundTexMat = new SoTexture2Transform;
  projGroundTexMat->translation.setValue(0, -2);

  SoTexture2Transform *floorGroundTransTex = new SoTexture2Transform;
  floorGroundTransTex->scaleFactor.setValue(2, 2);

  SoTexture2 *floorGroundTex = new SoTexture2;
  floorGroundTex->model = SoTexture2::REPLACE;
  floorGroundTex->filename.setValue(strcat(strcpy(Filename,SO_TEXTURE_PATH),"MarbleTiles.jpg"));

  SoCube *floorGround = new SoCube;
  floorGround->width =25;
  floorGround->depth = 25;
  floorGround->height = 0.125;
  
  floorGroundSep->addChild(transFloorGround);
  // Build another texture unit 1 with a wood texture
  floorGroundSep->addChild(floorGroundTransTex);
  floorGroundSep->addChild(floorGroundTex);
  floorGroundSep->addChild(projectiveTexGroup);
  floorGroundSep->addChild(projGroundTexMat);
  floorGroundSep->addChild(floorGround);

  scene3D->addChild(floorGroundSep);

  // Build a spot light
  SoSeparator *spotSep = new SoSeparator;

  SpotRotor = new SoRotor();
  SpotRotor->rotation.setValue(SbVec3f(0.0f, 1.0f, 0.0f), (float)(M_PI/32.0f)); // y axis
  SpotRotor->speed = 0.05f;
  if (!fullDemo) SpotRotor->on = FALSE;

  SoNodeSensor *nodeSensor = new SoNodeSensor(rotorSensorCB, NULL);
  nodeSensor->attach(SpotRotor);

  SoTransform *transfSpot = new SoTransform;
  transfSpot->rotation.setValue(SO_DEFAULT_SPOT_ORIENTATION);
  transfSpot->translation.setValue(SbVec3f(-16.5, 7, 0.));
  transfSpot->scaleFactor.setValue(SbVec3f(0.75, 0.75, 0.75));

  SoCone *spotPart1 = new SoCone;
  spotPart1->removePart(SoCone::BOTTOM);

  SoCylinder *spotPart2 = new SoCylinder;
  spotPart2->radius = 0.5f;
  spotPart2->height = 1.95f;

  SoTexture2 *spotTex1 = new SoTexture2;
  spotTex1->filename.setValue(strcat(strcpy(Filename,SO_TEXTURE_PATH),"Ste06.jpg"));

  SoTexture2Transform *transTexSpot3 = new SoTexture2Transform;
  transTexSpot3->translation.setValue(SbVec2f(0.5f,-0.10f));
  transTexSpot3->scaleFactor.setValue(SbVec2f(0.8f,0.8f));
  transTexSpot3->rotation.setValue(float(M_PI/4.0f));

  SoCone *spotPart3 = new SoCone;
  spotPart3->removePart(SoCone::SIDES);

  SpotRotorSep =  new SoSeparator;
  SpotRotorSep->addChild(SpotRotor);
  SpotRotorSep->addChild(spotSep);

  spotSep->addChild(transfSpot);
  spotSep->addChild(spotTex1);
  spotSep->addChild(spotPart1);
  spotSep->addChild(spotPart2);
  spotSep->addChild(BlinkerProjTex);
  spotSep->addChild(transTexSpot3);
  spotSep->addChild(spotPart3);

  scene3D->addChild(SpotRotorSep);


  MyViewer = new SoXtExaminerViewer(myWindow);
  camera->viewAll(scene3D, MyViewer->getViewportRegion());

  //************* Because of light mapping, no light is inserted ************
  //**************************** in the scene graph *************************
  MyViewer->setHeadlight(FALSE);
  MyViewer->setSceneGraph(root);
  MyViewer->setTitle(SO_TITLE);
  if (fullDemo) MyViewer->setFramesPerSecondCallback(fpsCB);
  
  MyViewer->show();

  updateProjectiveTexture();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  root->unref();
  delete MyViewer;
  SoXt::finish();

  return 0;
}


