/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoNormal.h>
#include <Inventor/nodes/SoNormalBinding.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoEventCallback.h>


static float obeliskHeight = 30.0F, obeliskWidth = 4.0F; // default obelisk value
static float heightDelta = 1.0F, widthDelta = 2.0F; // default obelisk increment

//  Eight polygons. The first four are triangles 
//  The second four are quadrilaterals for the sides.
static float 
vertices[28][3] =
{
  { 0.0f, obeliskHeight, 0.0f}, {-2.0f,27.0f, 2.0f}, { 2.0f,27.0f, 2.0f}, // front tri
  { 0.0f, obeliskHeight, 0.0f}, {-2.0f,27.0f,-2.0f}, {-2.0f,27.0f, 2.0f}, // left  tri
  { 0.0f, obeliskHeight, 0.0f}, { 2.0f,27.0f,-2.0f}, {-2.0f,27.0f,-2.0f}, // rear  tri
  { 0.0f, obeliskHeight, 0.0f}, { 2.0f,27.0f, 2.0f}, { 2.0f,27.0f,-2.0f}, // right tri
  {-2.0f, 27.0f, 2.0f}, {-obeliskWidth, 0.0f, obeliskWidth}, 
  { obeliskWidth, 0.0f, obeliskWidth}, { 2.0f,27.0f, 2.0f}, // front quad
  {-2.0f, 27.0f,-2.0f}, {-obeliskWidth, 0.0f,-obeliskWidth}, 
  {-obeliskWidth, 0.0f, obeliskWidth}, {-2.0f,27.0f, 2.0f}, // left  quad
  { 2.0f, 27.0f,-2.0f}, { obeliskWidth, 0.0f,-obeliskWidth}, 
  {-obeliskWidth, 0.0f,-obeliskWidth}, {-2.0f,27.0f,-2.0f}, // rear  quad
  { 2.0f, 27.0f, 2.0f}, { obeliskWidth, 0.0f, obeliskWidth}, 
  { obeliskWidth, 0.0f,-obeliskWidth}, { 2.0f,27.0f,-2.0f}  // right quad
};

// Number of vertices in each polygon:
static int32_t numvertices[8] = {3, 3, 3, 3, 4, 4, 4, 4};

SoSeparator *
makeObeliskFaceSet(SoMFVec3f *&verticesFieldReturned)
{
  SoSeparator *obelisk = new SoSeparator();
  obelisk->ref();
  
  SoVertexProperty *myVertexProperty = new SoVertexProperty;
  myVertexProperty->orderedRGBA.setValue(SbColor(1.0f, 0.15f, 0.15f).getPackedValue());

  // Define coordinates for vertices with USER data
  myVertexProperty->vertex.setValuesPointer(28, (float *)vertices);
  
  // Define the FaceSet
  SoFaceSet *myFaceSet = new SoFaceSet;
  myFaceSet->numVertices.setValuesPointer(8, numvertices);
  
  myFaceSet->vertexProperty.setValue(myVertexProperty);
  obelisk->addChild(myFaceSet);
  
  obelisk->unrefNoDelete();

  verticesFieldReturned = &(myVertexProperty->vertex);
  return obelisk;
}

void
obeliskHeightSetting(float newHeight)
{
  // Direct user-array setting
  vertices[0][1] = newHeight;
  vertices[3][1] = newHeight;
  vertices[6][1] = newHeight;
  vertices[9][1] = newHeight;
}

void
obeliskWidthSetting(float newWidth)
{
  // Direct user-array setting
  vertices[13][0] = -newWidth; vertices[13][2] =  newWidth;
  vertices[14][0] =  newWidth; vertices[14][2] =  newWidth;
  vertices[17][0] = -newWidth; vertices[17][2] = -newWidth;
  vertices[18][0] = -newWidth; vertices[18][2] =  newWidth;
  vertices[21][0] =  newWidth; vertices[21][2] = -newWidth;
  vertices[22][0] = -newWidth; vertices[22][2] = -newWidth;
  vertices[25][0] =  newWidth; vertices[25][2] =  newWidth;
  vertices[26][0] =  newWidth; vertices[26][2] = -newWidth;
}

static void
processKeyEvents(void *userData, SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();

  if (SO_KEY_PRESS_EVENT(ev, UP_ARROW) || 
      SO_KEY_PRESS_EVENT(ev, DOWN_ARROW)) {
    obeliskHeight += (SO_KEY_PRESS_EVENT(ev, UP_ARROW) ? heightDelta : -heightDelta);
    printf("Obelisk width = %f\n", obeliskHeight);
    obeliskHeightSetting(obeliskHeight);
    ((SoField *)userData)->touch(); // say that value has been changed by user
    eventCB->setHandled();
  }
  if (SO_KEY_PRESS_EVENT(ev, LEFT_ARROW) || 
      SO_KEY_PRESS_EVENT(ev, RIGHT_ARROW)) {
    obeliskWidth += (SO_KEY_PRESS_EVENT(ev, LEFT_ARROW) ? widthDelta : -widthDelta);
    if (obeliskWidth < 0) {
      obeliskWidth = 0.0F;
    }
    printf("Obelisk width = %f\n", obeliskWidth);
    obeliskWidthSetting(obeliskWidth);
    ((SoField *)userData)->touch(); // say that value has been changed by user
    eventCB->setHandled();
  }
}

int
main(int, char **argv)
{
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) {
    exit(1);
  }
  
  // Display usage
  printf("To change obelisk height and width: while in picking mode,\n");
  printf("press the UP/DOWN and LEFT/RIGHT arrows keys.\n");
  printf("Obelisk height = %f ; Obelisk width = %f\n", obeliskHeight, obeliskWidth);

  SoSeparator *root = new SoSeparator;
  root->ref();
  
  // To get SoMFVec3f vertices field from vertexProperty (from SoFaceSet)
  SoMFVec3f *myVerticesField;
  SoSeparator *obelisk = makeObeliskFaceSet(myVerticesField);

  // Add EventCallback so obelisk height changes to Events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(),
                            processKeyEvents, myVerticesField);

  root->addChild(eventCB);
  root->addChild(obelisk);
  
  // Create and realize examiner viewer
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("SoMFVec3f test with USER data setting");
  myViewer->show();
  myViewer->viewAll();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  root->unref();
  delete myViewer;
  SoXt::finish();
  
  return 0;
}



