/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Pascal DOUX (Jun 1998)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/SoDB.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/nodes/SoNurbsProperty.h>

typedef struct cbdata {
  SoXtExaminerViewer *vwr;
  SoNode *root;
  SoNurbsProperty *nurbsProperty;
} callbackData;

static void
processKeyEvents(void *data, SoEventCallback *cb)
{
  int val;
  SoKeyboardEvent *keyEvent = (SoKeyboardEvent *)cb->getEvent();
  SoKeyboardEvent::Key key = keyEvent->getKey();

  if (SoKeyboardEvent::isKeyPressEvent(cb->getEvent(), key)) 
    return;

  callbackData *cbData = (callbackData *)data;

  switch (key) {
  case SoKeyboardEvent::B: // Boundaries
    // Show/hide  Boundaries 
    val = cbData->nurbsProperty->drawStyle.getValue();
    if (val & SoNurbsProperty::BOUNDARIES_AND_TRIM_CURVES) {
      cbData->nurbsProperty->drawStyle = val & ~SoNurbsProperty::BOUNDARIES_AND_TRIM_CURVES;
      printf("hide Boundaries \n"); 
    } 
    else { 
      cbData->nurbsProperty->drawStyle = val | SoNurbsProperty::BOUNDARIES_AND_TRIM_CURVES;
      printf("show Boundaries \n"); 
    }
    break;
  case SoKeyboardEvent::I: // IsoParametic
    // Show/hide Iso Parametric Curves
    val = cbData->nurbsProperty->drawStyle.getValue();
    if (val & SoNurbsProperty::CENTRAL_ISO_PARAM_CURVES) {
      printf("hide Iso Parametric Curves \n"); 
      cbData->nurbsProperty->drawStyle = val & ~SoNurbsProperty::CENTRAL_ISO_PARAM_CURVES;
    } 
    else { 
      printf("show Iso Parametric Curves \n"); 
      cbData->nurbsProperty->drawStyle = val | SoNurbsProperty::CENTRAL_ISO_PARAM_CURVES;
    }
    break;
  case SoKeyboardEvent::N: // Normal
    // Show/hide Nurbs Surface
    val = cbData->nurbsProperty->drawStyle.getValue();
    if (val & SoNurbsProperty::NORMAL) {
      printf("hide Nurbs Surface \n"); 
      cbData->nurbsProperty->drawStyle = val & ~SoNurbsProperty::NORMAL;
    } 
    else {
      printf("show Nurbs Surface \n"); 
      cbData->nurbsProperty->drawStyle = val | SoNurbsProperty::NORMAL;
    }
    break;
  case SoKeyboardEvent::M: // More
    // Increase number of sample points on trimming curves
    val = cbData->nurbsProperty->numSamplePoints.getValue();
    cbData->nurbsProperty->numSamplePoints = val + 1;
    printf("increase number of sample points on trimming curves: %d \n", val+1); 
    break;
  case SoKeyboardEvent::L: // Less
    // Decrease number of sample points on trimming curves
    val = cbData->nurbsProperty->numSamplePoints.getValue();
    // -1 means to use the SoComplexity setting
    if (val > -1) cbData->nurbsProperty->numSamplePoints = val - 1;
    printf("decrease number of sample points on trimming curves: %d \n", val-1); 
    break;
  default:
    // do nothing
    break;
  }
  cb->setHandled(); 
}

int
main(int argc, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) 
    exit(1);

  const char *filename = "$OIVHOME/examples/data/Inventor/NurbsProperties/pick2.iv";
  if (argc < 2)
    printf("Usage: %s file_name \n", argv[0]);
  else if(argc >=2)
	  filename = argv[1] ;

  SoInput input;
  if (! input.openFile(filename)) {
    printf("Cannot open file %s \n", filename);
    exit (-1);
  }
   
  printf("reading %s...\n", filename);
  SoSeparator *root = SoDB::readAll(&input);
  root->ref();

  SoNurbsProperty *nurbsProperty;

  // use brace to release SearchAction before exit (SoDb::finish called before)
  {
    // look for the first NurbsProperty in the scene graph.
    SoSearchAction search;
    search.setType(SoNurbsProperty::getClassTypeId());
    search.setInterest(SoSearchAction::FIRST);
    search.apply(root);

    if (search.getPath() == 0) {
      // add one if does not exist.
      printf(" no NurbsProperty node found, adding one\n");
      nurbsProperty = new SoNurbsProperty;
      nurbsProperty->color.setValue(1, 1, 1);
      nurbsProperty->isoParamCurvesPattern.setValue(0xF0F0);
      root->insertChild(nurbsProperty, 0);
    }
    else {
      // use the first found 
      nurbsProperty = (SoNurbsProperty *)search.getPath()->getTail();
      // TODO: override is not implemented for nurbsProperty
      // and set override True to see the effect of our changes on the whole scene graph.
      //nurbsProperty->setOverride(TRUE);
    }
  }

  // add a callback for managing user input
  SoEventCallback *eventCB = new SoEventCallback;
  root->addChild(eventCB);

  printf("displaying the scene graph...\n");
  // Open a viewer for the simplified scene:
  SoXtExaminerViewer *viewer = new SoXtExaminerViewer(myWindow);
  viewer->setSceneGraph(root);
  viewer->setTitle("Nurbs Property Viewer");
  viewer->setViewing(FALSE);
  viewer->show();

  printf("\n Enter Esc to switch between viewing and picking mode \n");
  printf(" Enter: \n   B to toggle display of Nurbs and Trimming curves boundaries\n");
  printf("   I to toggle display of Central Iso Parametric Curves\n");
  printf("   N to toggle display of Nurbs Surface \n");
  printf("   M to increase the number of sample points on trimming curves\n");
  printf("   L to decrease the number of sample points on trimming curves\n");

  // Setup the event callback data and routine for switching levels
  callbackData *data = new callbackData;
   
  data->vwr = viewer;
  data->root = root;
  data->nurbsProperty = nurbsProperty;
   
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), 
                            processKeyEvents, data);
  SoXt::show(myWindow);
  SoXt::mainLoop();

  root->unref();
  delete viewer;
  SoXt::finish();

  return 0;
}


