#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoImageBackground.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/actions/SoSearchAction.h>

#include <Inventor/SoInput.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/Xt/SoXtColorEditor.h>

#include <Inventor/SoOffscreenRenderArea.h>
#include <DialogViz/dialog/SoDialogComboBox.h>

#include <DialogViz/SoDialogVizAll.h>

//include VolumeViz to be able to test different input files
#include <VolumeViz/nodes/SoVolumeData.h>

#include <Inventor/helpers/SbFileHelper.h>

Widget buildInterface(Widget);

SoXtRenderArea*  renderArea = NULL;
SoXtColorEditor* colorEditor = NULL;

SoTopLevelDialog* topLevelDialog = NULL;
SoDialogCustom*   customNode = NULL;

SbString ImageFileName;
SbString demoPath;
SoRef<SoSwitch> imageSwitch;

class auditorClass : public SoDialogAuditor
{
public:

  auditorClass();

  void dialogPushButton(SoDialogPushButton* pushbutton);
  void dialogComboBox(SoDialogComboBox* combobox);
  void dialogRealSlider(SoDialogRealSlider* slider);

  void refreshTransparency()
  {
    SoDialogComboBox* combobox = (SoDialogComboBox*)topLevelDialog->searchForAuditorId(SbString("transparencytype"));
    this->setTransparencyType(combobox->items[combobox->selectedItem.getValue()]);
  }

  void refreshAntialiasing()
  {
    SoDialogRealSlider* slider = (SoDialogRealSlider*)topLevelDialog->searchForAuditorId( SbString( "antialiasingquality" ) );
    this->setAntialiasingQuality( slider->value.getValue() );
  }

  void refreshImageBackground()
  {
    SoDialogComboBox* combobox = (SoDialogComboBox*)topLevelDialog->searchForAuditorId( SbString( "backgroundimage" ) );
    this->setBackgroundImage( combobox->selectedItem.getValue() );
  }

protected:

  void setTransparencyType( SbString name )
  {
    renderArea->getGLRenderAction()->setTransparencyType( transparencyTypes[name] );
    renderArea->render();
  }

  void setAntialiasingQuality( float quality )
  {
    renderArea->setAntialiasing( quality, antialiasingMode );
  }

  void setBackgroundImage( int32_t index )
  {
    if( index == 0 )
    {
      imageSwitch->whichChild = -1;
    }
    else
    {
      imageSwitch->whichChild = index - 1;
    }
  }

private:
  std::map<SbString, SoGLRenderAction::TransparencyType> transparencyTypes;
  SoSceneManager::AntialiasingMode antialiasingMode;
  float antialiasingQuality;
  SbVec2i32 myOffscreenSize;
};

// Give a select list of all possible OIV transparencies and antialiasing modes
auditorClass::auditorClass()
{
  transparencyTypes[SbString("NO_SORT")] = SoGLRenderAction::NO_SORT;
  transparencyTypes[SbString("OPAQUE_FIRST")] = SoGLRenderAction::OPAQUE_FIRST;
  transparencyTypes[SbString("SORTED_OBJECT")] = SoGLRenderAction::SORTED_OBJECT;
  transparencyTypes[SbString("SORTED_PIXEL")] = SoGLRenderAction::SORTED_PIXEL;

  antialiasingMode = SoSceneManager::AUTO;
  antialiasingQuality = 1.f;
  myOffscreenSize.setValue( 2500, 2500 );
}

// Transparency type and antialiasing mode of export should be the same of the viewer
void
auditorClass::dialogComboBox( SoDialogComboBox* combobox )
{
  if ( combobox->auditorID.getValue() == "transparencytype" ){
    this->setTransparencyType( combobox->items[combobox->selectedItem.getValue()] );
  }

  if (combobox->auditorID.getValue() == "backgroundimage"){
    this->setBackgroundImage( combobox->selectedItem.getValue() );
  }
}

void
colorEditorCB( void* /*userData*/, const SbColor* color )
{
  SbColorRGBA currentColor = renderArea->getSceneManager()->getBackgroundColorRGBA();
  currentColor[0] = (*color)[0];
  currentColor[1] = (*color)[1];
  currentColor[2] = (*color)[2];
  renderArea->getSceneManager()->setBackgroundColorRGBA( currentColor );
}

void
auditorClass::dialogRealSlider( SoDialogRealSlider* slider )
{
  if ( slider->auditorID.getValue() == "antialiasingquality" )
  {
    this->setAntialiasingQuality( slider->value.getValue() );
  }
  else if ( slider->auditorID.getValue() == "background_alpha" )
  {
    SbColorRGBA currentColor = renderArea->getSceneManager()->getBackgroundColorRGBA();
    currentColor[3] = slider->value.getValue();
    renderArea->getSceneManager()->setBackgroundColorRGBA( currentColor );
  }
}

void
auditorClass::dialogPushButton(SoDialogPushButton* cpt)
{
  if (cpt->auditorID.getValue() == "generate")
  {
    // Set width and height
    SoDialogIntegerSlider* cptWidth = (SoDialogIntegerSlider*)topLevelDialog->searchForAuditorId(SbString("width"));
    SoDialogIntegerSlider* cptHeight = (SoDialogIntegerSlider*)topLevelDialog->searchForAuditorId(SbString("height"));
    myOffscreenSize[0] = cptWidth->value.getValue();
    myOffscreenSize[1] = cptHeight->value.getValue();

    SoRef<SoOffscreenRenderArea> myOffscreen = new SoOffscreenRenderArea(renderArea->getNormalSoContext());
    myOffscreen->setViewportRegion(SbViewportRegion(myOffscreenSize));

    // Set subtiles
    SoDialogCheckBox* cptSubTiles = (SoDialogCheckBox*)topLevelDialog->searchForAuditorId(SbString("subTiles"));

    if (cptSubTiles->state.getValue() == TRUE)
    {
      SoDialogIntegerSlider* cptTileWidth = (SoDialogIntegerSlider*)topLevelDialog->searchForAuditorId(SbString("tileWidth"));
      SoDialogIntegerSlider* cptTileHeight = (SoDialogIntegerSlider*)topLevelDialog->searchForAuditorId(SbString("tileHeight"));

      int numEdgePixels = 4;
      myOffscreen->setTile(SbVec2i32(cptTileWidth->value.getValue(), cptTileHeight->value.getValue()), numEdgePixels);
    }

    // Set offscreen use same background as the viewer
    myOffscreen->setClearColor( renderArea->getSceneManager()->getBackgroundColorRGBA() );

    // Set offscreen use same transparency type as the viewer
    myOffscreen->setTransparencyType(renderArea->getTransparencyType());

    // Set offscreen use same antialiasing mode and quality as the viewer
    myOffscreen->getSceneManager()->setAntialiasing( renderArea->getAntialiasingQuality(), renderArea->getAntialiasingMode() );

    // Export or not background;
    SoSFInt32 currentBackground = imageSwitch->whichChild;
    SoDialogCheckBox* exportBackground = (SoDialogCheckBox*)topLevelDialog->searchForAuditorId( SbString( "exportbackground" ) );
    bool isExportBackground = exportBackground->state.getValue();
    if ( !isExportBackground )
    {
      imageSwitch->whichChild = -1;
    }

    // Render the image
    myOffscreen->setSceneGraph(renderArea->getSceneManager()->getSceneGraph());


    SoDialogComboBox* cptFileFormat = (SoDialogComboBox*)topLevelDialog->searchForAuditorId(SbString("fileFormat"));

    SbString fileExt;

    // Dump on disk in different file formats
    switch (cptFileFormat->selectedItem.getValue()) {
    case 0: //JPEG
      fileExt = ".jpg";
      break;
    case 1: //PNG
      fileExt = ".png";
      break;
    case 2: //PostScript
      fileExt = ".ps";
      break;
    case 3: //RGB
      fileExt = ".rgb";
      break;
    case 4: //TIFF
      fileExt = ".tiff";
      break;
    case 5: //BMP
      fileExt = ".bmp";
      break;
    }

    myOffscreen->renderToFile( ImageFileName + fileExt, SoOffscreenRenderArea::RGBA );

    // re set the background
    if (!isExportBackground)
    {
      imageSwitch->whichChild = currentBackground;
    }
  }
  else if ( cpt->auditorID.getValue() == "background_color" )
  {
    if ( colorEditor == NULL )
    {
      colorEditor = new SoXtColorEditor;
      colorEditor->setColor( renderArea->getBackgroundColor() );
      colorEditor->setCurrentSliders( SoXtColorEditor::HSV );
      colorEditor->addColorChangedCallback( colorEditorCB );
    }
    colorEditor->show();
  }
}

auditorClass* myAuditorClass;

int
main(int argc, char **argv)
{
  // Initialize Inventor. This returns a main window to use.
  // If unsuccessful, exit.
  Widget mainWindow = SoXt::init(argv[0]); // pass the app name

  SoDialogViz::init();
  SoVolumeRendering::init();

  if (mainWindow == NULL) exit(1);

  demoPath = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/OffscreenRendering");

  // Set up viewer
  Widget parent = buildInterface(mainWindow);
  renderArea = new SoXtRenderArea(parent);

  renderArea->setBackgroundColor(SbColor(0.7f, 0.7f, 0.7f));

  // Test.iv contains a simple scene with a transparent cube and a cone
  SbString ivFilename = demoPath + "/test.iv";

  // Eventually load a file given as parameter
  if (argc > 1)
    ivFilename = argv[1];

  FILE *fp = fopen(ivFilename.toLatin1(), "r");
  if (fp == NULL)
  {
    printf("Unable to open '%s'\n", ivFilename.toLatin1());
    ivFilename.makeNull();
  }
  else
    fclose(fp);

  // Read Inventor file or create simple scene
  SoRef<SoSeparator> myRoot = NULL;

  if (!ivFilename.isEmpty())
  {
    SoInput in;
    in.openFile(ivFilename.toLatin1());

    myRoot = SoDB::readAll(&in);
    in.closeFile();
    ImageFileName = ivFilename;
  }

  if (!myRoot.ptr())
  {
    printf("*** Unable to read file...\n");
    exit(1);
  }
  // Add camera and light if needed
  SoSearchAction sa;
  sa.setType(SoCamera::getClassTypeId());
  sa.apply(myRoot.ptr());
  SoRef<SoPath> pCamPath = sa.getPath();
  if ( !pCamPath.ptr() )
  {
    const SbViewportRegion vport(200, 200);
    SoRef<SoSeparator> scene = myRoot;
    myRoot = new SoSeparator;
    SoRef<SoPerspectiveCamera> pCam = new SoPerspectiveCamera;
    myRoot->addChild( pCam.ptr() );
    myRoot->addChild( new SoDirectionalLight );
    myRoot->addChild( scene.ptr() );
    pCam->viewAll( myRoot.ptr(), vport );
  }

  imageSwitch = new SoSwitch;

  SbString ressourcespath = SbFileHelper::expandString("$OIVHOME/data/textures/misc");
  SoRef<SoImageBackground> imageBackground1 = new SoImageBackground;
  imageBackground1->filename = ressourcespath + "/n036.rgb";
  imageSwitch->addChild( imageBackground1.ptr() );

  SoRef<SoImageBackground> imageBackground2 = new SoImageBackground;
  imageBackground2->filename = ressourcespath + "/n081.rgb";
  imageSwitch->addChild( imageBackground2.ptr() );

  myRoot->addChild( imageSwitch.ptr() );

  renderArea->setSceneGraph(myRoot.ptr());

  // Get imageBackground from GUI
  myAuditorClass->refreshImageBackground();
  // Get transparency type from GUI
  myAuditorClass->refreshTransparency();
  // Get antialiasing mode and quality from GUI
  myAuditorClass->refreshAntialiasing();

  renderArea->show();
  SoXt::show(mainWindow);  // Display main window
  SoXt::mainLoop();      // Main Inventor event loop

  delete renderArea;
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}

Widget
buildInterface(Widget window)
{
  SbString interfaceFilename = demoPath + "/OffscreenRendering_gui.iv";

  SoInput myInput;
  if ( !myInput.openFile(interfaceFilename) )
  {
    return window;
  }

  SoGroup* myGroup = SoDB::readAll( &myInput );
  if ( !myGroup )
  {
    return window;
  }

  topLevelDialog = ( SoTopLevelDialog* )myGroup->getChild( 0 );

  if ( topLevelDialog )
  {
    topLevelDialog->buildDialog( window, TRUE );
    myAuditorClass = new auditorClass;
    topLevelDialog->addAuditor( myAuditorClass );
    customNode = (SoDialogCustom*)topLevelDialog->searchForAuditorId( SbString("renderArea") );
    topLevelDialog->show();
  }

  return customNode ? customNode->getWidget() : window;
}
