#include <Inventor/SoDB.h>
#include <Inventor/SoOffscreenRenderArea.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSeparator.h>
#include "OffscreenStereo.h"

const char* EXAMPLE_FILE = "$OIVHOME/examples/data/Inventor/gear.iv";

void
usage( const char* name, const char* s )
{
  printf( "usage: %s [filename.iv]\n", name );
  printf( "default filename.iv is \"%s\"\n", EXAMPLE_FILE );
  printf( "generates two raster files, left and right stereo views of given scene\n" );
  if ( s != NULL )
    printf( "%s\n", s );
}

#define LEFT_FILENAME ".leftview"
#define RIGHT_FILENAME ".rightview"

typedef struct cbdata
{
  SoXtExaminerViewer* vwr;
  char* filename;
  SoNode* scene;
  OffscreenStereo* offStereo;
} callbackData;

void saveScene( SoOffscreenRenderArea*, void* );
void writeHTML( FILE*, const char*, int, int );

void
processKeyEvents( void* userData, SoEventCallback* cb )
{
  struct cbdata* data = ( struct cbdata* )userData;

  if ( SO_KEY_PRESS_EVENT( cb->getEvent(), P ) )
  {
    SimpleStereo simpleStereo;
    SbVec2s size = data->vwr->getSize();
    OffscreenStereo stereoViewer( data->scene, size[0], size[1] );
    printf( "Saving images on a %dx%d surface\n", size[0], size[1] );
    stereoViewer.setStereoViewType( &simpleStereo );
    stereoViewer.setViewer( data->vwr );
    simpleStereo.setStereoViewer( &stereoViewer );
    stereoViewer.setStereoOffset( 1.5 );
    stereoViewer.setStereoBalance( 1.0 );

    data->offStereo = &stereoViewer;
    stereoViewer.setCallback( ( OutputCB* )saveScene, data );
    stereoViewer.renderSceneInStereo();
    cb->setHandled();
  }
}

int
main( int argc, char** argv )
{
  // Initialize Inventor and Xt
  Widget appWindow = SoXt::init( argv[0] );
  if ( appWindow == NULL )
    exit( 1 );

  // Read the geometry from a file and add to the scene
  SoInput myInput;
  int fileIndex = -1;

  if ( argc == 2 )
  {
    if ( !myInput.openFile( argv[1] ) )
    {
      usage( argv[0], "could not open file\n" );
      exit( 2 );
    }
    fileIndex = 1;
  }
  else if ( argc == 1 )
  {
    if ( !myInput.openFile( EXAMPLE_FILE ) )
    {
      usage( argv[0], "could not open default file\n" );
      exit( 3 );
    }
  }
  else
  {
    usage( argv[0], "invalid number of arguments\n" );
    exit( 4 );
  }
  printf( "To render the scene in stereo offscreen: "
          "press the 'p' key while in picking mode\n" );

  SoSeparator* geomObject = SoDB::readAll( &myInput );
  if ( geomObject == NULL )
  {
    usage( argv[0], "no scene could be read from file\n" );
    exit( 5 );
  }

  // Make a scene containing an event callback node
  SoSeparator* root = new SoSeparator;
  SoEventCallback* eventCB = new SoEventCallback;
  root->ref();
  root->addChild( eventCB );
  root->addChild( geomObject );

  SoXtExaminerViewer* viewer =
    new SoXtExaminerViewer( appWindow, NULL, TRUE, SoXtExaminerViewer::BUILD_ALL, SoXtExaminerViewer::EDITOR );
  viewer->setSceneGraph( root );

  // Setup the event callback data
  callbackData* data = new callbackData;
  data->vwr = viewer;
  if ( fileIndex != -1 )
  {
    data->filename = new char[strlen( argv[fileIndex] ) + 1];
    sprintf( data->filename, "%s", argv[fileIndex] );
    data->filename[strlen( argv[fileIndex] ) - 3] = '\0';
  }
  else
    data->filename = ( char* )"gear";
  data->scene = viewer->getSceneGraph();
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), processKeyEvents, data );
  viewer->show();

  SoXt::show( appWindow );
  SoXt::mainLoop();

  root->unref();
  delete viewer;
  SoXt::finish();

  return 0;
}

void
saveScene( SoOffscreenRenderArea* offscreenRenderArea, void* userData )
{
  struct cbdata* data = ( struct cbdata* )userData;
  OffscreenStereo* os = data->offStereo;

  SoCamera* camera = os->getViewerCamera();
  SoCamera::StereoMode mode = camera->getStereoMode();
  SbBool left = ( mode == SoCamera::LEFT_VIEW );

  switch ( mode )
  {
  case SoCamera::LEFT_VIEW:
  case SoCamera::RIGHT_VIEW:
    // save the one stereo view image
    printf( "Saving %s view...\n", left ? "left" : "right" );
    {
      const char* viewname = left ? LEFT_FILENAME : RIGHT_FILENAME;
      char* name = new char[strlen( data->filename ) + strlen( viewname ) + 5];
      sprintf( name, "%s%s.jpg", data->filename, viewname );
      offscreenRenderArea->renderToFile( name );

      delete[] name;
    }
    break;
  case SoCamera::MONOSCOPIC:
    // save the html file
    printf( "Saving html file...\n" );
    {
      char* name = new char[strlen( data->filename ) + 6];
      sprintf( name, "%s.html", data->filename );
      FILE* fp = fopen( name, "w" );
      if ( !fp )
      {
        printf( "failed to open html file %s\n", name );
      }
      else
      {
        int width = os->getSize()[0];
        int height = os->getSize()[1];
        writeHTML( fp, data->filename, width, height );
        printf( "done\n" );
        fclose( fp );
      }
      delete[] name;
    }
    break;
  default:
    printf( "Error, the camera was in an undefined mode\n" );
    break;
  }
}

void
writeHTML( FILE* fp, const char* name, int width, int height )
{
  if ( name == NULL )
    name = "";
  fprintf( fp, "<html>\n" );
  fprintf( fp, "<head>\n" );
  fprintf( fp, "\t<title>Stereo Pair for %s</title>\n", name );
  fprintf( fp, "</head>\n" );

  fprintf( fp, "<body>\n" );

  fprintf( fp, "<center>\n" );
  fprintf( fp, "<h1>Stereo Images From the Scene</h1>\n" );
  fprintf( fp, "<h1>%s</h1>", name );
  fprintf( fp, "</center>\n" );

  fprintf( fp, "<center>\n" );
  float ratio = ( height == 0 ) ? 1 : width / ( float )height;
  int tabWidth, tabHeight;
  if ( ratio > 1 )
  {
    tabWidth = 200;
    tabHeight = ( int )( 200.0 / ratio );
  }
  else
  {
    tabHeight = 200;
    tabWidth = ( int )( 200.0 * ratio );
  }
  fprintf( fp, "<table COLS=2 WIDTH=\"200\" HEIGHT=\"200\" >\n" );
  fprintf( fp, "<tr>\n" );

  const char* viewname = RIGHT_FILENAME;
  char* filename = new char[strlen( name ) + strlen( viewname ) + 5];
  sprintf( filename, "%s%s.jpg", name, viewname );
  fprintf( fp, "\t<td><img SRC=\"%s\" height=%d width=%d></td>\n", filename, tabHeight, tabWidth );
  delete[] filename;

  viewname = LEFT_FILENAME;
  filename = new char[strlen( name ) + strlen( viewname ) + 5];
  sprintf( filename, "%s%s.jpg", name, viewname );
  fprintf( fp, "\t<td><img SRC=\"%s\" height=%d width=%d></td>\n", filename, tabHeight, tabWidth );
  delete[] filename;

  fprintf( fp, "</tr>\n" );
  fprintf( fp, "</table>\n" );
  fprintf( fp, "</center>\n" );

  fprintf( fp, "</body>\n" );
  fprintf( fp, "</html>\n" );
}
