#include "OffscreenStereo.h"

#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoOffscreenRenderArea.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

/////////////////////////////////////////////////////////////////////
//
// Implementing the OffscreenStereo viewer
//
/////////////////////////////////////////////////////////////////////

OffscreenStereo::OffscreenStereo( SoNode* myScene )
{
  setStereoViewType( NULL );
  this->scene = myScene;
  if ( myScene )
    myScene->ref();
  size[0] = size[1] = 200;
  topLeft.setValue( 0, 0 );
  bottomRight.setValue( 200, 200 );
  camera = NULL;
  camera = getViewerCamera();
  outputCB = NULL;
  outputCBud = NULL;
}

OffscreenStereo::OffscreenStereo( SoNode* myScene, int w, int h )
{
  setStereoViewType( NULL );
  scene = myScene;
  if ( myScene )
    myScene->ref();
  size.setValue( w, h );
  topLeft.setValue( 0, 0 );
  bottomRight.setValue( w, h );
  camera = NULL;
  camera = getViewerCamera();
  outputCB = NULL;
  outputCBud = NULL;
}

OffscreenStereo::~OffscreenStereo()
{
  if ( scene )
    scene->unref();
  stereoViewType->clearStereo();
}

void
OffscreenStereo::renderSceneInStereo()
{
  if ( camera != NULL )
  {
    camera->setStereoAdjustment( offset );
    camera->setBalanceAdjustment( balance );
  }
  stereoViewType->renderStereoView();
  ( *outputCB )( NULL, outputCBud );
}

void
OffscreenStereo::actualRendering()
{
  if ( !scene || !outputCB )
    return;

  SbViewportRegion vp( size );
  SoRef< SoOffscreenRenderArea > offscreenRenderArea = new SoOffscreenRenderArea( m_viewer->getNormalSoContext() );
  offscreenRenderArea->setTransparencyType( SoGLRenderAction::SORTED_OBJECT );
  offscreenRenderArea->getSceneManager()->setAntialiasing( 1.f, SoSceneManager::AUTO );
  offscreenRenderArea->setViewportRegion( vp );
  offscreenRenderArea->setSceneGraph( scene );
  ( *outputCB )( offscreenRenderArea.ptr(), outputCBud );
}

SoCamera*
OffscreenStereo::getViewerCamera()
{
  if ( camera )
    return camera;
  if ( !scene )
    return NULL;

  SoSearchAction sa;
  sa.setType( SoCamera::getClassTypeId() );
  sa.setSearchingAll( TRUE );
  sa.setInterest( SoSearchAction::FIRST );

  scene->ref();
  sa.apply( scene );
  SoPath* path = sa.getPath();
  scene->unrefNoDelete();

  SoCamera* cam = NULL;
  if ( path != NULL )
  {
    cam = ( SoCamera* )( ( SoFullPath* )sa.getPath() )->getTail();
  }
  else
  {
    cam = new SoPerspectiveCamera;
    if ( scene->isOfType( SoGroup::getClassTypeId() ) )
    {
      ( ( SoGroup* )scene )->insertChild( cam, 0 );
    }
    else
    {
      SoGroup* group = new SoGroup;
      group->ref();
      group->addChild( cam );
      group->addChild( scene );
      scene->unref(); // already ref'ed in the constructor
      scene = group;
    }
    cam->viewAll( scene, SbViewportRegion( size ) );
  }
  return cam;
}

void
OffscreenStereo::setViewport( short left, short bottom, short width, short height )
{
  topLeft.setValue( bottom + height, left );
  bottomRight.setValue( bottom, left + width );
  size.setValue( width, height );
}

void
OffscreenStereo::getViewport( short& left, short& bottom, short& width, short& height )
{
  left = topLeft[1];
  bottom = bottomRight[0];
  width = size[0];
  height = size[1];
}

/////////////////////////////////////////////////////////////////////
//
// Implementing the SimpleStereo technique
//
/////////////////////////////////////////////////////////////////////

void
SimpleStereo::renderStereoView()
{
  if ( m_stereoViewer == NULL )
    return;

  SoCamera* camera = m_stereoViewer->getViewerCamera();

  // setting the camera in one mode (left or right, according to the
  // reverse setting)
  camera->setStereoMode( reverse ? SoCamera::RIGHT_VIEW : SoCamera::LEFT_VIEW );
  m_stereoViewer->actualRendering();

  // setting the camera in one mode (right or left, according to the
  // reverse setting)
  camera->setStereoMode( reverse ? SoCamera::LEFT_VIEW : SoCamera::RIGHT_VIEW );
  m_stereoViewer->actualRendering();
  camera->setStereoMode( SoCamera::MONOSCOPIC );
}
